// ===== PWA - Progressive Web App =====

// متغيرات PWA
let deferredPrompt = null;
let isAppInstalled = false;

// التحقق من حالة التثبيت
function checkInstallState() {
    // التحقق إذا كان التطبيق مثبت بالفعل
    if (window.matchMedia('(display-mode: standalone)').matches) {
        isAppInstalled = true;
        hideInstallPrompt();
        return true;
    }
    
    // التحقق من iOS standalone
    if (window.navigator.standalone === true) {
        isAppInstalled = true;
        hideInstallPrompt();
        return true;
    }
    
    return false;
}

// تسجيل Service Worker
function registerServiceWorker() {
    if ('serviceWorker' in navigator) {
        window.addEventListener('load', () => {
            navigator.serviceWorker.register('./sw.js')
                .then((registration) => {
                    console.log('Service Worker تم تسجيل:', registration.scope);
                    
                    // التحقق من التحديثات
                    registration.addEventListener('updatefound', () => {
                        const newWorker = registration.installing;
                        newWorker.addEventListener('statechange', () => {
                            if (newWorker.state === 'installed' && navigator.serviceWorker.controller) {
                                // يوجد تحديث جديد
                                showUpdateNotification();
                            }
                        });
                    });
                })
                .catch((error) => {
                    console.log('Service Worker فشل التسجيل:', error);
                });
        });
    }
}

// عرض إشعار التحديث
function showUpdateNotification() {
    const updateToast = document.createElement('div');
    updateToast.className = 'update-toast';
    updateToast.innerHTML = `
        <p>يتوفر تحديث جديد</p>
        <button onclick="location.reload()">تحديث</button>
    `;
    document.body.appendChild(updateToast);
    
    setTimeout(() => {
        updateToast.classList.add('show');
    }, 100);
}

// معالجة حدث beforeinstallprompt
function handleBeforeInstallPrompt() {
    window.addEventListener('beforeinstallprompt', (e) => {
        // منع الظهور التلقائي
        e.preventDefault();
        
        // حفظ الحدث لاستخدامه لاحقاً
        deferredPrompt = e;
        
        // عرض زر التثبيت إذا لم يكن مثبت
        if (!checkInstallState()) {
            showInstallPrompt();
        }
    });
}

// عرض نافذة التثبيت
function showInstallPrompt() {
    const installPrompt = document.getElementById('installPrompt');
    if (installPrompt && !isAppInstalled) {
        // التحقق إذا تم إغلاقها سابقاً
        const dismissed = localStorage.getItem('installPromptDismissed');
        const dismissedTime = localStorage.getItem('installPromptDismissedTime');
        
        // إظهارها مجدداً بعد 24 ساعة
        if (dismissed && dismissedTime) {
            const timePassed = Date.now() - parseInt(dismissedTime);
            if (timePassed < 24 * 60 * 60 * 1000) {
                return;
            }
        }
        
        setTimeout(() => {
            installPrompt.classList.add('show');
        }, 3000);
    }
}

// إخفاء نافذة التثبيت
function hideInstallPrompt() {
    const installPrompt = document.getElementById('installPrompt');
    if (installPrompt) {
        installPrompt.classList.remove('show');
    }
}

// تثبيت التطبيق
async function installApp() {
    if (!deferredPrompt) {
        // إذا كان iOS، أظهر تعليمات التثبيت
        if (isIOS()) {
            showIOSModal();
            return;
        }
        return;
    }
    
    // إظهار نافذة التثبيت الأصلية
    deferredPrompt.prompt();
    
    // انتظار اختيار المستخدم
    const { outcome } = await deferredPrompt.userChoice;
    
    if (outcome === 'accepted') {
        console.log('تم قبول التثبيت');
        isAppInstalled = true;
    } else {
        console.log('تم رفض التثبيت');
    }
    
    // إعادة تعيين المتغير
    deferredPrompt = null;
    hideInstallPrompt();
}

// التحقق إذا كان iOS
function isIOS() {
    return /iPad|iPhone|iPod/.test(navigator.userAgent) && !window.MSStream;
}

// التحقق إذا كان Safari
function isSafari() {
    return /^((?!chrome|android).)*safari/i.test(navigator.userAgent);
}

// عرض نافذة تعليمات iOS
function showIOSModal() {
    const iosModal = document.getElementById('iosModal');
    if (iosModal) {
        iosModal.classList.add('show');
        document.body.style.overflow = 'hidden';
    }
}

// إغلاق نافذة iOS
function hideIOSModal() {
    const iosModal = document.getElementById('iosModal');
    if (iosModal) {
        iosModal.classList.remove('show');
        document.body.style.overflow = '';
    }
}

// معالجة حدث appinstalled
function handleAppInstalled() {
    window.addEventListener('appinstalled', () => {
        console.log('تم تثبيت التطبيق بنجاح');
        isAppInstalled = true;
        hideInstallPrompt();
        deferredPrompt = null;
        
        // عرض رسالة شكر
        showInstallSuccessToast();
    });
}

// عرض رسالة نجاح التثبيت
function showInstallSuccessToast() {
    const toast = document.createElement('div');
    toast.className = 'success-toast';
    toast.innerHTML = `
        <span>✓</span>
        <p>تم تثبيت التطبيق بنجاح!</p>
    `;
    document.body.appendChild(toast);
    
    setTimeout(() => {
        toast.classList.add('show');
    }, 100);
    
    setTimeout(() => {
        toast.classList.remove('show');
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

// تهيئة PWA
function initPWA() {
    // تسجيل Service Worker
    registerServiceWorker();
    
    // التحقق من حالة التثبيت
    checkInstallState();
    
    // معالجة أحداث التثبيت
    handleBeforeInstallPrompt();
    handleAppInstalled();
    
    // ربط أحداث الأزرار
    const installBtn = document.getElementById('installBtn');
    const installClose = document.getElementById('installClose');
    const iosModalClose = document.getElementById('iosModalClose');
    const iosModalBtn = document.getElementById('iosModalBtn');
    
    if (installBtn) {
        installBtn.addEventListener('click', () => {
            if (isIOS()) {
                showIOSModal();
                hideInstallPrompt();
            } else {
                installApp();
            }
        });
    }
    
    if (installClose) {
        installClose.addEventListener('click', () => {
            hideInstallPrompt();
            localStorage.setItem('installPromptDismissed', 'true');
            localStorage.setItem('installPromptDismissedTime', Date.now().toString());
        });
    }
    
    if (iosModalClose) {
        iosModalClose.addEventListener('click', hideIOSModal);
    }
    
    if (iosModalBtn) {
        iosModalBtn.addEventListener('click', hideIOSModal);
    }
    
    // إذا كان iOS وليس مثبت، أظهر التعليمات
    if (isIOS() && !checkInstallState()) {
        setTimeout(() => {
            showInstallPrompt();
        }, 5000);
    }
}

// تشغيل PWA عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', initPWA);
