// ===== التطبيق الرئيسي =====

// الحالة العامة للتطبيق
const appState = {
    selectedDay: 1,
    selectedGovernorate: 'baghdad',
    selectedDistrict: 'baghdad-center',
    selectedMadhab: 'shia', // sunni or shia
    menuOpen: false,
    duaaModalOpen: false
};

// ===== وظائف المساعدة =====

// الحصول على مواقيت الصلاة للمحافظة والمذهب المحدد
function getPrayerTimes() {
    const govTimes = prayerTimes[appState.selectedGovernorate];
    if (!govTimes) return prayerTimes['baghdad'][appState.selectedMadhab];
    return govTimes[appState.selectedMadhab];
}

// الحصول على دعاء اليوم
function getDailyDuaa() {
    return dailyDuaa.find(d => d.day === appState.selectedDay) || dailyDuaa[0];
}

// الحصول على اسم المحافظة
function getGovernorateName() {
    const gov = governorates.find(g => g.id === appState.selectedGovernorate);
    return gov ? gov.name : 'بغداد';
}

// الحصول على مناطق المحافظة المحددة
function getDistricts() {
    const gov = governorates.find(g => g.id === appState.selectedGovernorate);
    return gov ? gov.districts : [];
}

// الحصول على اسم المنطقة
function getDistrictName() {
    const districts = getDistricts();
    const district = districts.find(d => d.id === appState.selectedDistrict);
    return district ? district.name : '';
}

// ===== تحديث واجهة المستخدم =====

// تحديث عنوان اليوم
function updateDayTitle() {
    const dayTitle = document.getElementById('dayTitle');
    if (dayTitle) {
        dayTitle.textContent = `${appState.selectedDay} رمضان`;
    }
}

// تحديث أوقات الصلاة
function updatePrayerTimes() {
    const times = getPrayerTimes();
    
    document.getElementById('imsakTime').textContent = times.imsak + ' ص';
    document.getElementById('fajrTime').textContent = times.fajr + ' ص';
    document.getElementById('sunriseTime').textContent = times.sunrise + ' ص';
    document.getElementById('dhuhrTime').textContent = times.dhuhr + ' م';
    document.getElementById('maghribTime').textContent = times.maghrib + ' م';
    document.getElementById('midnightTime').textContent = times.midnight + ' م';
}

// تحديث أزرار الأيام
function updateDayButtons() {
    document.querySelectorAll('.day-btn').forEach(btn => {
        const day = parseInt(btn.dataset.day);
        if (day === appState.selectedDay) {
            btn.classList.add('active');
        } else {
            btn.classList.remove('active');
        }
    });
}

// تحديث محتوى الدعاء في الـ Modal
function updateDuaaModal() {
    const duaa = getDailyDuaa();
    const duaaDay = document.getElementById('duaaDay');
    const duaaText = document.getElementById('duaaText');
    
    if (duaaDay) duaaDay.textContent = `دعاء اليوم ${appState.selectedDay} من رمضان`;
    if (duaaText) duaaText.textContent = duaa.text;
}

// تحديث قائمة المحافظات
function populateGovernorates() {
    const select = document.getElementById('governorateSelect');
    if (!select) return;
    
    select.innerHTML = '';
    governorates.forEach(gov => {
        const option = document.createElement('option');
        option.value = gov.id;
        option.textContent = gov.name;
        if (gov.id === appState.selectedGovernorate) {
            option.selected = true;
        }
        select.appendChild(option);
    });
}

// تحديث قائمة المناطق
function populateDistricts() {
    const select = document.getElementById('districtSelect');
    if (!select) return;
    
    const districts = getDistricts();
    select.innerHTML = '';
    
    districts.forEach(dist => {
        const option = document.createElement('option');
        option.value = dist.id;
        option.textContent = dist.name;
        if (dist.id === appState.selectedDistrict) {
            option.selected = true;
        }
        select.appendChild(option);
    });
}

// تحديث المذهب المحدد
function updateMadhabSelection() {
    const sunniRadio = document.getElementById('sunniRadio');
    const shiaRadio = document.getElementById('shiaRadio');
    
    if (sunniRadio && shiaRadio) {
        sunniRadio.checked = appState.selectedMadhab === 'sunni';
        shiaRadio.checked = appState.selectedMadhab === 'shia';
    }
}

// تحديث معلومات الموقع في الهيدر
function updateLocationInfo() {
    const locationText = document.getElementById('locationText');
    if (locationText) {
        const govName = getGovernorateName();
        const madhab = appState.selectedMadhab === 'sunni' ? 'الوقف السني' : 'الوقف الشيعي';
        locationText.textContent = `${govName} • ${madhab}`;
    }
}

// ===== معالجات الأحداث =====

// اختيار يوم
function selectDay(day) {
    appState.selectedDay = day;
    updateDayTitle();
    updateDayButtons();
    updateDuaaModal();
    saveState();
}

// تغيير المحافظة
function changeGovernorate(governorateId) {
    appState.selectedGovernorate = governorateId;
    const districts = getDistricts();
    if (districts.length > 0) {
        appState.selectedDistrict = districts[0].id;
    }
    populateDistricts();
    updatePrayerTimes();
    updateLocationInfo();
    saveState();
}

// تغيير المنطقة
function changeDistrict(districtId) {
    appState.selectedDistrict = districtId;
    saveState();
}

// تغيير المذهب
function changeMadhab(madhab) {
    appState.selectedMadhab = madhab;
    updatePrayerTimes();
    updateLocationInfo();
    saveState();
}

// فتح القائمة الجانبية
function openMenu() {
    appState.menuOpen = true;
    const menu = document.getElementById('sideMenu');
    const overlay = document.getElementById('menuOverlay');
    if (menu) menu.classList.add('open');
    if (overlay) overlay.classList.add('open');
    document.body.style.overflow = 'hidden';
}

// إغلاق القائمة الجانبية
function closeMenu() {
    appState.menuOpen = false;
    const menu = document.getElementById('sideMenu');
    const overlay = document.getElementById('menuOverlay');
    if (menu) menu.classList.remove('open');
    if (overlay) overlay.classList.remove('open');
    document.body.style.overflow = '';
}

// فتح نافذة الدعاء
function openDuaaModal() {
    appState.duaaModalOpen = true;
    updateDuaaModal();
    const modal = document.getElementById('duaaModal');
    if (modal) modal.classList.add('open');
    document.body.style.overflow = 'hidden';
}

// إغلاق نافذة الدعاء
function closeDuaaModal() {
    appState.duaaModalOpen = false;
    const modal = document.getElementById('duaaModal');
    if (modal) modal.classList.remove('open');
    if (!appState.menuOpen) {
        document.body.style.overflow = '';
    }
}

// ===== حفظ واسترجاع الحالة =====

function saveState() {
    const stateToSave = {
        selectedGovernorate: appState.selectedGovernorate,
        selectedDistrict: appState.selectedDistrict,
        selectedMadhab: appState.selectedMadhab
    };
    localStorage.setItem('ramadanAppState', JSON.stringify(stateToSave));
}

function loadState() {
    const saved = localStorage.getItem('ramadanAppState');
    if (saved) {
        try {
            const parsed = JSON.parse(saved);
            appState.selectedGovernorate = parsed.selectedGovernorate || 'baghdad';
            appState.selectedDistrict = parsed.selectedDistrict || 'baghdad-center';
            appState.selectedMadhab = parsed.selectedMadhab || 'shia';
        } catch (e) {
            console.error('Error loading state:', e);
        }
    }
}

// ===== إنشاء أزرار الأيام =====

function createDayButtons() {
    const container = document.getElementById('daysGrid');
    if (!container) return;
    
    container.innerHTML = '';
    
    for (let i = 1; i <= 30; i++) {
        const btn = document.createElement('button');
        btn.className = 'day-btn' + (i === appState.selectedDay ? ' active' : '');
        btn.dataset.day = i;
        btn.innerHTML = `
            <span class="day-label">رمضان</span>
            <span class="day-number">${i}</span>
        `;
        btn.addEventListener('click', () => selectDay(i));
        container.appendChild(btn);
    }
}

// ===== تهيئة التطبيق =====

function initApp() {
    // تحميل الحالة المحفوظة
    loadState();
    
    // إنشاء أزرار الأيام
    createDayButtons();
    
    // ملء قوائم الاختيار
    populateGovernorates();
    populateDistricts();
    
    // تحديث واجهة المستخدم
    updateDayTitle();
    updatePrayerTimes();
    updateMadhabSelection();
    updateLocationInfo();
    
    // ربط الأحداث
    
    // زر القائمة
    const menuBtn = document.getElementById('menuBtn');
    if (menuBtn) {
        menuBtn.addEventListener('click', openMenu);
    }
    
    // زر إغلاق القائمة
    const closeMenuBtn = document.getElementById('closeMenuBtn');
    if (closeMenuBtn) {
        closeMenuBtn.addEventListener('click', closeMenu);
    }
    
    // خلفية القائمة للإغلاق
    const menuOverlay = document.getElementById('menuOverlay');
    if (menuOverlay) {
        menuOverlay.addEventListener('click', closeMenu);
    }
    
    // قائمة المحافظات
    const govSelect = document.getElementById('governorateSelect');
    if (govSelect) {
        govSelect.addEventListener('change', (e) => changeGovernorate(e.target.value));
    }
    
    // قائمة المناطق
    const distSelect = document.getElementById('districtSelect');
    if (distSelect) {
        distSelect.addEventListener('change', (e) => changeDistrict(e.target.value));
    }
    
    // اختيار المذهب
    const sunniRadio = document.getElementById('sunniRadio');
    const shiaRadio = document.getElementById('shiaRadio');
    
    if (sunniRadio) {
        sunniRadio.addEventListener('change', () => changeMadhab('sunni'));
    }
    if (shiaRadio) {
        shiaRadio.addEventListener('change', () => changeMadhab('shia'));
    }
    
    // زر عرض الدعاء
    const showDuaaBtn = document.getElementById('showDuaaBtn');
    if (showDuaaBtn) {
        showDuaaBtn.addEventListener('click', openDuaaModal);
    }
    
    // إغلاق نافذة الدعاء
    const closeDuaaBtn = document.getElementById('closeDuaaBtn');
    if (closeDuaaBtn) {
        closeDuaaBtn.addEventListener('click', closeDuaaModal);
    }
    
    const duaaModalOverlay = document.getElementById('duaaModal');
    if (duaaModalOverlay) {
        duaaModalOverlay.addEventListener('click', (e) => {
            if (e.target === duaaModalOverlay) {
                closeDuaaModal();
            }
        });
    }
    
    // الإغلاق بزر Escape
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
            if (appState.duaaModalOpen) {
                closeDuaaModal();
            } else if (appState.menuOpen) {
                closeMenu();
            }
        }
    });
}

// تشغيل التطبيق عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', initApp);
