<?php

use App\Http\Controllers\CartController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\CheckoutController;
use App\Http\Controllers\ClientDashboardController;
use App\Http\Controllers\ConfiguratorController;
use App\Http\Controllers\ContactController;
use App\Http\Controllers\DesignLibraryController;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\OffersController;
use App\Http\Controllers\OrderPrintController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\ShopController;
use App\Http\Controllers\StickerController;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;

Route::middleware(['web', 'store'])->group(function () {

    /* =====================================================
     | 🟢 عامة + المتجر (للجميع: زائر + عميل)
     ===================================================== */
    Route::get('/', [HomeController::class, 'index'])->name('home');

    Route::get('/about-us', [ContactController::class, 'about'])->name('about-us');
    Route::get('/contact-us', [ContactController::class, 'index'])->name('contact-us');

    Route::get('/shop', [ShopController::class, 'index'])->name('shop');
    Route::get('/shop/product/{slug}', [ProductController::class, 'show'])->name('shop.product');
    Route::get('/category/{category}', [CategoryController::class, 'show'])->name('category.show');
    Route::post('/product/{slug}', [ProductController::class, 'fetch'])->name('product.fetch');
    Route::get('/api/product/{id}/quick-view', [ProductController::class, 'quickView'])->name('product.quickview');

    // صفحة العروض
    Route::get('/offers', [OffersController::class, 'index'])->name('offers');

    // API endpoints for store
    Route::prefix('api/store')->name('api.store.')->group(function () {
        Route::get('/products', [ShopController::class, 'products'])->name('products');
        Route::get('/categories', [ShopController::class, 'categories'])->name('categories');
        Route::get('/brands', [ShopController::class, 'brands'])->name('brands');
        Route::get('/offers', [OffersController::class, 'products'])->name('offers');
    });

    /* =====================================================
     | 🎨 Product Configurator & Design Library
     ===================================================== */
    Route::prefix('configurator')->name('configurator.')->group(function () {
        Route::get('/model/{model}', [ConfiguratorController::class, 'show'])->name('show');
        Route::post('/model/{model}/save', [ConfiguratorController::class, 'saveDesign'])->name('save')->middleware('auth:client');
        Route::get('/design/{design}', [ConfiguratorController::class, 'loadDesign'])->name('load');
        Route::put('/design/{design}', [ConfiguratorController::class, 'updateDesign'])->name('update')->middleware('auth:client');
        Route::delete('/design/{design}', [ConfiguratorController::class, 'deleteDesign'])->name('delete')->middleware('auth:client');
        Route::post('/design/{design}/like', [ConfiguratorController::class, 'toggleLike'])->name('like')->middleware('auth:client');
    });

    // مكتبة الملصقات (API)
    Route::prefix('stickers')->name('stickers.')->group(function () {
        Route::get('/approved', [StickerController::class, 'approved'])->name('approved');
        Route::post('/{sticker}/track', [StickerController::class, 'trackUsage'])->name('track');
        Route::post('/upload', [StickerController::class, 'store'])->name('store')->middleware('auth:client');
        Route::delete('/{sticker}', [StickerController::class, 'destroy'])->name('destroy')->middleware('auth:client');
        Route::get('/my-stickers', [StickerController::class, 'myStickers'])->name('my')->middleware('auth:client');
    });

    Route::prefix('designs')->name('designs.')->group(function () {
        Route::get('/', \App\Livewire\DesignLibrary::class)->name('index');
        Route::get('/gallery', [DesignLibraryController::class, 'designs'])->name('gallery');
        Route::get('/{design}', [DesignLibraryController::class, 'show'])->name('show');
    });

    /* =====================================================
     | 🛒 السلة (للجميع: زائر + عميل)
     |  - التخزين يكون بالـ session للزوار
     |  - أو لو عندك DB لاحقاً تقدر تربطها
     ===================================================== */
    Route::prefix('cart')->name('cart.')->group(function () {
        Route::get('/', [CartController::class, 'index'])->name('index');                 // عرض السلة
        Route::get('/count', [CartController::class, 'count'])->name('count');            // عدد العناصر
        Route::get('/summary', [CartController::class, 'summary'])->name('summary');      // ملخص/المجموع

        Route::post('/add', [CartController::class, 'add'])->name('add');                 // إضافة
        Route::post('/update', [CartController::class, 'update'])->name('update');        // تعديل كمية
        Route::post('/remove', [CartController::class, 'remove'])->name('remove');        // حذف عنصر
        Route::post('/clear', [CartController::class, 'clear'])->name('clear');           // تفريغ
    });
    /* =====================================================
     | ✅ Checkout (للمسجلين فقط)
     |  - الزائر يقدر يعبّي سلة
     |  - لكن لا يقدر يكمل الطلب إلا بعد تسجيل الدخول
     ===================================================== */
    Route::middleware('auth:client')->prefix('checkout')->name('checkout.')->group(function () {
        Route::get('/', [CheckoutController::class, 'index'])->name('index');
        Route::post('/', [CheckoutController::class, 'store'])->name('store');
        Route::post('/apply-coupon', [CheckoutController::class, 'applyCoupon'])->name('apply-coupon');
        Route::post('/shipping-fee', [CheckoutController::class, 'getShippingFee'])->name('shipping-fee');
        Route::get('/success/{order?}', [CheckoutController::class, 'success'])->name('success');
    });

    /* =====================================================
     | 👤 مسارات العميل (Prefix: /client)
     ===================================================== */
    Route::prefix('client')->name('client.')->group(function () {

        /* -----------------------------
         | 🔓 ضيوف فقط
         ----------------------------- */
        Route::middleware(['guest:client', 'client.guest.redirect'])->group(function () {

            Route::get('/login', fn () => view('auth.login'))->name('login');

            Route::get('/register', fn () => view('auth.register'))->name('register');

            Route::get('/forgot-password', fn () => view('auth.forgot-password'))->name('forgot.password');

            Route::get('/reset-password/{token}', fn (string $token) => view('auth.reset-password', compact('token')))->name('password.reset');

            Route::get('/verify', fn () => view('auth.verify'))->name('verify');
        });

        /* -----------------------------
         | 🔐 مسجلين فقط
         ----------------------------- */
        Route::middleware('auth:client')->group(function () {

            Route::post('/logout', function () {
                Auth::guard('client')->logout();
                request()->session()->invalidate();
                request()->session()->regenerateToken();

                return redirect()->route('home');
            })->name('logout');

            // لوحة التحكم
            Route::get('/dashboard', [ClientDashboardController::class, 'index'])->name('dashboard');

            // الطلبات
            Route::get('/orders', [ClientDashboardController::class, 'orders'])->name('orders');
            Route::get('/orders/{id}', [ClientDashboardController::class, 'orderDetails'])->name('orders.show');

            // الملف الشخصي
            Route::get('/profile', [ClientDashboardController::class, 'profile'])->name('profile');
            Route::put('/profile', [ClientDashboardController::class, 'updateProfile'])->name('profile.update');
            Route::put('/profile/password', [ClientDashboardController::class, 'updatePassword'])->name('profile.password');

            // تصاميمي (للمصمم)
            Route::get('/my-designs', [ClientDashboardController::class, 'myDesigns'])->name('designs');

            // مكتبة ملصقاتي (للمصمم)
            Route::get('/my-stickers', [ClientDashboardController::class, 'myStickers'])->name('stickers');

            // العناوين
            Route::get('/addresses', [ClientDashboardController::class, 'addresses'])->name('addresses');
            Route::post('/addresses', [ClientDashboardController::class, 'storeAddress'])->name('addresses.store');
            Route::put('/addresses/{id}', [ClientDashboardController::class, 'updateAddress'])->name('addresses.update');
            Route::delete('/addresses/{id}', [ClientDashboardController::class, 'deleteAddress'])->name('addresses.delete');
            Route::post('/addresses/{id}/default', [ClientDashboardController::class, 'setDefaultAddress'])->name('addresses.default');
        });
    });

    /* =====================================================
     | 🖨️ طباعة الطلبات (للموظفين فقط)
     ===================================================== */
    Route::middleware('auth')->prefix('admin/orders')->name('orders.')->group(function () {
        Route::get('/{order}/print', [OrderPrintController::class, 'print'])->name('print');
        Route::get('/{order}/pdf', [OrderPrintController::class, 'pdf'])->name('pdf');
    });
});
