@extends('layouts.master')

@section('title', $product->name . ' - نساج')
@section('page', 'product')

@section('hero_title', $product->name)
@section('hero_breadcrumbs')
    <span class="text-white/30 mx-0.5">›</span>
    <a href="{{ route('shop') }}" class="text-nassaj-gold hover:text-nassaj-gold-light transition-colors">المتجر</a>
    <span class="text-white/30 mx-0.5">›</span>
    <span class="text-white/60">{{ $product->name }}</span>
@endsection

@php
    // Get product images
    $thumbnail = $product->thumbnail_url;
    $gallery = $product->gallery_urls ?? [];
    $allImages = array_filter(array_merge([$thumbnail], $gallery));

    // Get variants
    $variants = $product->variants;
    $defaultVariant = $variants->firstWhere('is_default', true) ?? $variants->first();

    // Calculate price
    $basePrice = 0;
    if ($defaultVariant) {
        $priceRow = \App\Models\ProductPrice::where('product_id', $product->id)
            ->where('product_variant_id', $defaultVariant->id)
            ->first();

        if ($priceRow) {
            $basePrice = (int) ($priceRow->retail_price ?? 0);
        }

        if ($basePrice <= 0 && $defaultVariant->cost) {
            $margin = $priceRow->retail_margin ?? 20;
            $basePrice = (int) round($defaultVariant->cost * (1 + $margin / 100));
        }
    }

    // Check stock
    $totalStock = $variants->sum(fn($v) => $v->stockBalances->sum('quantity') ?? 0);
    $inStock = $totalStock > 0;

    // Get unique colors and sizes from variants
    $colors = [];
    $sizes = [];
    $colorAttributeName = '';
    $sizeAttributeName = '';

    foreach ($variants as $v) {
        foreach ($v->attributeValues as $av) {
            $attrName = strtolower($av->attribute->name ?? '');
            if (in_array($attrName, ['color', 'لون', 'اللون'])) {
                if (!$colorAttributeName) {
                    $colorAttributeName = $av->attribute->display_name ?? ($av->attribute->name ?? 'اللون');
                }
                $colors[$av->id] = [
                    'id' => $av->id,
                    'name' => $av->display_value ?? $av->value,
                    'color_code' => $av->color_code ?? '#808080',
                ];
            } elseif (in_array($attrName, ['size', 'مقاس', 'الحجم', 'حجم'])) {
                if (!$sizeAttributeName) {
                    $sizeAttributeName = $av->attribute->display_name ?? ($av->attribute->name ?? 'المقاس');
                }
                $sizes[$av->id] = [
                    'id' => $av->id,
                    'name' => $av->display_value ?? $av->value,
                ];
            }
        }
    }
    $colors = array_values($colors);
    $sizes = array_values($sizes);
@endphp

@section('content')
    {{-- Breadcrumbs --}}
    <div class="bg-white dark:bg-[#0a0a0a] border-b border-gray-200 dark:border-white/5">
        <div class="site-container py-4">
            <nav class="flex items-center gap-2 text-sm">
                <a href="{{ route('home') }}"
                    class="text-gray-600 dark:text-gray-400 hover:text-nassaj-gold transition-colors">الرئيسية</a>
                <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7" />
                </svg>
                <a href="{{ route('shop') }}"
                    class="text-gray-600 dark:text-gray-400 hover:text-nassaj-gold transition-colors">المتجر</a>
                @if ($product->category)
                    <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7" />
                    </svg>
                    <a href="{{ route('category.show', $product->category->slug) }}"
                        class="text-gray-600 dark:text-gray-400 hover:text-nassaj-gold transition-colors">{{ $product->category->name }}</a>
                @endif
                <svg class="w-4 h-4 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7" />
                </svg>
                <span class="text-nassaj-gold font-bold">{{ $product->name }}</span>
            </nav>
        </div>
    </div>

    {{-- Product Details --}}
    <section class="py-12 bg-white dark:bg-[#0a0a0a]" x-data="{
        activeImage: 0,
        selectedColor: {{ count($colors) > 0 ? $colors[0]['id'] : 'null' }},
        selectedSize: {{ count($sizes) > 0 ? $sizes[0]['id'] : 'null' }},
        selectedColorName: '{{ count($colors) > 0 ? $colors[0]['name'] : '' }}',
        selectedSizeName: '{{ count($sizes) > 0 ? $sizes[0]['name'] : '' }}',
        qty: 1,
        price: {{ $basePrice }},
        activeTab: 'specs',
        formatPrice(num) {
            return num.toLocaleString('en-US');
        }
    }">
        <div class="site-container">
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-10 lg:gap-16">
                {{-- Product Gallery --}}
                <div class="space-y-4">
                    {{-- Main Image --}}
                    <div
                        class="relative aspect-square rounded-2xl overflow-hidden bg-gray-50 dark:bg-[#0f0f0f] border border-gray-200 dark:border-white/5">
                        @if (count($allImages) > 0)
                            @foreach ($allImages as $index => $image)
                                <img x-show="activeImage === {{ $index }}"
                                    x-transition:enter="transition ease-out duration-300"
                                    x-transition:enter-start="opacity-0 scale-95"
                                    x-transition:enter-end="opacity-100 scale-100" src="{{ $image }}"
                                    alt="{{ $product->name }}" class="w-full h-full object-cover">
                            @endforeach
                        @else
                            <div class="w-full h-full flex items-center justify-center text-gray-300 dark:text-gray-700">
                                <svg class="w-24 h-24" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1"
                                        d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                </svg>
                            </div>
                        @endif

                        {{-- Badges --}}
                        <div class="absolute top-3 right-3 flex flex-col gap-2 z-10">
                            @if ($product->is_new)
                                <span
                                    class="px-3 py-1.5 rounded-lg text-xs font-bold bg-emerald-500 text-white shadow-lg">جديد</span>
                            @endif
                            @if ($basePrice > 0)
                                <span class="px-3 py-1.5 rounded-lg text-xs font-bold bg-red-500 text-white shadow-lg">خصم
                                    15%</span>
                            @endif
                        </div>
                    </div>

                    {{-- Thumbnails --}}
                    @if (count($allImages) > 1)
                        <div class="flex gap-3 overflow-x-auto hide-scrollbar pb-2">
                            @foreach ($allImages as $index => $image)
                                <button @click="activeImage = {{ $index }}"
                                    class="shrink-0 w-20 h-20 rounded-xl overflow-hidden border-2 transition-all duration-200"
                                    :class="activeImage === {{ $index }} ? 'border-nassaj-gold scale-105 shadow-lg' :
                                        'border-gray-200 dark:border-white/10 hover:border-nassaj-gold/50'">
                                    <img src="{{ $image }}" alt="صورة {{ $index + 1 }}"
                                        class="w-full h-full object-cover">
                                </button>
                            @endforeach
                        </div>
                    @endif
                </div>

                {{-- Product Info --}}
                <div class="space-y-6">
                    {{-- Title & Rating --}}
                    <div>
                        <div class="flex items-center gap-3 mb-3">
                            <span
                                class="px-3 py-1 rounded-full text-xs font-bold {{ $inStock ? 'bg-emerald-500/20 text-emerald-400' : 'bg-red-500/20 text-red-400' }}">
                                {{ $inStock ? 'متوفر' : 'غير متوفر' }}
                            </span>
                            @if ($defaultVariant?->sku)
                                <span class="text-gray-500 text-sm">SKU: {{ $defaultVariant->sku }}</span>
                            @endif
                        </div>

                        <h1 class="cairo text-3xl md:text-4xl font-black text-white mb-3">
                            {{ $product->name }}
                        </h1>

                        {{-- Rating & Sales --}}
                        <div class="flex items-center gap-4 flex-wrap">
                            {{-- Rating Stars --}}
                            <div class="flex items-center gap-2">
                                <div class="flex items-center gap-0.5">
                                    @for ($i = 1; $i <= 5; $i++)
                                        <svg class="w-5 h-5 {{ $i <= 4 ? 'text-[#D4A356]' : 'text-gray-600' }}"
                                            fill="currentColor" viewBox="0 0 20 20">
                                            <path
                                                d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                        </svg>
                                    @endfor
                                </div>
                                <span class="text-white font-bold">4.8</span>
                                <span class="text-gray-500 text-sm">(124 تقييم)</span>
                            </div>
                            <span class="text-gray-600">|</span>
                            <span class="text-gray-400 text-sm">
                                <span class="text-[#D4A356] font-bold">350</span> تم بيعه
                            </span>
                        </div>

                        @if ($product->brand)
                            <p class="text-gray-400 text-sm mt-2">الماركة: <span
                                    class="text-[#D4A356]">{{ $product->brand->name }}</span></p>
                        @endif
                    </div>

                    {{-- Price Box --}}
                    <div class="p-6 rounded-2xl glass border border-white/5">
                        <div class="flex items-baseline gap-4 flex-wrap">
                            <span class="cairo text-4xl font-black text-[#D4A356]">{{ number_format($basePrice) }}
                                د.ع</span>
                            @if ($basePrice > 0)
                                <span
                                    class="text-gray-500 text-xl line-through">{{ number_format((int) ($basePrice * 1.15)) }}
                                    د.ع</span>
                                <span class="px-3 py-1 rounded-full text-sm font-bold bg-red-500/20 text-red-400">
                                    وفر 15%
                                </span>
                            @endif
                        </div>
                        <p class="text-emerald-400 text-sm mt-2 flex items-center gap-2">
                            <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                            </svg>
                            شحن مجاني للطلبات فوق 100,000 د.ع
                        </p>
                    </div>

                    {{-- Description --}}
                    @if ($product->description)
                        <div>
                            <p class="text-gray-400 leading-relaxed">{{ $product->description }}</p>
                        </div>
                    @endif

                    {{-- Color Selection --}}
                    @if (count($colors) > 0)
                        <div>
                            <h3 class="text-white font-bold text-sm mb-4 flex items-center gap-2">
                                {{ $colorAttributeName }}:
                                <span class="text-[#D4A356]" x-text="selectedColorName"></span>
                            </h3>
                            <div class="flex items-center gap-3">
                                @foreach ($colors as $color)
                                    <button
                                        @click="selectedColor = {{ $color['id'] }}; selectedColorName = '{{ $color['name'] }}'"
                                        class="w-10 h-10 rounded-full border-2 transition-all duration-200 relative"
                                        :class="selectedColor === {{ $color['id'] }} ?
                                            'border-[#D4A356] scale-110 ring-2 ring-[#D4A356]/30' :
                                            'border-transparent hover:scale-105'"
                                        style="background-color: {{ $color['color_code'] }}"
                                        title="{{ $color['name'] }}">
                                        <span x-show="selectedColor === {{ $color['id'] }}"
                                            class="absolute inset-0 flex items-center justify-center text-white">
                                            <svg class="w-5 h-5 drop-shadow-lg" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="3"
                                                    d="M5 13l4 4L19 7" />
                                            </svg>
                                        </span>
                                    </button>
                                @endforeach
                            </div>
                        </div>
                    @endif

                    {{-- Size Selection --}}
                    @if (count($sizes) > 0)
                        <div>
                            <h3 class="text-white font-bold text-sm mb-4 flex items-center gap-2">
                                {{ $sizeAttributeName }}:
                                <span class="text-[#D4A356]" x-text="selectedSizeName"></span>
                                <a href="#" class="text-[#D4A356] text-xs underline mr-auto">دليل المقاسات</a>
                            </h3>
                            <div class="flex flex-wrap gap-3">
                                @foreach ($sizes as $size)
                                    <button
                                        @click="selectedSize = {{ $size['id'] }}; selectedSizeName = '{{ $size['name'] }}'"
                                        class="min-w-[48px] h-12 px-4 rounded-xl border-2 font-bold text-sm transition-all duration-200"
                                        :class="selectedSize === {{ $size['id'] }} ?
                                            'border-[#D4A356] bg-[#D4A356] text-black' :
                                            'border-white/20 text-white hover:border-[#D4A356]/50'">
                                        {{ $size['name'] }}
                                    </button>
                                @endforeach
                            </div>
                        </div>
                    @endif

                    {{-- Quantity & Actions --}}
                    <div class="space-y-5">
                        {{-- Quantity & Quick Actions Row --}}
                        <div class="flex items-center justify-between">
                            {{-- Quantity Controls --}}
                            <div
                                class="flex items-center bg-gradient-to-r from-[#1a1a1a] to-[#151515] rounded-2xl border-2 border-white/10 shadow-lg">
                                <button @click="qty = Math.max(qty - 1, 1)"
                                    class="w-16 h-16 flex items-center justify-center text-white text-2xl font-bold rounded-l-2xl">
                                    <span>−</span>
                                </button>
                                <div class="px-8 min-w-[100px] text-center">
                                    <span class="cairo text-2xl font-black text-white" x-text="qty"></span>
                                </div>
                                <button @click="qty = Math.min(qty + 1, 99)"
                                    class="w-16 h-16 flex items-center justify-center text-white text-2xl font-bold rounded-r-2xl">
                                    <span>+</span>
                                </button>
                            </div>

                            {{-- Action Icons (Grouped) --}}
                            <div class="flex items-center gap-2">
                                {{-- Wishlist --}}
                                <button
                                    @click="window.dispatchEvent(new CustomEvent('wishlist-add-instant', {
                                        detail: {
                                            id: {{ $product->id }},
                                            name: '{{ addslashes($product->name) }}',
                                            price: {{ $basePrice }},
                                            image: '{{ $thumbnail ?? '' }}'
                                        }
                                    }))"
                                    class="w-14 h-14 rounded-xl bg-gradient-to-br from-[#1a1a1a] to-[#0d0d0d] border-2 border-white/10 hover:border-red-500/50 hover:bg-red-500/10 text-white hover:text-red-500 flex items-center justify-center transition-all shadow-lg group">
                                    <svg class="w-6 h-6 group-hover:scale-110 transition-transform" fill="none"
                                        stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                                    </svg>
                                </button>

                                {{-- Compare --}}
                                <button
                                    @click="window.dispatchEvent(new CustomEvent('compare-add-instant', {
                                        detail: {
                                            id: {{ $product->id }},
                                            name: '{{ addslashes($product->name) }}',
                                            price: {{ $basePrice }},
                                            image: '{{ $thumbnail ?? '' }}'
                                        }
                                    }))"
                                    class="w-14 h-14 rounded-xl bg-gradient-to-br from-[#1a1a1a] to-[#0d0d0d] border-2 border-white/10 hover:border-blue-500/50 hover:bg-blue-500/10 text-white hover:text-blue-500 flex items-center justify-center transition-all shadow-lg group">
                                    <svg class="w-6 h-6 group-hover:scale-110 transition-transform" fill="none"
                                        stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                                    </svg>
                                </button>

                                {{-- Share --}}
                                <button
                                    onclick="navigator.share ? navigator.share({title: '{{ $product->name }}', url: window.location.href}) : navigator.clipboard.writeText(window.location.href)"
                                    class="w-14 h-14 rounded-xl bg-gradient-to-br from-[#1a1a1a] to-[#0d0d0d] border-2 border-white/10 hover:border-green-500/50 hover:bg-green-500/10 text-white hover:text-green-500 flex items-center justify-center transition-all shadow-lg group">
                                    <svg class="w-6 h-6 group-hover:scale-110 transition-transform" fill="none"
                                        stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M8.684 13.342C8.886 12.938 9 12.482 9 12c0-.482-.114-.938-.316-1.342m0 2.684a3 3 0 110-2.684m0 2.684l6.632 3.316m-6.632-6l6.632-3.316m0 0a3 3 0 105.367-2.684 3 3 0 00-5.367 2.684zm0 9.316a3 3 0 105.368 2.684 3 3 0 00-5.368-2.684z" />
                                    </svg>
                                </button>
                            </div>
                        </div>
                        {{-- Buttons Row --}}
                        <div class="flex gap-3">
                            {{-- Add to Cart Button --}}
                            <button
                                @click="window.dispatchEvent(new CustomEvent('cart-add-instant', {
                                    detail: {
                                        id: {{ $product->id }},
                                        name: '{{ addslashes($product->name) }}',
                                        price: {{ $basePrice }},
                                        image: '{{ $thumbnail ?? '' }}',
                                        quantity: qty
                                    }
                                }))"
                                class="flex-1 bg-gradient-to-r from-[#D4A356] via-[#442247] to-[#D4A356] text-black py-4 rounded-2xl font-black text-base flex items-center justify-center gap-3 hover:shadow-2xl hover:shadow-[#D4A356]/40 transition-all hover:scale-[1.02] active:scale-[0.98] border-2 border-[#D4A356]/20 relative overflow-hidden group">
                                <div
                                    class="absolute inset-0 bg-gradient-to-r from-transparent via-white/10 to-transparent translate-x-[-100%] group-hover:translate-x-[100%] transition-transform duration-700">
                                </div>
                                <svg class="w-5 h-5 relative z-10" fill="none" stroke="currentColor"
                                    stroke-width="2.5" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round"
                                        d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z" />
                                </svg>
                                <span class="relative z-10">أضف للسلة</span>
                                <span class="relative z-10 text-black/40 mx-1">|</span>
                                <span class="cairo relative z-10 font-black"
                                    x-text="formatPrice(qty * price) + ' د.ع'"></span>
                            </button>

                            {{-- Buy Now --}}
                            <a href="{{ route('checkout.index') }}"
                                class="flex-1 bg-white/5 backdrop-blur-xl text-white py-4 rounded-2xl font-bold text-base flex items-center justify-center hover:bg-white/10 transition-all duration-300 border-2 border-[#D4A356]/30 hover:border-[#D4A356]/60 hover:shadow-xl hover:shadow-[#D4A356]/20 group relative overflow-hidden">
                                <div
                                    class="absolute inset-0 bg-gradient-to-r from-[#D4A356]/0 via-[#D4A356]/10 to-[#D4A356]/0 translate-y-[100%] group-hover:translate-y-0 transition-transform duration-300">
                                </div>
                                <span class="relative z-10">اشتري الآن</span>
                            </a>
                        </div>
                    </div>

                    {{-- Features Grid --}}
                    <div class="grid grid-cols-2 gap-3">
                        <div
                            class="flex items-center gap-3 p-5 rounded-2xl bg-gradient-to-br from-[#1a1a1a] to-[#0d0d0d] border border-white/10 hover:border-[#D4A356]/30 transition-all group">
                            <div
                                class="w-12 h-12 rounded-xl bg-gradient-to-br from-[#D4A356]/20 to-[#D4A356]/10 flex items-center justify-center group-hover:scale-110 transition-transform">
                                <svg class="w-6 h-6 text-[#D4A356]" fill="none" stroke="currentColor"
                                    stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round"
                                        d="M5 8h14M5 8a2 2 0 110-4h14a2 2 0 110 4M5 8v10a2 2 0 002 2h10a2 2 0 002-2V8m-9 4h4" />
                                </svg>
                            </div>
                            <div>
                                <span class="text-white font-bold text-sm block">شحن سريع</span>
                                <span class="text-gray-500 text-xs">2-3 أيام عمل</span>
                            </div>
                        </div>
                        <div
                            class="flex items-center gap-3 p-5 rounded-2xl bg-gradient-to-br from-[#1a1a1a] to-[#0d0d0d] border border-white/10 hover:border-emerald-500/30 transition-all group">
                            <div
                                class="w-12 h-12 rounded-xl bg-gradient-to-br from-emerald-500/20 to-emerald-500/10 flex items-center justify-center group-hover:scale-110 transition-transform">
                                <svg class="w-6 h-6 text-emerald-400" fill="none" stroke="currentColor"
                                    stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round"
                                        d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
                                </svg>
                            </div>
                            <div>
                                <span class="text-white font-bold text-sm block">ضمان سنة</span>
                                <span class="text-gray-500 text-xs">ضمان المصنع</span>
                            </div>
                        </div>
                        <div
                            class="flex items-center gap-3 p-5 rounded-2xl bg-gradient-to-br from-[#1a1a1a] to-[#0d0d0d] border border-white/10 hover:border-blue-500/30 transition-all group">
                            <div
                                class="w-12 h-12 rounded-xl bg-gradient-to-br from-blue-500/20 to-blue-500/10 flex items-center justify-center group-hover:scale-110 transition-transform">
                                <svg class="w-6 h-6 text-blue-400" fill="none" stroke="currentColor" stroke-width="2"
                                    viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round"
                                        d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                                </svg>
                            </div>
                            <div>
                                <span class="text-white font-bold text-sm block">إرجاع مجاني</span>
                                <span class="text-gray-500 text-xs">خلال 14 يوم</span>
                            </div>
                        </div>
                        <div
                            class="flex items-center gap-3 p-5 rounded-2xl bg-gradient-to-br from-[#1a1a1a] to-[#0d0d0d] border border-white/10 hover:border-purple-500/30 transition-all group">
                            <div
                                class="w-12 h-12 rounded-xl bg-gradient-to-br from-purple-500/20 to-purple-500/10 flex items-center justify-center group-hover:scale-110 transition-transform">
                                <svg class="w-6 h-6 text-purple-400" fill="none" stroke="currentColor"
                                    stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round"
                                        d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z" />
                                </svg>
                            </div>
                            <div>
                                <span class="text-white font-bold text-sm block">دفع آمن</span>
                                <span class="text-gray-500 text-xs">بطاقات ائتمان</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            {{-- Tabs Section --}}
            <div class="mt-16">
                {{-- Tab Headers --}}
                <div class="flex items-center gap-1 border-b border-white/10 overflow-x-auto hide-scrollbar">
                    <button @click="activeTab = 'specs'"
                        class="px-6 py-4 font-bold text-sm whitespace-nowrap transition-all border-b-2 -mb-px"
                        :class="activeTab === 'specs' ? 'text-[#D4A356] border-[#D4A356]' :
                            'text-gray-400 border-transparent hover:text-white'">
                        المواصفات
                    </button>
                    <button @click="activeTab = 'details'"
                        class="px-6 py-4 font-bold text-sm whitespace-nowrap transition-all border-b-2 -mb-px"
                        :class="activeTab === 'details' ? 'text-[#D4A356] border-[#D4A356]' :
                            'text-gray-400 border-transparent hover:text-white'">
                        التفاصيل
                    </button>
                    <button @click="activeTab = 'reviews'"
                        class="px-6 py-4 font-bold text-sm whitespace-nowrap transition-all border-b-2 -mb-px"
                        :class="activeTab === 'reviews' ? 'text-[#D4A356] border-[#D4A356]' :
                            'text-gray-400 border-transparent hover:text-white'">
                        التقييمات (124)
                    </button>
                    <button @click="activeTab = 'shipping'"
                        class="px-6 py-4 font-bold text-sm whitespace-nowrap transition-all border-b-2 -mb-px"
                        :class="activeTab === 'shipping' ? 'text-[#D4A356] border-[#D4A356]' :
                            'text-gray-400 border-transparent hover:text-white'">
                        الشحن والإرجاع
                    </button>
                </div>

                {{-- Tab Content --}}
                <div class="py-8">
                    {{-- Specifications Tab --}}
                    <div x-show="activeTab === 'specs'" x-transition:enter="transition ease-out duration-200"
                        x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div class="flex items-center justify-between p-4 rounded-xl glass">
                                <span class="text-gray-400">الماركة</span>
                                <span class="text-white font-bold">{{ $product->brand?->name ?? 'نساج' }}</span>
                            </div>
                            <div class="flex items-center justify-between p-4 rounded-xl glass">
                                <span class="text-gray-400">الفئة</span>
                                <span class="text-white font-bold">{{ $product->category?->name ?? 'عام' }}</span>
                            </div>
                            <div class="flex items-center justify-between p-4 rounded-xl glass">
                                <span class="text-gray-400">الخامة</span>
                                <span class="text-white font-bold">قطن 100%</span>
                            </div>
                            <div class="flex items-center justify-between p-4 rounded-xl glass">
                                <span class="text-gray-400">بلد المنشأ</span>
                                <span class="text-white font-bold">تركيا</span>
                            </div>
                            <div class="flex items-center justify-between p-4 rounded-xl glass">
                                <span class="text-gray-400">نوع القماش</span>
                                <span class="text-white font-bold">قطن مصري</span>
                            </div>
                            <div class="flex items-center justify-between p-4 rounded-xl glass">
                                <span class="text-gray-400">موسم</span>
                                <span class="text-white font-bold">كل المواسم</span>
                            </div>
                        </div>
                    </div>

                    {{-- Details Tab --}}
                    <div x-show="activeTab === 'details'" x-cloak x-transition:enter="transition ease-out duration-200"
                        x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100">
                        <div class="prose prose-invert max-w-none">
                            <p class="text-gray-400 leading-relaxed mb-4">
                                {{ $product->description ?? 'منتج عالي الجودة من نساج، مصنوع من أجود الخامات لضمان الراحة والأناقة. تصميم عصري يناسب جميع المناسبات.' }}
                            </p>
                            <h4 class="text-white font-bold mt-6 mb-3">مميزات المنتج:</h4>
                            <ul class="space-y-2">
                                <li class="text-gray-400 flex items-center gap-2">
                                    <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M5 13l4 4L19 7" />
                                    </svg>
                                    خامة عالية الجودة
                                </li>
                                <li class="text-gray-400 flex items-center gap-2">
                                    <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M5 13l4 4L19 7" />
                                    </svg>
                                    تصميم عصري وأنيق
                                </li>
                                <li class="text-gray-400 flex items-center gap-2">
                                    <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M5 13l4 4L19 7" />
                                    </svg>
                                    مناسب لجميع المناسبات
                                </li>
                                <li class="text-gray-400 flex items-center gap-2">
                                    <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M5 13l4 4L19 7" />
                                    </svg>
                                    سهل الغسيل والعناية
                                </li>
                            </ul>
                        </div>
                    </div>

                    {{-- Reviews Tab --}}
                    <div x-show="activeTab === 'reviews'" x-cloak x-transition:enter="transition ease-out duration-200"
                        x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100">
                        {{-- Rating Summary --}}
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-8 mb-8">
                            <div class="text-center p-6 rounded-2xl glass">
                                <div class="text-5xl font-black text-[#D4A356] mb-2">4.8</div>
                                <div class="flex items-center justify-center gap-1 mb-2">
                                    @for ($i = 1; $i <= 5; $i++)
                                        <svg class="w-5 h-5 {{ $i <= 5 ? 'text-[#D4A356]' : 'text-gray-600' }}"
                                            fill="currentColor" viewBox="0 0 20 20">
                                            <path
                                                d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                        </svg>
                                    @endfor
                                </div>
                                <p class="text-gray-400 text-sm">من 124 تقييم</p>
                            </div>

                            <div class="col-span-2 space-y-3">
                                @foreach ([['stars' => 5, 'count' => 89, 'percent' => 72], ['stars' => 4, 'count' => 25, 'percent' => 20], ['stars' => 3, 'count' => 7, 'percent' => 6], ['stars' => 2, 'count' => 2, 'percent' => 1], ['stars' => 1, 'count' => 1, 'percent' => 1]] as $rating)
                                    <div class="flex items-center gap-3">
                                        <span class="text-white text-sm w-20">{{ $rating['stars'] }} نجوم</span>
                                        <div class="flex-1 h-2 bg-white/10 rounded-full overflow-hidden">
                                            <div class="h-full bg-[#D4A356] rounded-full"
                                                style="width: {{ $rating['percent'] }}%"></div>
                                        </div>
                                        <span class="text-gray-400 text-sm w-12">{{ $rating['percent'] }}%</span>
                                    </div>
                                @endforeach
                            </div>
                        </div>

                        {{-- Sample Reviews --}}
                        <div class="space-y-6">
                            @foreach ([['name' => 'أحمد محمد', 'date' => 'منذ 3 أيام', 'rating' => 5, 'comment' => 'منتج ممتاز جداً، الخامة عالية الجودة والتوصيل سريع. أنصح به بشدة!'], ['name' => 'سارة علي', 'date' => 'منذ أسبوع', 'rating' => 4, 'comment' => 'جودة جيدة والمقاس مناسب. التغليف كان ممتاز.'], ['name' => 'محمد خالد', 'date' => 'منذ أسبوعين', 'rating' => 5, 'comment' => 'أفضل شراء قمت به هذا الشهر. سأشتري المزيد بالتأكيد!']] as $review)
                                <div class="p-6 rounded-2xl glass">
                                    <div class="flex items-start justify-between mb-4">
                                        <div class="flex items-center gap-3">
                                            <div
                                                class="w-12 h-12 rounded-full bg-[#D4A356]/20 flex items-center justify-center text-[#D4A356] font-bold">
                                                {{ mb_substr($review['name'], 0, 1) }}
                                            </div>
                                            <div>
                                                <h4 class="text-white font-bold">{{ $review['name'] }}</h4>
                                                <p class="text-gray-500 text-sm">{{ $review['date'] }}</p>
                                            </div>
                                        </div>
                                        <div class="flex items-center gap-0.5">
                                            @for ($i = 1; $i <= 5; $i++)
                                                <svg class="w-4 h-4 {{ $i <= $review['rating'] ? 'text-[#D4A356]' : 'text-gray-600' }}"
                                                    fill="currentColor" viewBox="0 0 20 20">
                                                    <path
                                                        d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                                </svg>
                                            @endfor
                                        </div>
                                    </div>
                                    <p class="text-gray-400">{{ $review['comment'] }}</p>
                                </div>
                            @endforeach
                        </div>

                        {{-- Load More --}}
                        <div class="text-center mt-8">
                            <button
                                class="px-8 py-3 rounded-full glass text-white font-bold hover:bg-white/10 transition-colors">
                                عرض المزيد من التقييمات
                            </button>
                        </div>
                    </div>

                    {{-- Shipping Tab --}}
                    <div x-show="activeTab === 'shipping'" x-cloak x-transition:enter="transition ease-out duration-200"
                        x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                            {{-- Shipping Info --}}
                            <div class="space-y-6">
                                <h3 class="text-white font-bold text-xl mb-4">معلومات الشحن</h3>
                                <div class="space-y-4">
                                    <div class="flex items-start gap-4 p-4 rounded-xl glass">
                                        <div
                                            class="w-10 h-10 rounded-full bg-[#D4A356]/15 flex items-center justify-center shrink-0">
                                            <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                                            </svg>
                                        </div>
                                        <div>
                                            <h4 class="text-white font-bold mb-1">التوصيل داخل بغداد</h4>
                                            <p class="text-gray-400 text-sm">1-2 أيام عمل - 5,000 د.ع</p>
                                        </div>
                                    </div>
                                    <div class="flex items-start gap-4 p-4 rounded-xl glass">
                                        <div
                                            class="w-10 h-10 rounded-full bg-[#D4A356]/15 flex items-center justify-center shrink-0">
                                            <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M3.055 11H5a2 2 0 012 2v1a2 2 0 002 2 2 2 0 012 2v2.945M8 3.935V5.5A2.5 2.5 0 0010.5 8h.5a2 2 0 012 2 2 2 0 104 0 2 2 0 012-2h1.064M15 20.488V18a2 2 0 012-2h3.064M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                            </svg>
                                        </div>
                                        <div>
                                            <h4 class="text-white font-bold mb-1">التوصيل للمحافظات</h4>
                                            <p class="text-gray-400 text-sm">2-4 أيام عمل - 7,000 د.ع</p>
                                        </div>
                                    </div>
                                    <div class="flex items-start gap-4 p-4 rounded-xl glass">
                                        <div
                                            class="w-10 h-10 rounded-full bg-emerald-500/15 flex items-center justify-center shrink-0">
                                            <svg class="w-5 h-5 text-emerald-400" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M5 13l4 4L19 7" />
                                            </svg>
                                        </div>
                                        <div>
                                            <h4 class="text-white font-bold mb-1">شحن مجاني</h4>
                                            <p class="text-gray-400 text-sm">للطلبات فوق 100,000 د.ع</p>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {{-- Return Policy --}}
                            <div class="space-y-6">
                                <h3 class="text-white font-bold text-xl mb-4">سياسة الإرجاع</h3>
                                <div class="space-y-4">
                                    <div class="flex items-start gap-4 p-4 rounded-xl glass">
                                        <div
                                            class="w-10 h-10 rounded-full bg-[#D4A356]/15 flex items-center justify-center shrink-0">
                                            <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                                            </svg>
                                        </div>
                                        <div>
                                            <h4 class="text-white font-bold mb-1">إرجاع مجاني</h4>
                                            <p class="text-gray-400 text-sm">خلال 14 يوم من الاستلام</p>
                                        </div>
                                    </div>
                                    <div class="flex items-start gap-4 p-4 rounded-xl glass">
                                        <div
                                            class="w-10 h-10 rounded-full bg-[#D4A356]/15 flex items-center justify-center shrink-0">
                                            <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                                            </svg>
                                        </div>
                                        <div>
                                            <h4 class="text-white font-bold mb-1">شروط الإرجاع</h4>
                                            <p class="text-gray-400 text-sm">المنتج بحالته الأصلية مع جميع الملحقات</p>
                                        </div>
                                    </div>
                                    <div class="flex items-start gap-4 p-4 rounded-xl glass">
                                        <div
                                            class="w-10 h-10 rounded-full bg-[#D4A356]/15 flex items-center justify-center shrink-0">
                                            <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                                            </svg>
                                        </div>
                                        <div>
                                            <h4 class="text-white font-bold mb-1">استرداد المبلغ</h4>
                                            <p class="text-gray-400 text-sm">خلال 3-5 أيام عمل بعد الموافقة</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    {{-- Related Products --}}
    @if ($relatedProducts->isNotEmpty())
        <section class="py-16 bg-[#0a0a0a] border-t border-white/5">
            <div class="site-container">
                <div class="flex items-center justify-between mb-10">
                    <div>
                        <span
                            class="inline-block px-4 py-1.5 rounded-full glass text-xs font-bold text-[#D4A356] tracking-wider mb-3">
                            قد يعجبك أيضاً
                        </span>
                        <h2 class="cairo text-2xl md:text-3xl font-black text-white">
                            منتجات <span class="text-[#D4A356]">مشابهة</span>
                        </h2>
                    </div>
                    <a href="{{ route('shop', ['category' => $product->category_id]) }}"
                        class="hidden sm:flex items-center gap-2 text-[#D4A356] font-bold text-sm hover:underline">
                        عرض الكل
                        <svg class="w-4 h-4 rotate-180" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M14 5l7 7m0 0l-7 7m7-7H3" />
                        </svg>
                    </a>
                </div>

                <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 md:gap-6">
                    @foreach ($relatedProducts as $related)
                        <livewire:products.product-card :product-id="$related->id" wire:key="related-{{ $related->id }}" />
                    @endforeach
                </div>
            </div>
        </section>
    @endif
@endsection
