@extends('layouts.master')

@section('title', 'إتمام الطلب - نساج')

@section('hero_title', 'إتمام الطلب')
@section('hero_description', 'أكمل بيانات الشحن والدفع لإتمام طلبك')
@section('hero_breadcrumbs')
    <span class="text-white/30 mx-0.5">›</span>
    <a href="{{ route('cart.index') }}" class="text-nassaj-gold hover:text-nassaj-gold-light transition-colors">سلة التسوق</a>
    <span class="text-white/30 mx-0.5">›</span>
    <span class="text-white/60">إتمام الطلب</span>
@endsection

@push('styles')
    <style>
        .scrollbar-thin::-webkit-scrollbar {
            height: 4px;
        }

        .scrollbar-thin::-webkit-scrollbar-track {
            background: transparent;
        }

        .scrollbar-thin::-webkit-scrollbar-thumb {
            background: rgba(212, 163, 86, 0.5);
            border-radius: 2px;
        }
    </style>
@endpush

@section('content')
    {{-- Checkout Content --}}
    <section class="py-8 bg-[#0a0a0a]" x-data="checkoutPage()">
        <div class="site-container">

            {{-- Error Messages --}}
            @if (session('error'))
                <div class="mb-6 p-4 rounded-xl bg-red-500/10 border border-red-500/20 text-red-400">
                    {{ session('error') }}
                </div>
            @endif

            <form action="{{ route('checkout.store') }}" method="POST" id="checkout-form">
                @csrf
                <input type="hidden" name="shipping_address_id" :value="useExistingAddress ? selectedAddressId : ''">
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    {{-- Main Content --}}
                    <div class="lg:col-span-2 space-y-6">

                        {{-- Saved Addresses --}}
                        @if ($savedAddresses->isNotEmpty())
                            <div class="bg-[#111111] rounded-2xl border border-white/10 p-6">
                                <div class="flex items-center justify-between mb-4">
                                    <div class="flex items-center gap-3">
                                        <div
                                            class="w-10 h-10 rounded-full bg-[#D4A356]/10 flex items-center justify-center">
                                            <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M5 5a2 2 0 012-2h10a2 2 0 012 2v16l-7-3.5L5 21V5z" />
                                            </svg>
                                        </div>
                                        <h2 class="text-white font-black text-lg">العناوين المحفوظة</h2>
                                    </div>
                                    <a href="{{ route('client.addresses') }}"
                                        class="text-[#D4A356] text-sm font-bold hover:text-[#442247] transition-colors flex items-center gap-1">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
                                        </svg>
                                        إدارة العناوين
                                    </a>
                                </div>
                                <div class="grid grid-cols-1 sm:grid-cols-2 gap-3">
                                    @foreach ($savedAddresses as $addr)
                                        <label class="relative p-4 rounded-xl border-2 cursor-pointer transition-all"
                                            :class="selectedAddressId == {{ $addr->id }} ?
                                                'border-[#D4A356] bg-[#D4A356]/5' :
                                                'border-white/10 bg-white/5 hover:border-white/20'"
                                            @click="selectAddress({{ json_encode($addr->toArray()) }})">
                                            <input type="radio" name="saved_address" value="{{ $addr->id }}"
                                                x-model="selectedAddressId" class="hidden">
                                            <div class="flex items-start gap-3">
                                                <div class="w-5 h-5 rounded-full border-2 flex items-center justify-center shrink-0 mt-0.5"
                                                    :class="selectedAddressId == {{ $addr->id }} ? 'border-[#D4A356]' :
                                                        'border-gray-500'">
                                                    <div class="w-3 h-3 rounded-full bg-[#D4A356]"
                                                        x-show="selectedAddressId == {{ $addr->id }}"></div>
                                                </div>
                                                <div class="flex-1 min-w-0">
                                                    <div class="flex items-center gap-2">
                                                        <span
                                                            class="text-white font-bold text-sm">{{ $addr->name }}</span>
                                                        @if ($addr->is_default)
                                                            <span
                                                                class="px-2 py-0.5 bg-[#D4A356]/20 text-[#D4A356] text-[10px] font-bold rounded">افتراضي</span>
                                                        @endif
                                                    </div>
                                                    <p class="text-gray-400 text-xs mt-1 direction-ltr text-right">
                                                        {{ $addr->phone }}</p>
                                                    <p class="text-gray-500 text-xs mt-1 line-clamp-2">
                                                        {{ $addr->city?->name }} - {{ $addr->address }}</p>
                                                </div>
                                            </div>
                                        </label>
                                    @endforeach
                                </div>
                            </div>
                        @endif

                        {{-- Shipping Form --}}
                        <div class="bg-[#111111] rounded-2xl border border-white/10 p-6">
                            <div class="flex items-center gap-3 mb-6">
                                <div class="w-10 h-10 rounded-full bg-[#D4A356]/10 flex items-center justify-center">
                                    <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                                    </svg>
                                </div>
                                <h2 class="text-white font-black text-xl">معلومات التوصيل</h2>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                {{-- Full Name --}}
                                <div>
                                    <label class="block text-sm font-bold text-gray-400 mb-2">الاسم الكامل *</label>
                                    <input type="text" name="name" x-model="form.name" required
                                        @input="onAddressFieldChange()"
                                        class="w-full px-4 py-3 rounded-xl bg-white/5 border border-white/10 text-white placeholder-gray-500 outline-none focus:border-[#D4A356] transition-colors"
                                        placeholder="الاسم الكامل">
                                    @error('name')
                                        <p class="text-red-400 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                </div>

                                {{-- Phone --}}
                                <div>
                                    <label class="block text-sm font-bold text-gray-400 mb-2">رقم الهاتف *</label>
                                    <input type="tel" name="phone" x-model="form.phone" required maxlength="11"
                                        pattern="[0-9]{11}"
                                        oninput="this.value = this.value.replace(/[^0-9]/g, '').slice(0, 11)"
                                        @input="onAddressFieldChange()"
                                        class="w-full px-4 py-3 rounded-xl bg-white/5 border border-white/10 text-white placeholder-gray-500 outline-none focus:border-[#D4A356] transition-colors text-right direction-ltr"
                                        placeholder="07xxxxxxxxx">
                                    @error('phone')
                                        <p class="text-red-400 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                </div>

                                {{-- City/Governorate --}}
                                <div>
                                    <label class="block text-sm font-bold text-gray-400 mb-2">المحافظة *</label>
                                    <select name="city_id" x-model="form.city_id" required
                                        @change="updateShipping(); onAddressFieldChange()"
                                        class="w-full px-4 py-3 rounded-xl bg-[#1a1a1a] border border-white/10 text-white outline-none focus:border-[#D4A356] transition-colors">
                                        <option value="">اختر المحافظة</option>
                                        @foreach ($cities as $city)
                                            <option value="{{ $city->id }}" data-fee="{{ $city->shipping_fee }}">
                                                {{ $city->name }}</option>
                                        @endforeach
                                    </select>
                                    @error('city_id')
                                        <p class="text-red-400 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                </div>

                                {{-- Area/District --}}
                                <div>
                                    <label class="block text-sm font-bold text-gray-400 mb-2">المنطقة / الحي *</label>
                                    <input type="text" name="area" x-model="form.area" required
                                        @input="onAddressFieldChange()"
                                        class="w-full px-4 py-3 rounded-xl bg-white/5 border border-white/10 text-white placeholder-gray-500 outline-none focus:border-[#D4A356] transition-colors"
                                        placeholder="المنطقة أو الحي">
                                    @error('area')
                                        <p class="text-red-400 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                </div>

                                {{-- Detailed Address --}}
                                <div class="md:col-span-2">
                                    <label class="block text-sm font-bold text-gray-400 mb-2">أقرب نقطة دالة *</label>
                                    <input type="text" name="address" x-model="form.address" required
                                        @input="onAddressFieldChange()"
                                        class="w-full px-4 py-3 rounded-xl bg-white/5 border border-white/10 text-white placeholder-gray-500 outline-none focus:border-[#D4A356] transition-colors"
                                        placeholder="الشارع، المعلم المجاور، رقم البيت">
                                    @error('address')
                                        <p class="text-red-400 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                </div>

                                {{-- Notes --}}
                                <div class="md:col-span-2">
                                    <label class="block text-sm font-bold text-gray-400 mb-2">ملاحظات (اختياري)</label>
                                    <textarea name="notes" x-model="form.notes" rows="2"
                                        class="w-full px-4 py-3 rounded-xl bg-white/5 border border-white/10 text-white placeholder-gray-500 outline-none focus:border-[#D4A356] transition-colors resize-none"
                                        placeholder="أي ملاحظات خاصة بالطلب أو التوصيل"></textarea>
                                </div>
                            </div>
                        </div>

                        {{-- Payment Method --}}
                        <div class="bg-[#111111] rounded-2xl border border-white/10 p-6">
                            <div class="flex items-center gap-3 mb-6">
                                <div class="w-10 h-10 rounded-full bg-[#D4A356]/10 flex items-center justify-center">
                                    <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z" />
                                    </svg>
                                </div>
                                <h2 class="text-white font-black text-xl">طريقة الدفع</h2>
                            </div>

                            <div class="space-y-3">
                                {{-- Cash on Delivery --}}
                                <label
                                    class="flex items-center justify-between p-4 rounded-xl border-2 cursor-pointer transition-all"
                                    :class="form.payment_method === 'cod' ? 'border-[#D4A356] bg-[#D4A356]/5' :
                                        'border-white/10 bg-white/5 hover:border-white/20'">
                                    <div class="flex items-center gap-4">
                                        <input type="radio" name="payment_method" value="cod"
                                            x-model="form.payment_method" class="hidden">
                                        <div class="w-5 h-5 rounded-full border-2 flex items-center justify-center"
                                            :class="form.payment_method === 'cod' ? 'border-[#D4A356]' : 'border-gray-500'">
                                            <div class="w-3 h-3 rounded-full bg-[#D4A356]"
                                                x-show="form.payment_method === 'cod'"></div>
                                        </div>
                                        <div>
                                            <span class="text-white font-bold block">الدفع عند الاستلام</span>
                                            <span class="text-gray-500 text-sm">ادفع نقداً عند استلام الطلب</span>
                                        </div>
                                    </div>
                                    <svg class="w-8 h-8 text-[#D4A356]" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                            d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z" />
                                    </svg>
                                </label>
                            </div>
                        </div>
                    </div>

                    {{-- Order Summary Sidebar --}}
                    <div class="lg:col-span-1">
                        <div class="bg-[#111111] rounded-2xl border border-white/10 p-6 sticky top-24">
                            <h2 class="text-white font-black text-xl mb-6">ملخص الطلب</h2>

                            {{-- Cart Items Preview --}}
                            <div class="space-y-3 max-h-[200px] overflow-y-auto mb-6">
                                @foreach ($items as $item)
                                    <div class="flex items-center gap-3 p-3 bg-white/5 rounded-xl">
                                        <div class="w-12 h-12 rounded-lg overflow-hidden bg-[#1a1a1a] shrink-0">
                                            @if ($item['image'])
                                                <img src="{{ $item['image'] }}" alt="{{ $item['name'] }}"
                                                    class="w-full h-full object-cover">
                                            @endif
                                        </div>
                                        <div class="flex-1 min-w-0">
                                            <h4 class="text-white text-sm font-bold line-clamp-1">{{ $item['name'] }}</h4>
                                            <p class="text-gray-500 text-xs">x{{ $item['quantity'] }}</p>
                                        </div>
                                        <span
                                            class="text-[#D4A356] font-bold text-sm shrink-0">{{ number_format($item['line_total']) }}</span>
                                    </div>
                                @endforeach
                            </div>

                            {{-- Coupon Section --}}
                            <div class="mb-6 p-4 bg-white/5 rounded-xl border border-white/10">
                                <div class="flex items-center gap-2 mb-3">
                                    <svg class="w-5 h-5 text-[#D4A356]" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M7 7h.01M7 3h5c.512 0 1.024.195 1.414.586l7 7a2 2 0 010 2.828l-7 7a2 2 0 01-2.828 0l-7-7A1.994 1.994 0 013 12V7a4 4 0 014-4z" />
                                    </svg>
                                    <span class="text-white font-bold text-sm">كود الخصم</span>
                                </div>

                                @if ($availableCoupons->isNotEmpty())
                                    <div class="flex gap-2 overflow-x-auto pb-2 mb-3 scrollbar-thin"
                                        x-show="!couponApplied">
                                        @foreach ($availableCoupons as $coupon)
                                            @if ($coupon['can_use'])
                                                <button type="button"
                                                    @click="couponCode = '{{ $coupon['code'] }}'; applyCoupon()"
                                                    :disabled="couponLoading"
                                                    class="flex-shrink-0 px-3 py-2 bg-[#D4A356]/10 rounded-lg border border-[#D4A356]/30 hover:bg-[#D4A356]/20 transition-colors">
                                                    <span
                                                        class="text-[#D4A356] font-bold text-xs">{{ $coupon['code'] }}</span>
                                                    <span
                                                        class="text-gray-400 text-xs block">{{ $coupon['description'] }}</span>
                                                </button>
                                            @endif
                                        @endforeach
                                    </div>
                                @endif

                                <template x-if="!couponApplied">
                                    <div>
                                        <div class="flex gap-2">
                                            <input type="text" x-model="couponCode" placeholder="أدخل كود الخصم"
                                                class="flex-1 px-3 py-2 rounded-lg bg-white/5 border border-white/10 text-white text-sm placeholder-gray-500 outline-none focus:border-[#D4A356]">
                                            <button type="button" @click="applyCoupon()" :disabled="couponLoading"
                                                class="px-4 py-2 rounded-lg bg-[#D4A356] text-black font-bold text-sm hover:bg-[#442247] transition-colors disabled:opacity-50">
                                                تطبيق
                                            </button>
                                        </div>
                                        <p x-show="couponError" x-text="couponError" class="mt-2 text-red-400 text-xs">
                                        </p>
                                    </div>
                                </template>
                                <template x-if="couponApplied">
                                    <div
                                        class="flex items-center justify-between p-3 bg-emerald-500/10 border border-emerald-500/20 rounded-lg">
                                        <div class="flex items-center gap-2">
                                            <svg class="w-5 h-5 text-emerald-400" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                                            </svg>
                                            <span class="text-emerald-400 font-bold text-sm"
                                                x-text="appliedCouponCode"></span>
                                        </div>
                                        <button type="button" @click="removeCoupon()"
                                            class="text-red-400 hover:text-red-300">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M6 18L18 6M6 6l12 12" />
                                            </svg>
                                        </button>
                                    </div>
                                </template>
                                <input type="hidden" name="coupon_code" :value="appliedCouponCode">
                            </div>

                            {{-- Totals --}}
                            <div class="space-y-3 border-t border-white/10 pt-4">
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-400">المجموع الفرعي</span>
                                    <span class="text-white font-bold">{{ number_format($subtotal) }} د.ع</span>
                                </div>

                                <div class="flex justify-between items-center" x-show="discount > 0">
                                    <span class="text-emerald-400">الخصم</span>
                                    <span class="text-emerald-400 font-bold" x-text="'-' + formatIQD(discount)"></span>
                                </div>

                                <div class="flex justify-between items-center">
                                    <span class="text-gray-400">التوصيل</span>
                                    <span class="text-white font-bold"
                                        x-text="shipping === 0 ? 'مجاني' : formatIQD(shipping)"></span>
                                </div>

                                <div x-show="shipping === 0"
                                    class="flex items-center gap-2 p-2 bg-emerald-500/10 rounded-lg border border-emerald-500/20">
                                    <svg class="w-4 h-4 text-emerald-400" fill="none" stroke="currentColor"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                            d="M5 13l4 4L19 7" />
                                    </svg>
                                    <span class="text-emerald-400 text-xs">توصيل مجاني!</span>
                                </div>

                                <div class="border-t border-white/10 pt-4">
                                    <div class="flex justify-between items-center">
                                        <span class="text-white font-bold text-lg">الإجمالي</span>
                                        <span class="text-[#D4A356] font-black text-2xl"
                                            x-text="formatIQD(getTotal())"></span>
                                    </div>
                                </div>
                            </div>

                            {{-- Submit Button --}}
                            <button type="submit" :disabled="isSubmitting"
                                class="w-full mt-6 flex items-center justify-center gap-3 bg-gradient-to-r from-[#D4A356] to-[#442247] text-black px-8 py-4 rounded-xl font-black text-lg hover:shadow-lg hover:shadow-[#D4A356]/20 transition-all disabled:opacity-50">
                                <span x-show="!isSubmitting">تأكيد الطلب</span>
                                <span x-show="isSubmitting" class="flex items-center gap-2">
                                    <svg class="animate-spin w-5 h-5" fill="none" viewBox="0 0 24 24">
                                        <circle class="opacity-25" cx="12" cy="12" r="10"
                                            stroke="currentColor" stroke-width="4"></circle>
                                        <path class="opacity-75" fill="currentColor"
                                            d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4z"></path>
                                    </svg>
                                    جاري الإرسال...
                                </span>
                            </button>

                            <div class="mt-4 flex items-center justify-center gap-2 text-gray-500 text-sm">
                                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                        d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
                                </svg>
                                دفع آمن ومشفر
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </section>
@endsection

@push('scripts')
    <script>
        function checkoutPage() {
            // City shipping fees map
            const cityFees = @json($cities->pluck('shipping_fee', 'id'));
            const freeThreshold = {{ $freeShippingThreshold }};

            // Default address data
            const defaultAddr = @json($defaultAddress);
            const client = @json($client);

            return {
                subtotal: {{ $subtotal }},
                shipping: {{ $shipping }},
                discount: 0,
                couponCode: '',
                couponApplied: false,
                appliedCouponCode: '',
                couponError: '',
                couponLoading: false,
                isSubmitting: false,
                selectedAddressId: defaultAddr ? defaultAddr.id : null,
                useExistingAddress: defaultAddr ? true : false,

                form: {
                    name: defaultAddr?.name || client?.name || '',
                    phone: defaultAddr?.phone || client?.phone || '',
                    city_id: defaultAddr?.city_id || client?.city_id || '',
                    area: '',
                    address: '',
                    notes: '',
                    payment_method: 'cod'
                },

                // Store original address data for comparison
                originalAddress: defaultAddr ? {
                    ...defaultAddr
                } : null,

                init() {
                    // Parse address if default exists
                    if (defaultAddr && defaultAddr.address) {
                        const parts = defaultAddr.address.split(', ');
                        this.form.area = parts[0] || '';
                        this.form.address = parts.slice(1).join(', ') || '';
                    }

                    // Set initial shipping based on city
                    if (this.form.city_id && cityFees[this.form.city_id]) {
                        this.shipping = this.subtotal >= freeThreshold ? 0 : cityFees[this.form.city_id];
                    }

                    // Watch form submission
                    this.$watch('isSubmitting', (val) => {
                        if (val) {
                            document.getElementById('checkout-form').submit();
                        }
                    });
                },

                selectAddress(addr) {
                    this.selectedAddressId = addr.id;
                    this.useExistingAddress = true;
                    this.originalAddress = {
                        ...addr
                    };
                    this.form.name = addr.name || '';
                    this.form.phone = addr.phone || '';
                    this.form.city_id = addr.city_id || '';

                    // Parse address
                    if (addr.address) {
                        const parts = addr.address.split(', ');
                        this.form.area = parts[0] || '';
                        this.form.address = parts.slice(1).join(', ') || '';
                    }

                    this.updateShipping();
                },

                // Check if form data differs from selected address
                hasAddressChanged() {
                    if (!this.originalAddress) return true;

                    const currentAddress = this.form.area + ', ' + this.form.address;
                    return (
                        this.form.name !== this.originalAddress.name ||
                        this.form.phone !== this.originalAddress.phone ||
                        this.form.city_id != this.originalAddress.city_id ||
                        currentAddress !== this.originalAddress.address
                    );
                },

                // Called when any address field changes
                onAddressFieldChange() {
                    if (this.hasAddressChanged()) {
                        this.useExistingAddress = false;
                    }
                },

                updateShipping() {
                    if (this.form.city_id && cityFees[this.form.city_id]) {
                        const subtotalAfterDiscount = this.subtotal - this.discount;
                        this.shipping = subtotalAfterDiscount >= freeThreshold ? 0 : cityFees[this.form.city_id];
                    }
                },

                async applyCoupon() {
                    if (!this.couponCode.trim()) {
                        this.couponError = 'يرجى إدخال كود الخصم';
                        return;
                    }

                    this.couponLoading = true;
                    this.couponError = '';

                    try {
                        const response = await fetch('{{ route('checkout.apply-coupon') }}', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                                'Accept': 'application/json'
                            },
                            body: JSON.stringify({
                                code: this.couponCode,
                                subtotal: this.subtotal
                            })
                        });

                        const data = await response.json();

                        if (data.success) {
                            this.couponApplied = true;
                            this.appliedCouponCode = data.code;
                            this.discount = data.discount;
                            this.couponCode = '';
                            this.updateShipping();
                        } else {
                            this.couponError = data.message || 'كود الخصم غير صالح';
                        }
                    } catch (error) {
                        this.couponError = 'حدث خطأ، يرجى المحاولة مرة أخرى';
                    } finally {
                        this.couponLoading = false;
                    }
                },

                removeCoupon() {
                    this.couponApplied = false;
                    this.appliedCouponCode = '';
                    this.discount = 0;
                    this.updateShipping();
                },

                getTotal() {
                    return (this.subtotal - this.discount) + this.shipping;
                },

                formatIQD(amount) {
                    return new Intl.NumberFormat('en-US').format(amount) + ' د.ع';
                }
            }
        }
    </script>
@endpush
