@extends('layouts.master')

@section('title', 'سلة التسوق - نساج')

@section('hero_title', 'سلة التسوق')
@section('hero_description', 'راجع منتجاتك وأتم عملية الشراء بسهولة')
@section('hero_breadcrumbs')
    <span class="text-white/30 mx-0.5">›</span>
    <span class="text-nassaj-gold">سلة التسوق</span>
@endsection

@section('content')
    {{-- Cart Content --}}
    <section class="py-8 bg-[#0a0a0a]" x-data="cartPage(@js($items->toArray()), {{ $subtotal ?? 0 }}, {{ $freeShippingThreshold ?? 100000 }}, {{ $shipping ?? 0 }})">
        <div class="site-container">
            <template x-if="items.length > 0">
                <div class="flex flex-col lg:flex-row gap-8">
                    {{-- Sidebar (Left in RTL) --}}
                    <div class="lg:w-[380px] shrink-0 space-y-5 order-2 lg:order-1">
                        {{-- Free Shipping Progress --}}
                        <div class="bg-[#111111] rounded-2xl border border-white/10 p-5">
                            <template x-if="shippingProgress >= 100">
                                <p class="text-emerald-400 text-sm font-bold text-center mb-3">🎉 تهانينا! حصلت على توصيل
                                    مجاني</p>
                            </template>
                            <template x-if="shippingProgress < 100">
                                <p class="text-gray-300 text-sm text-center mb-3">
                                    أضف <span class="text-[#D4A356] font-bold"
                                        x-text="formatPrice(remainingForFreeShipping)"></span> أخرى للحصول على توصيل مجاني!
                                </p>
                            </template>
                            <div class="w-full bg-white/10 rounded-full h-2.5 overflow-hidden">
                                <div class="h-full bg-gradient-to-r from-[#D4A356] to-[#442247] rounded-full transition-all duration-300"
                                    :style="'width: ' + shippingProgress + '%'"></div>
                            </div>
                        </div>

                        {{-- Coupon Section --}}
                        <div class="bg-[#111111] rounded-2xl border border-white/10 p-5">
                            <div class="flex items-center gap-2 mb-4">
                                <span class="text-xl">🎁</span>
                                <h3 class="text-[#D4A356] font-bold">أكواد الخصم</h3>
                            </div>

                            {{-- Coupon Input --}}
                            <div class="flex gap-2 mb-5">
                                <input type="text" x-model="couponCode" placeholder="أدخل كود الخصم"
                                    class="flex-1 px-4 py-3 rounded-xl bg-white/5 border border-white/10 text-white placeholder-gray-500 outline-none focus:border-[#D4A356] transition-colors text-sm">
                                <button @click="applyCoupon()"
                                    class="px-5 py-3 rounded-xl bg-white/5 border border-white/10 text-white font-bold hover:bg-white/10 transition-colors text-sm">
                                    تطبيق
                                </button>
                            </div>

                            {{-- Available Coupons --}}
                            @if (isset($coupons) && $coupons->isNotEmpty())
                                <div class="space-y-3">
                                    @foreach ($coupons->take(4) as $coupon)
                                        <div class="p-3 bg-white/5 rounded-xl border border-white/5">
                                            <div class="flex items-center justify-between mb-2">
                                                <span
                                                    class="text-white font-mono font-bold text-sm">{{ $coupon->code }}</span>
                                                <span
                                                    class="px-2 py-1 rounded-lg text-xs font-bold {{ $coupon->type === 'percentage' ? 'bg-[#D4A356]/20 text-[#D4A356]' : 'bg-emerald-500/20 text-emerald-400' }}">
                                                    @if ($coupon->type === 'percentage')
                                                        خصم {{ (int) $coupon->discount_value }}%
                                                    @else
                                                        خصم {{ number_format($coupon->discount_value) }} د.ع
                                                    @endif
                                                </span>
                                            </div>
                                            <div class="flex items-center justify-between">
                                                <span
                                                    class="text-gray-400 text-xs">{{ $coupon->description ?? $coupon->name }}</span>
                                                <button @click="couponCode = '{{ $coupon->code }}'; applyCoupon()"
                                                    class="px-3 py-1.5 rounded-lg bg-white/10 text-white text-xs font-bold hover:bg-white/20 transition-colors">
                                                    تطبيق
                                                </button>
                                            </div>
                                        </div>
                                    @endforeach
                                </div>
                            @endif
                        </div>

                        {{-- Order Summary --}}
                        <div class="bg-[#111111] rounded-2xl border border-white/10 p-5">
                            <div class="space-y-4 mb-5">
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-400">المجموع الفرعي</span>
                                    <span class="text-white font-bold" x-text="formatPrice(subtotal)"></span>
                                </div>
                                <template x-if="discount > 0">
                                    <div class="flex justify-between items-center">
                                        <span class="text-gray-400">الخصم</span>
                                        <span class="text-emerald-400 font-bold" x-text="formatPrice(discount)"></span>
                                    </div>
                                </template>
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-400">التوصيل</span>
                                    <span class="text-white font-bold"
                                        x-text="calculatedShipping > 0 ? formatPrice(calculatedShipping) : 'مجاني'"></span>
                                </div>
                                <div class="border-t border-white/10 pt-4">
                                    <div class="flex justify-between items-center">
                                        <span class="text-white font-bold text-lg">الإجمالي</span>
                                        <span class="text-[#D4A356] font-black text-2xl" x-text="formatPrice(total)"></span>
                                    </div>
                                </div>
                            </div>

                            {{-- Checkout Button --}}
                            <a href="{{ route('checkout.index') }}"
                                class="w-full flex items-center justify-center gap-3 bg-gradient-to-r from-[#D4A356] to-[#442247] text-black px-8 py-4 rounded-xl font-black text-lg hover:shadow-lg hover:shadow-[#D4A356]/20 transition-all">
                                إتمام الطلب
                            </a>

                            <a href="{{ route('shop') }}"
                                class="w-full mt-3 flex items-center justify-center gap-2 bg-white/5 border border-white/10 text-white px-8 py-3 rounded-xl font-bold hover:bg-white/10 transition-colors">
                                متابعة التسوق
                            </a>
                        </div>
                    </div>

                    {{-- Products Table (Right in RTL) --}}
                    <div class="flex-1 order-1 lg:order-2">
                        <div class="bg-[#111111] rounded-2xl border border-white/10 overflow-hidden">
                            {{-- Table Header --}}
                            <div
                                class="hidden md:grid grid-cols-12 gap-4 px-6 py-4 bg-[#1a1a0f] border-b border-[#D4A356]/30">
                                <div class="col-span-5 text-right">
                                    <span class="text-white font-bold">المنتج</span>
                                </div>
                                <div class="col-span-2 text-center">
                                    <span class="text-white font-bold">السعر</span>
                                </div>
                                <div class="col-span-2 text-center">
                                    <span class="text-white font-bold">الكمية</span>
                                </div>
                                <div class="col-span-2 text-center">
                                    <span class="text-white font-bold">المجموع</span>
                                </div>
                                <div class="col-span-1 text-center">
                                    <span class="text-white font-bold">إجراءات</span>
                                </div>
                            </div>

                            {{-- Cart Items --}}
                            <div class="divide-y divide-white/5">
                                <template x-for="(item, index) in items" :key="item.id">
                                    <div class="p-4 md:px-6 md:py-5 hover:bg-white/[0.02] transition-colors">
                                        {{-- Desktop Layout --}}
                                        <div class="hidden md:grid grid-cols-12 gap-4 items-center">
                                            {{-- Product Info --}}
                                            <div class="col-span-5 flex items-center gap-4">
                                                <div
                                                    class="w-16 h-16 rounded-xl overflow-hidden bg-[#1a1a1a] border border-[#D4A356]/30 shrink-0">
                                                    <img :src="item.image || '/images/placeholder.png'"
                                                        :alt="item.name" class="w-full h-full object-cover">
                                                </div>
                                                <div class="min-w-0">
                                                    <a :href="'/shop/product/' + item.slug"
                                                        class="text-white font-bold hover:text-[#D4A356] transition-colors line-clamp-1"
                                                        x-text="item.name"></a>
                                                    <p class="text-gray-500 text-sm mt-0.5" x-show="item.variant_name"
                                                        x-text="item.variant_name"></p>
                                                </div>
                                            </div>

                                            {{-- Price --}}
                                            <div class="col-span-2 text-center">
                                                <span class="text-white font-bold"
                                                    x-text="formatPrice(item.unit_price)"></span>
                                            </div>

                                            {{-- Quantity --}}
                                            <div class="col-span-2 flex items-center justify-center">
                                                <div
                                                    class="flex items-center border border-white/20 rounded-lg overflow-hidden">
                                                    <button @click="incrementQty(index)"
                                                        class="w-10 h-10 text-white flex items-center justify-center hover:bg-white/10 transition-colors font-bold text-lg border-l border-white/20">
                                                        +
                                                    </button>
                                                    <span class="text-white font-bold w-12 text-center"
                                                        x-text="item.quantity"></span>
                                                    <button @click="decrementQty(index)"
                                                        class="w-10 h-10 text-white flex items-center justify-center hover:bg-white/10 transition-colors font-bold text-lg border-r border-white/20">
                                                        −
                                                    </button>
                                                </div>
                                            </div>

                                            {{-- Total --}}
                                            <div class="col-span-2 text-center">
                                                <span class="text-white font-bold"
                                                    x-text="formatPrice(item.unit_price * item.quantity)"></span>
                                            </div>

                                            {{-- Delete --}}
                                            <div class="col-span-1 flex justify-center">
                                                <button @click="removeItem(index)"
                                                    class="flex items-center gap-1.5 px-3 py-2 rounded-lg border border-red-500/50 bg-red-500/10 text-red-400 hover:bg-red-500/20 transition-colors text-sm">
                                                    <svg class="w-4 h-4" fill="none" stroke="currentColor"
                                                        viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round"
                                                            stroke-width="2"
                                                            d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                                    </svg>
                                                    حذف
                                                </button>
                                            </div>
                                        </div>

                                        {{-- Mobile Layout --}}
                                        <div class="md:hidden flex gap-4">
                                            <div
                                                class="w-20 h-20 rounded-xl overflow-hidden bg-[#1a1a1a] border border-[#D4A356]/30 shrink-0">
                                                <img :src="item.image || '/images/placeholder.png'" :alt="item.name"
                                                    class="w-full h-full object-cover">
                                            </div>
                                            <div class="flex-1 min-w-0">
                                                <a :href="'/shop/product/' + item.slug"
                                                    class="text-white font-bold hover:text-[#D4A356] transition-colors line-clamp-1"
                                                    x-text="item.name"></a>
                                                <p class="text-gray-500 text-xs" x-show="item.variant_name"
                                                    x-text="item.variant_name"></p>
                                                <p class="text-[#D4A356] font-bold text-sm mt-1"
                                                    x-text="formatPrice(item.unit_price)"></p>

                                                <div class="flex items-center justify-between mt-2">
                                                    <div
                                                        class="flex items-center border border-white/20 rounded-lg overflow-hidden">
                                                        <button @click="incrementQty(index)"
                                                            class="w-8 h-8 text-white flex items-center justify-center hover:bg-white/10 transition-colors text-sm border-l border-white/20">+</button>
                                                        <span class="text-white font-bold w-8 text-center text-sm"
                                                            x-text="item.quantity"></span>
                                                        <button @click="decrementQty(index)"
                                                            class="w-8 h-8 text-white flex items-center justify-center hover:bg-white/10 transition-colors text-sm border-r border-white/20">−</button>
                                                    </div>
                                                    <div class="flex items-center gap-2">
                                                        <span class="text-white font-bold text-sm"
                                                            x-text="formatPrice(item.unit_price * item.quantity)"></span>
                                                        <button @click="removeItem(index)"
                                                            class="p-1.5 rounded border border-red-500/50 bg-red-500/10 text-red-400 hover:bg-red-500/20 transition-colors">
                                                            <svg class="w-4 h-4" fill="none" stroke="currentColor"
                                                                viewBox="0 0 24 24">
                                                                <path stroke-linecap="round" stroke-linejoin="round"
                                                                    stroke-width="2"
                                                                    d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                                            </svg>
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </template>
                            </div>
                        </div>
                    </div>
                </div>
            </template>

            {{-- Empty State --}}
            <template x-if="items.length === 0">
                <div class="text-center py-20">
                    <div class="w-32 h-32 mx-auto mb-8 rounded-full bg-white/5 flex items-center justify-center">
                        <svg class="w-16 h-16 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5"
                                d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z" />
                        </svg>
                    </div>
                    <h3 class="text-white font-black text-3xl mb-4">سلتك فارغة</h3>
                    <p class="text-gray-400 mb-8 max-w-md mx-auto text-lg">لم تقم بإضافة أي منتجات للسلة بعد. تصفح منتجاتنا
                        المميزة وأضف ما يعجبك!</p>
                    <a href="{{ route('shop') }}"
                        class="inline-flex items-center gap-3 bg-gradient-to-r from-[#D4A356] to-[#442247] text-black px-10 py-4 rounded-full font-black text-lg hover:shadow-lg hover:shadow-[#D4A356]/20 transition-all hover:scale-[1.02]">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z" />
                        </svg>
                        تسوق الآن
                    </a>
                </div>
            </template>
        </div>
    </section>
@endsection

@push('scripts')
    <script>
        function cartPage(initialItems, initialSubtotal, freeShippingThreshold, baseShipping) {
            return {
                items: initialItems || [],
                freeShippingThreshold: freeShippingThreshold,
                baseShipping: baseShipping,
                couponCode: '',
                discount: 0,
                syncTimeout: null,

                // Computed properties
                get subtotal() {
                    return this.items.reduce((sum, item) => sum + (item.unit_price * item.quantity), 0);
                },

                get calculatedShipping() {
                    return this.subtotal >= this.freeShippingThreshold ? 0 : this.baseShipping;
                },

                get total() {
                    return this.subtotal - this.discount + this.calculatedShipping;
                },

                get shippingProgress() {
                    return Math.min(100, (this.subtotal / this.freeShippingThreshold) * 100);
                },

                get remainingForFreeShipping() {
                    return Math.max(0, this.freeShippingThreshold - this.subtotal);
                },

                get totalItems() {
                    return this.items.reduce((sum, item) => sum + item.quantity, 0);
                },

                formatPrice(amount) {
                    return new Intl.NumberFormat('en-US').format(amount) + ' د.ع';
                },

                incrementQty(index) {
                    this.items[index].quantity++;
                    this.scheduleSync(this.items[index]);
                },

                decrementQty(index) {
                    if (this.items[index].quantity > 1) {
                        this.items[index].quantity--;
                        this.scheduleSync(this.items[index]);
                    } else {
                        this.removeItem(index);
                    }
                },

                removeItem(index) {
                    const item = this.items[index];
                    this.items.splice(index, 1);
                    // Sync removal with server
                    this.syncRemove(item.id);
                },

                scheduleSync(item) {
                    // Debounce server sync - wait 500ms after last change
                    clearTimeout(this.syncTimeout);
                    this.syncTimeout = setTimeout(() => {
                        this.syncQuantity(item.id, item.quantity);
                    }, 500);
                },

                syncQuantity(itemId, quantity) {
                    // Silent background sync with server
                    Livewire.dispatch('cart:update-quantity', {
                        itemId: itemId,
                        quantity: quantity
                    });
                },

                syncRemove(itemId) {
                    // Sync removal with server
                    Livewire.dispatch('cart:remove-item', {
                        itemId: itemId
                    });
                },

                applyCoupon() {
                    if (!this.couponCode.trim()) {
                        alert('يرجى إدخال كود الخصم');
                        return;
                    }
                    // TODO: Implement coupon validation via API
                    alert('تم تطبيق كود الخصم: ' + this.couponCode);
                }
            };
        }
    </script>
@endpush
