<div x-data="wishlistDrawer(@js($items->toArray()))" x-on:wishlist-open.window="open = true"
    x-on:wishlist-add-instant.window="addItemInstant($event.detail)" @keydown.escape.window="if(open) open = false">
    {{-- Wishlist Overlay --}}
    <div x-show="open" x-cloak class="fixed inset-0 z-[9999]">
        {{-- Backdrop --}}
        <div x-show="open" x-transition:enter="transition ease-out duration-100" x-transition:enter-start="opacity-0"
            x-transition:enter-end="opacity-100" x-transition:leave="transition ease-in duration-75"
            x-transition:leave-start="opacity-100" x-transition:leave-end="opacity-0" @click="open = false"
            class="absolute inset-0 bg-black/60 backdrop-blur-sm"></div>

        {{-- Drawer --}}
        <div x-show="open" x-transition:enter="transition ease-out duration-100"
            x-transition:enter-start="translate-x-full" x-transition:enter-end="translate-x-0"
            x-transition:leave="transition ease-in duration-75" x-transition:leave-start="translate-x-0"
            x-transition:leave-end="translate-x-full"
            class="absolute left-0 top-0 h-full w-full max-w-md bg-[#0a0a0a] border-r border-white/10 flex flex-col shadow-2xl">
            {{-- Header --}}
            <div class="p-6 border-b border-white/10">
                <div class="flex items-center justify-between">
                    <h2 class="cairo text-2xl font-bold text-white flex items-center gap-2">
                        <svg class="w-6 h-6 text-red-500" fill="currentColor" viewBox="0 0 24 24">
                            <path
                                d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                        </svg>
                        المفضلة (<span x-text="localItems.length"></span>)
                    </h2>
                    <button @click="open = false" class="p-2 hover:bg-white/5 rounded-full transition-colors">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
            </div>

            {{-- Items --}}
            <div class="flex-1 overflow-y-auto p-6">
                <template x-if="localItems.length === 0">
                    <div class="text-center py-12 text-gray-400">
                        <svg class="w-24 h-24 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor"
                            viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                        </svg>
                        <p class="text-lg">قائمة المفضلة فارغة</p>
                    </div>
                </template>

                <template x-if="localItems.length > 0">
                    <div class="space-y-4">
                        <template x-for="(item, index) in localItems" :key="item.id">
                            <div class="flex gap-4 bg-white/5 rounded-xl p-4 border border-white/10">
                                {{-- Image --}}
                                <div class="w-20 h-20 flex-shrink-0 rounded-lg overflow-hidden bg-white/10">
                                    <template x-if="item.image">
                                        <img :src="item.image" :alt="item.name"
                                            class="w-full h-full object-cover">
                                    </template>
                                    <template x-if="!item.image">
                                        <div class="w-full h-full flex items-center justify-center text-gray-500">
                                            <svg class="w-8 h-8" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                            </svg>
                                        </div>
                                    </template>
                                </div>

                                {{-- Details --}}
                                <div class="flex-1 min-w-0">
                                    <span class="font-bold text-white line-clamp-2" x-text="item.name"></span>
                                    <p class="text-[#D4A356] font-bold mt-2" x-text="formatPrice(item.price)"></p>

                                    <div class="flex items-center gap-2 mt-3">
                                        <button @click="addToCart(item)"
                                            class="flex-1 bg-gradient-to-r from-[#D4A356] to-[#F5D799] text-black py-2 rounded-lg font-bold text-sm hover:shadow-lg transition-all">
                                            أضف للسلة
                                        </button>
                                        <button @click="removeItem(index)"
                                            class="p-2 text-red-400 hover:bg-red-500/20 rounded-lg transition-colors">
                                            <svg class="w-5 h-5" fill="none" stroke="currentColor"
                                                viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                                    d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                            </svg>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </template>

                        {{-- Clear All --}}
                        <button @click="clearAll()"
                            class="w-full mt-4 text-red-400 hover:text-red-300 text-sm transition-colors">
                            تفريغ المفضلة
                        </button>
                    </div>
                </template>
            </div>

            {{-- Footer --}}
            <template x-if="localItems.length > 0">
                <div class="p-6 border-t border-white/10">
                    <button @click="addAllToCart()"
                        class="w-full bg-gradient-to-r from-[#D4A356] to-[#F5D799] text-black py-4 rounded-full font-bold text-lg hover:shadow-2xl transition-all hover:scale-[1.02]">
                        إضافة الكل للسلة
                    </button>
                </div>
            </template>
        </div>
    </div>
</div>

<script>
    function wishlistDrawer(initialItems = []) {
        return {
            open: false,
            localItems: initialItems,

            formatPrice(val) {
                return new Intl.NumberFormat('en-US').format(val) + ' د.ع';
            },

            addItemInstant(detail) {
                // Check if item already exists (use product_id for server items, id for client items)
                const existingIndex = this.localItems.findIndex(item => (item.product_id || item.id) === detail.id);

                if (existingIndex < 0) {
                    // Add new item
                    this.localItems.push({
                        id: detail.id,
                        product_id: detail.id,
                        name: detail.name,
                        price: detail.price,
                        image: detail.image || null
                    });
                }

                // Open drawer immediately after adding
                this.open = true;

                // Sync with server in background
                Livewire.dispatch('wishlist:toggle', {
                    productId: detail.id
                });
            },

            removeItem(index) {
                const item = this.localItems[index];
                this.localItems.splice(index, 1);
                Livewire.dispatch('wishlist:toggle', {
                    productId: item.product_id || item.id
                });
            },

            addToCart(item) {
                window.dispatchEvent(new CustomEvent('cart-add-instant', {
                    detail: {
                        id: item.product_id || item.id,
                        name: item.name,
                        price: item.price,
                        image: item.image
                    }
                }));
            },

            addAllToCart() {
                this.localItems.forEach(item => {
                    window.dispatchEvent(new CustomEvent('cart-add-instant', {
                        detail: {
                            id: item.product_id || item.id,
                            name: item.name,
                            price: item.price,
                            image: item.image
                        }
                    }));
                });
                this.localItems = [];
                Livewire.dispatch('wishlist:clear');
            },

            clearAll() {
                this.localItems = [];
                Livewire.dispatch('wishlist:clear');
            }
        };
    }
</script>
