<div x-data="{
    open: false,
    qty: 1,
    product: null,
    variants: [],
    selectedVariant: null,
    selectedColor: null,
    selectedSize: null,
    colors: [],
    sizes: [],
    images: [],
    currentImage: 0,
    price: 0,
    oldPrice: 0,
    loading: false,
    attributeNames: { color: 'اللون', size: 'المقاس' },

    async loadProduct(productId) {
        this.open = true;
        this.loading = true;
        this.qty = 1;
        this.currentImage = 0;

        try {
            const response = await fetch(`/api/product/${productId}/quick-view`);
            const data = await response.json();

            if (data.success) {
                this.product = data.product;
                this.variants = data.product.variants || [];
                this.images = data.images || [];
                this.price = data.product.price || 0;
                this.oldPrice = data.product.old_price || 0;
                this.attributeNames = data.attributeNames || { color: 'اللون', size: 'المقاس' };

                // Extract unique colors and sizes from variants
                this.extractOptions();

                // Select default variant
                this.selectedVariant = this.variants.find(v => v.is_default) || this.variants[0] || null;
                if (this.selectedVariant) {
                    this.price = this.selectedVariant.price || this.price;
                    this.oldPrice = this.selectedVariant.old_price || this.oldPrice;
                }
            }
        } catch (e) {
            console.error(e);
        }

        this.loading = false;
    },

    extractOptions() {
        const colorSet = new Map();
        const sizeSet = new Map();

        this.variants.forEach(v => {
            if (v.color_name && v.color_hex) {
                colorSet.set(v.color_name, { name: v.color_name, hex: v.color_hex, id: v.id });
            }
            if (v.size_name) {
                sizeSet.set(v.size_name, { name: v.size_name, id: v.id });
            }
        });

        this.colors = Array.from(colorSet.values());
        this.sizes = Array.from(sizeSet.values());

        if (this.colors.length > 0) this.selectedColor = this.colors[0];
        if (this.sizes.length > 0) this.selectedSize = this.sizes[0];
    },

    selectColor(color) {
        this.selectedColor = color;
        const variant = this.variants.find(v => v.color_name === color.name);
        if (variant) {
            this.selectedVariant = variant;
            this.price = variant.price || this.price;
            this.oldPrice = variant.old_price || this.oldPrice;
        }
    },

    selectSize(size) {
        this.selectedSize = size;
        const variant = this.variants.find(v => v.size_name === size.name);
        if (variant) {
            this.selectedVariant = variant;
            this.price = variant.price || this.price;
            this.oldPrice = variant.old_price || this.oldPrice;
        }
    },

    addToCart() {
        if (this.product) {
            Livewire.dispatch('cart:add', {
                id: this.product.id,
                qty: this.qty,
                variantId: this.selectedVariant?.id
            });
            this.$dispatch('cart-open');
            this.open = false;
        }
    },

    addToWishlist() {
        if (this.product) {
            Livewire.dispatch('wishlist:toggle', { productId: this.product.id });
        }
    },

    addToCompare() {
        if (this.product) {
            Livewire.dispatch('compare:toggle', { productId: this.product.id });
        }
    },

    share() {
        if (this.product) {
            const url = window.location.origin + '/shop/product/' + this.product.slug;
            if (navigator.share) {
                navigator.share({ title: this.product.name, url: url });
            } else {
                navigator.clipboard.writeText(url);
                alert('تم نسخ الرابط!');
            }
        }
    },

    customize() {
        if (this.product) {
            this.close();
            setTimeout(() => {
                window.location.href = '/configurator/' + this.product.id;
            }, 300);
        }
    },

    formatPrice(val) {
        return new Intl.NumberFormat('en-US').format(val) + ' د.ع';
    },

    close() {
        this.open = false;
    }
}" x-on:quickview-open.window="loadProduct($event.detail.productId)"
    @keydown.escape.window="if(open) close()">

    {{-- QuickView Modal --}}
    <template x-teleport="body">
        <div x-show="open" x-cloak class="fixed inset-0 z-[9999] flex items-center justify-center p-4"
            style="backdrop-filter: blur(8px);">
            {{-- Backdrop --}}
            <div x-show="open" x-transition:enter="transition ease-out duration-300"
                x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100"
                x-transition:leave="transition ease-in duration-200" x-transition:leave-start="opacity-100"
                x-transition:leave-end="opacity-0" @click="close()" class="absolute inset-0 bg-black/90 backdrop-blur-md">
            </div>

            {{-- Modal Content --}}
            <div x-show="open" x-transition:enter="transition ease-out duration-300 transform"
                x-transition:enter-start="opacity-0 scale-95" x-transition:enter-end="opacity-100 scale-100"
                x-transition:leave="transition ease-in duration-200 transform"
                x-transition:leave-start="opacity-100 scale-100" x-transition:leave-end="opacity-0 scale-95"
                class="relative w-full max-w-6xl bg-white dark:bg-[#1a1a1a] rounded-3xl overflow-hidden shadow-2xl border border-gray-200 dark:border-white/10 max-h-[90vh] overflow-y-auto"
                dir="rtl">

                {{-- Close Button --}}
                <button @click="close()"
                    class="absolute top-5 left-5 z-50 w-11 h-11 rounded-full bg-white dark:bg-white/5 hover:bg-gray-100 dark:hover:bg-white/10 border border-gray-200 dark:border-white/10 flex items-center justify-center transition-all hover:rotate-90">
                    <svg class="w-5 h-5 text-gray-800 dark:text-white" fill="none" stroke="currentColor"
                        viewBox="0 0 24 24" stroke-width="2.5">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>

                {{-- Loading State --}}
                <div x-show="loading" class="min-h-[600px] flex items-center justify-center">
                    <div class="text-center">
                        <div
                            class="w-16 h-16 mx-auto mb-4 rounded-full border-4 border-nassaj-gold/20 border-t-nassaj-gold animate-spin">
                        </div>
                        <p class="text-gray-600 dark:text-gray-400 cairo font-bold">جاري التحميل...</p>
                    </div>
                </div>

                {{-- Content --}}
                <div x-show="!loading && product" class="grid lg:grid-cols-2 gap-0">
                    {{-- Right Side - Product Images --}}
                    <div class="relative bg-gray-50 dark:bg-white/5 flex items-center justify-center p-8 lg:p-12">
                        {{-- Main Image --}}
                        <div class="relative w-full max-w-lg">
                            <template x-if="images.length > 0">
                                <div class="aspect-square rounded-2xl overflow-hidden bg-white dark:bg-gray-900 shadow-lg">
                                    <img :src="images[currentImage]" :alt="product?.name"
                                        class="w-full h-full object-contain transition-transform duration-500 hover:scale-105">
                                </div>
                            </template>
                            <template x-if="images.length === 0">
                                <div
                                    class="aspect-square rounded-2xl flex items-center justify-center bg-gradient-to-br from-gray-100 to-gray-200 dark:from-gray-800 dark:to-gray-900 text-gray-400">
                                    <svg class="w-32 h-32" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1"
                                            d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                    </svg>
                                </div>
                            </template>

                            {{-- Image Navigation Dots --}}
                            <template x-if="images.length > 1">
                                <div class="absolute -bottom-4 left-1/2 -translate-x-1/2 flex items-center gap-2">
                                    <template x-for="(img, idx) in images" :key="idx">
                                        <button @click="currentImage = idx"
                                            class="h-2 rounded-full transition-all duration-300"
                                            :class="currentImage === idx ? 'bg-nassaj-gold w-8' : 'bg-gray-300 dark:bg-white/30 w-2 hover:bg-nassaj-gold/50'">
                                        </button>
                                    </template>
                                </div>
                            </template>
                        </div>
                    </div>

                    {{-- Left Side - Product Info --}}
                    <div class="p-8 lg:p-10 flex flex-col bg-white dark:bg-[#1a1a1a]">
                        {{-- Title & Rating --}}
                        <div class="mb-6">
                            <h2 class="cairo text-2xl lg:text-3xl font-black text-gray-900 dark:text-white mb-3 leading-tight"
                                x-text="product?.name"></h2>

                            <div class="flex items-center gap-3 mb-4">
                                <div class="flex gap-1">
                                    <template x-for="i in 5">
                                        <svg class="w-4 h-4 text-yellow-400 fill-current" viewBox="0 0 20 20">
                                            <path
                                                d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                        </svg>
                                    </template>
                                </div>
                                <span class="text-gray-600 dark:text-gray-400 text-sm">(4.5)</span>
                            </div>
                        </div>

                        {{-- Description --}}
                        <p class="text-gray-600 dark:text-gray-400 text-sm leading-relaxed mb-6"
                            x-text="product?.description || 'منتج عالي الجودة من تصميم نساج'"></p>

                        {{-- Price --}}
                        <div class="mb-6">
                            <div class="flex items-center gap-3">
                                <span class="text-nassaj-gold cairo text-3xl font-black" x-text="formatPrice(price)"></span>
                                <template x-if="oldPrice > 0 && oldPrice > price">
                                    <span class="text-gray-400 text-xl line-through" x-text="formatPrice(oldPrice)"></span>
                                </template>
                            </div>
                        </div>

                        {{-- Options Section --}}
                        <div class="space-y-5 mb-6">
                            {{-- Colors --}}
                            <template x-if="colors.length > 0">
                                <div>
                                    <h4 class="text-gray-900 dark:text-white text-sm font-bold mb-3">
                                        <span x-text="attributeNames.color || 'اللون'"></span>
                                    </h4>
                                    <div class="flex flex-wrap gap-2">
                                        <template x-for="color in colors" :key="color.name">
                                            <button @click="selectColor(color)"
                                                class="px-6 py-2.5 rounded-full text-sm font-bold transition-all duration-300 border-2"
                                                :class="selectedColor?.name === color.name
                                                    ? 'bg-nassaj-gold text-black border-nassaj-gold'
                                                    : 'bg-transparent text-gray-700 dark:text-gray-300 border-gray-300 dark:border-gray-700 hover:border-nassaj-gold/50'"
                                                x-text="color.name"></button>
                                        </template>
                                    </div>
                                </div>
                            </template>

                            {{-- Sizes --}}
                            <template x-if="sizes.length > 0">
                                <div>
                                    <h4 class="text-gray-900 dark:text-white text-sm font-bold mb-3">
                                        <span x-text="attributeNames.size || 'المقاس'"></span>
                                    </h4>
                                    <div class="flex flex-wrap gap-2">
                                        <template x-for="size in sizes" :key="size.name">
                                            <button @click="selectSize(size)"
                                                class="px-6 py-2.5 rounded-full text-sm font-bold transition-all duration-300 border-2"
                                                :class="selectedSize?.name === size.name
                                                    ? 'bg-nassaj-gold text-black border-nassaj-gold'
                                                    : 'bg-transparent text-gray-700 dark:text-gray-300 border-gray-300 dark:border-gray-700 hover:border-nassaj-gold/50'"
                                                x-text="size.name"></button>
                                        </template>
                                    </div>
                                </div>
                            </template>

                            {{-- Quantity --}}
                            <div>
                                <h4 class="text-gray-900 dark:text-white text-sm font-bold mb-3">الكمية</h4>
                                <div
                                    class="inline-flex items-center gap-3 p-3 bg-gray-100 dark:bg-white/5 rounded-2xl border border-gray-200 dark:border-white/10">
                                    <button @click="qty = Math.max(qty - 1, 1)"
                                        class="w-10 h-10 flex items-center justify-center hover:bg-nassaj-gold/20 rounded-lg transition-colors text-gray-900 dark:text-white">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" stroke-width="3"
                                            viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" d="M18 12H6" />
                                        </svg>
                                    </button>
                                    <span class="w-12 text-center text-lg font-black text-gray-900 dark:text-white"
                                        x-text="qty"></span>
                                    <button @click="qty = Math.min(qty + 1, 99)"
                                        class="w-10 h-10 flex items-center justify-center hover:bg-nassaj-gold/20 rounded-lg transition-colors text-gray-900 dark:text-white">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" stroke-width="3"
                                            viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" d="M12 6v12m6-6H6" />
                                        </svg>
                                    </button>
                                </div>
                            </div>
                        </div>

                        {{-- Spacer --}}
                        <div class="flex-1"></div>

                        {{-- Actions --}}
                        <div class="space-y-3">
                            {{-- Add to Cart --}}
                            <button @click="addToCart()"
                                class="w-full bg-gradient-to-r from-nassaj-gold via-nassaj-gold-light to-nassaj-gold text-black py-4 rounded-2xl font-black text-base flex items-center justify-center gap-3 hover:shadow-2xl hover:shadow-nassaj-gold/30 transition-all hover:scale-[1.02] group">
                                <svg class="w-5 h-5 transition-transform group-hover:scale-110" fill="none"
                                    stroke="currentColor" stroke-width="2.5" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round"
                                        d="M2.25 3h1.386c.51 0 .955.343 1.087.835l.383 1.437M7.5 14.25a3 3 0 00-3 3h15.75m-12.75-3h11.218c1.121-2.3 2.1-4.684 2.924-7.138a60.114 60.114 0 00-16.536-1.84M7.5 14.25L5.106 5.272M6 20.25a.75.75 0 11-1.5 0 .75.75 0 011.5 0zm12.75 0a.75.75 0 11-1.5 0 .75.75 0 011.5 0z" />
                                </svg>
                                <span>أضف للسلة</span>
                            </button>

                            {{-- Secondary Actions Row --}}
                            <div class="grid grid-cols-4 gap-2">
                                {{-- Wishlist --}}
                                <button @click="addToWishlist()"
                                    class="flex items-center justify-center p-3 bg-gray-100 dark:bg-white/5 hover:bg-nassaj-gold/10 dark:hover:bg-nassaj-gold/20 border border-gray-200 dark:border-white/10 rounded-xl transition-all"
                                    title="المفضلة">
                                    <svg class="w-5 h-5 text-gray-700 dark:text-gray-300" fill="none"
                                        stroke="currentColor" viewBox="0 0 24 24" stroke-width="2">
                                        <path stroke-linecap="round" stroke-linejoin="round"
                                            d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                                    </svg>
                                </button>

                                {{-- Compare --}}
                                <button @click="addToCompare()"
                                    class="flex items-center justify-center p-3 bg-gray-100 dark:bg-white/5 hover:bg-nassaj-gold/10 dark:hover:bg-nassaj-gold/20 border border-gray-200 dark:border-white/10 rounded-xl transition-all"
                                    title="مقارنة">
                                    <svg class="w-5 h-5 text-gray-700 dark:text-gray-300" fill="none"
                                        stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round"
                                            d="M3 13.125C3 12.504 3.504 12 4.125 12h2.25c.621 0 1.125.504 1.125 1.125v6.75C7.5 20.496 6.996 21 6.375 21h-2.25A1.125 1.125 0 013 19.875v-6.75zM9.75 8.625c0-.621.504-1.125 1.125-1.125h2.25c.621 0 1.125.504 1.125 1.125v11.25c0 .621-.504 1.125-1.125 1.125h-2.25a1.125 1.125 0 01-1.125-1.125V8.625zM16.5 4.125c0-.621.504-1.125 1.125-1.125h2.25C20.496 3 21 3.504 21 4.125v15.75c0 .621-.504 1.125-1.125 1.125h-2.25a1.125 1.125 0 01-1.125-1.125V4.125z" />
                                    </svg>
                                </button>

                                {{-- Share --}}
                                <button @click="share()"
                                    class="flex items-center justify-center p-3 bg-gray-100 dark:bg-white/5 hover:bg-nassaj-gold/10 dark:hover:bg-nassaj-gold/20 border border-gray-200 dark:border-white/10 rounded-xl transition-all"
                                    title="مشاركة">
                                    <svg class="w-5 h-5 text-gray-700 dark:text-gray-300" fill="none"
                                        stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round"
                                            d="M7.217 10.907a2.25 2.25 0 100 2.186m0-2.186c.18.324.283.696.283 1.093s-.103.77-.283 1.093m0-2.186l9.566-5.314m-9.566 7.5l9.566 5.314m0 0a2.25 2.25 0 103.935 2.186 2.25 2.25 0 00-3.935-2.186zm0-12.814a2.25 2.25 0 103.933-2.185 2.25 2.25 0 00-3.933 2.185z" />
                                    </svg>
                                </button>

                                {{-- Customize --}}
                                <button @click="customize()"
                                    class="flex items-center justify-center p-3 bg-nassaj-primary hover:bg-nassaj-primary/90 border border-nassaj-primary text-white rounded-xl transition-all"
                                    title="تخصيص">
                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" stroke-width="2"
                                        viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round"
                                            d="M9.594 3.94c.09-.542.56-.94 1.11-.94h2.593c.55 0 1.02.398 1.11.94l.213 1.281c.063.374.313.686.645.87.074.04.147.083.22.127.324.196.72.257 1.075.124l1.217-.456a1.125 1.125 0 011.37.49l1.296 2.247a1.125 1.125 0 01-.26 1.431l-1.003.827c-.293.24-.438.613-.431.992a6.759 6.759 0 010 .255c-.007.378.138.75.43.99l1.005.828c.424.35.534.954.26 1.43l-1.298 2.247a1.125 1.125 0 01-1.369.491l-1.217-.456c-.355-.133-.75-.072-1.076.124a6.57 6.57 0 01-.22.128c-.331.183-.581.495-.644.869l-.213 1.28c-.09.543-.56.941-1.11.941h-2.594c-.55 0-1.02-.398-1.11-.94l-.213-1.281c-.062-.374-.312-.686-.644-.87a6.52 6.52 0 01-.22-.127c-.325-.196-.72-.257-1.076-.124l-1.217.456a1.125 1.125 0 01-1.369-.49l-1.297-2.247a1.125 1.125 0 01.26-1.431l1.004-.827c.292-.24.437-.613.43-.992a6.932 6.932 0 010-.255c.007-.378-.138-.75-.43-.99l-1.004-.828a1.125 1.125 0 01-.26-1.43l1.297-2.247a1.125 1.125 0 011.37-.491l1.216.456c.356.133.751.072 1.076-.124.072-.044.146-.087.22-.128.332-.183.582-.495.644-.869l.214-1.281z" />
                                        <path stroke-linecap="round" stroke-linejoin="round"
                                            d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                    </svg>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </template>
</div>
