/* Product Details Page
   - Uses global products[] from app.js
   - Supports gallery thumbnails, variants/options pricing, qty pricing
   - Fixes: breadcrumbs duplication, action buttons binding, wishlist/compare active states
*/

// Basic HTML escaping (prevents broken markup / XSS in demo data)
function escapeHtml(input) {
  return String(input ?? '')
    .replaceAll('&', '&amp;')
    .replaceAll('<', '&lt;')
    .replaceAll('>', '&gt;')
    .replaceAll('"', '&quot;')
    .replaceAll("'", '&#039;');
}



(function () {
  const qs = (s) => document.querySelector(s);
  const qsa = (s) => Array.from(document.querySelectorAll(s));
  const byId = (id) => document.getElementById(id);

  if (typeof products === 'undefined' || !Array.isArray(products) || !products.length) return;

  const url = new URL(window.location.href);
  const id = Number(url.searchParams.get('id') || 0);
  const p = products.find((x) => Number(x.id) === id) || products[0];
  if (!p) return;

  // Breadcrumbs (support both ids)
  const breadName = byId('breadcrumbProductName') || byId('breadcrumbProduct');
  if (breadName) breadName.textContent = p.name;

  // Basic info
  const titleEl = byId('productTitle');
  const priceEl = byId('productPrice');
  const oldEl = byId('productOld');
  const totalEl = byId('productTotal'); // optional
  const descEl = byId('productDesc');
  const badgeEl = byId('productBadge');
  const metaEl = byId('productMeta');

  const brandEl = byId('productBrand');
  const originEl = byId('productOrigin');
  if (brandEl) brandEl.textContent = p.brand || '—';
  if (originEl) originEl.textContent = p.originCountry || '—';

  if (titleEl) titleEl.textContent = p.name;
  if (descEl) descEl.textContent = p.description || 'وصف المنتج غير متوفر حالياً.';
  if (badgeEl) {
    badgeEl.textContent = p.badge || '';
    badgeEl.classList.toggle('hidden', !p.badge);
  }
  if (metaEl) {
    const r = Number(p.rating || 0);
    metaEl.textContent = `${p.subcategory || '—'} • ${p.brand || '—'} • ⭐ ${r.toFixed(1)}`;
  }

  // Images (gallery + thumbnails)
  const mainImg = byId('mainImage');
  const thumbs = byId('thumbs');

  const images = Array.isArray(p.images) && p.images.length
    ? p.images
    : (p.image ? [p.image, p.image + '&sat=-10', p.image + '&sat=-20', p.image + '&sat=-30'] : []);

  const setActive = (src) => {
    if (mainImg) {
      mainImg.style.opacity = '0.6';
      setTimeout(() => {
        mainImg.src = src;
        mainImg.style.opacity = '1';
      }, 120);
    }
    thumbs?.querySelectorAll('button[data-src]').forEach((b) => {
      const active = b.getAttribute('data-src') === src;
      b.classList.toggle('ring-2', active);
      b.classList.toggle('ring-[#d4a356]', active);
    });
  };

  if (mainImg && images[0]) mainImg.src = images[0];
  if (thumbs && images.length) {
    thumbs.innerHTML = images.map((src, idx) => `
      <button type="button" data-src="${src}" class="w-20 h-20 rounded-2xl overflow-hidden bg-white/5 border border-white/10 ${idx === 0 ? 'ring-2 ring-[#d4a356]' : ''}">
        <img src="${src}" alt="صورة" class="w-full h-full object-cover" />
      </button>
    `).join('');
    thumbs.querySelectorAll('button[data-src]').forEach((b) => {
      b.addEventListener('click', () => setActive(b.getAttribute('data-src')));
    });
  }

  // Options (variants)
  const optionsRoot = byId('productOptions');
  const variantHintEl = byId('variantHint');
  const variantSummaryEl = byId('variantSummary');
  const selected = {};
  const groups = (typeof getProductOptionGroups === 'function') ? getProductOptionGroups(p) : [];

  // Default selections
  for (const g of groups) {
    if (g.key === 'color') selected[g.key] = g.items?.[0]?.code ?? null;
    else selected[g.key] = g.items?.[0] ?? null;
  }

  const renderOptions = () => {
    if (!optionsRoot || !groups.length) {
      // hide the whole variants section if empty
      optionsRoot?.classList.add('hidden');
      variantHintEl?.classList.add('hidden');
      variantSummaryEl?.classList.add('hidden');
      const wrap = optionsRoot?.closest('div.mt-6');
      wrap?.classList.add('hidden');
      return;
    }

    optionsRoot.classList.remove('hidden');
    const wrap = optionsRoot.closest('div.mt-6');
    wrap?.classList.remove('hidden');

    optionsRoot.innerHTML = groups.map((g) => {
      // Title (matches the requested UI)
      const title = `<div class="variant-title">${escapeHtml(g.label)}</div>`;

      if (g.type === 'color') {
        const items = (g.items || []).map((c, i) => {
          const code = String(c.code);
          const isActive = String(selected[g.key]) === code || (!selected[g.key] && i === 0);
          const bg = String(c.value || '#fff');

          return `
            <button type="button"
              class="v-swatch ${isActive ? 'is-active' : ''}"
              data-opt-key="${g.key}" data-opt-val="${code}"
              title="${escapeHtml(c.name)}" aria-label="${escapeHtml(c.name)}">
              <span class="v-swatch__fill" style="background:${bg}"></span>
            </button>
          `;
        }).join('');

        return `
          <div class="variant-group">
            ${title}
            <div class="variant-values variant-values--swatches">${items}</div>
          </div>
        `;
      }

      // Default chips (size / power / ...)
      const chips = (g.items || []).map((s, i) => {
        const val = String(s);
        const isActive = String(selected[g.key]) === val || (!selected[g.key] && i === 0);
        return `
          <button type="button"
            class="v-chip ${isActive ? 'is-active' : ''}"
            data-opt-key="${g.key}" data-opt-val="${escapeHtml(val)}">${escapeHtml(val)}</button>
        `;
      }).join('');

      return `
        <div class="variant-group">
          ${title}
          <div class="variant-values variant-values--chips">${chips}</div>
        </div>
      `;
    }).join('');

    optionsRoot.querySelectorAll('[data-opt-key]').forEach((btn) => {
      btn.addEventListener('click', () => {
        const key = btn.getAttribute('data-opt-key');
        const val = btn.getAttribute('data-opt-val');

        selected[key] = val;

        // active state (per group)
        optionsRoot.querySelectorAll(`[data-opt-key="${CSS.escape(key)}"]`).forEach((x) => {
          x.classList.remove('is-active');
        });
        btn.classList.add('is-active');

        updatePrices();
      });
    });
  };

  const renderVariantSummary = () => {
    if (!variantSummaryEl || !groups.length) return;
    const parts = [];
    for (const g of groups) {
      const val = selected[g.key];
      if (val == null) continue;
      if (g.type === 'color') {
        const found = (g.items || []).find(x => String(x.code) === String(val));
        parts.push(`${g.label}: ${found?.name ?? val}`);
      } else {
        parts.push(`${g.label}: ${val}`);
      }
    }
    if (!parts.length) {
      variantSummaryEl.classList.add('hidden');
      return;
    }
    variantSummaryEl.textContent = parts.join(' • ');
    variantSummaryEl.classList.remove('hidden');
  };

  // Qty controls (يدعم إدخال يدوي)
  const qtyEl = byId('qty'); // input[type=number]
  const minus = byId('qtyMinus');
  const plus = byId('qtyPlus');
  let qty = 1;

  const setQty = (q) => {
    qty = Math.max(1, Math.min(99, parseInt(q || 1, 10) || 1));
    if (qtyEl) {
      // input field
      qtyEl.value = String(qty);
    }
    updatePrices();
  };

  minus?.addEventListener('click', () => setQty(qty - 1));
  plus?.addEventListener('click', () => setQty(qty + 1));

  // إدخال يدوي: تحديث فوري + تصحيح عند الخروج
  qtyEl?.addEventListener('input', () => {
    // لا نكسر الكتابة، لكن نحدث السعر إذا كانت قيمة رقمية
    const v = parseInt(qtyEl.value || '1', 10);
    if (Number.isFinite(v)) {
      qty = Math.max(1, Math.min(99, v));
      updatePrices();
    }
  });
  qtyEl?.addEventListener('blur', () => setQty(qtyEl.value));

  // Prices (unit + total)
  const updatePrices = () => {
    const unit = (typeof computeUnitPrice === 'function') ? computeUnitPrice(p, selected) : normalizeIQD(p.price);
    const total = (typeof computeTotalPrice === 'function') ? computeTotalPrice(p, selected, qty) : unit * qty;

    if (priceEl) priceEl.textContent = formatIQD(unit);
    if (totalEl) totalEl.textContent = formatIQD(total);

    if (oldEl) {
      oldEl.textContent = p.originalPrice ? formatIQD(p.originalPrice) : '';
      oldEl.classList.toggle('hidden', !p.originalPrice);
    }

    // Free shipping progress (on this product page)
    const fill = byId('productProgressFill');
    const text = byId('productProgressText');
    if (fill && text && typeof FREE_SHIPPING_THRESHOLD !== 'undefined') {
      const t = Number(FREE_SHIPPING_THRESHOLD || 100000);
      const pct = Math.max(0, Math.min(100, (total / t) * 100));
      fill.style.width = pct + '%';
      if (total >= t) {
        text.innerHTML = `<span class="text-green-400 font-bold">مبروك! طلبك مؤهل للتوصيل المجاني.</span>`;
      } else {
        text.innerHTML = `أضف <span class="font-black" style="color:var(--primary-gold)">${formatIQD(t - total)}</span> للحصول على التوصيل المجاني.`;
      }
    }

    // variants summary line
    renderVariantSummary();
  };

  const syncFavCompareUI = () => {
    const inWish = (typeof wishlist !== 'undefined') ? wishlist.some(x => Number(x.productId) === Number(p.id)) : false;
    const inCmp = (typeof compare !== 'undefined') ? compare.some(x => Number(x.productId) === Number(p.id)) : false;

    const wishBtns = [byId('productWishlist'), ...qsa('[data-action="wishlist"]')].filter(Boolean);
    const cmpBtns  = [byId('productCompare'), ...qsa('[data-action="compare"]')].filter(Boolean);

    wishBtns.forEach(btn => btn.classList.toggle('ring-2', inWish));
    cmpBtns.forEach(btn => btn.classList.toggle('ring-2', inCmp));
  };

  renderOptions();
  updatePrices();
  syncFavCompareUI();

  // Buttons (side panel)
  const addBtn = byId('productAddToCart');
  const wishBtn = byId('productWishlist');
  const cmpBtn = byId('productCompare');

  addBtn?.addEventListener('click', () => {
    const unitPrice = (typeof computeUnitPrice === 'function') ? computeUnitPrice(p, selected) : normalizeIQD(p.price);
    if (typeof addToCart === 'function') addToCart(p.id, { unitPrice, qty, options: selected });
    if (typeof openCart === 'function') openCart();
  });

  wishBtn?.addEventListener('click', () => {
    if (typeof toggleWishlist === 'function') toggleWishlist(p.id);
    syncFavCompareUI();
  });

  cmpBtn?.addEventListener('click', () => {
    if (typeof toggleCompare === 'function') toggleCompare(p.id);
    syncFavCompareUI();
  });

  // Quick info buttons under actions
  byId('productShare')?.addEventListener('click', async () => {
    try {
      const url = window.location.href;
      if (navigator.clipboard?.writeText) await navigator.clipboard.writeText(url);
      if (typeof toast === 'function') toast('تم نسخ رابط المنتج');
    } catch {
      if (typeof toast === 'function') toast('تعذر نسخ الرابط');
    }
  });

  byId('productQuestion')?.addEventListener('click', () => {
    if (typeof toast === 'function') toast('راسلنا على واتساب: 0770 566 6777');
  });

  byId('productDelivery')?.addEventListener('click', () => {
    if (typeof toast === 'function') toast('التوصيل داخل البصرة وخارجها. الإرجاع خلال 7 أيام وفق الحالة.');
  });

  // Buttons (overlay on image) via data-action
  qsa('[data-action="quickview"]').forEach((btn) => {
    btn.addEventListener('click', () => {
      if (typeof openQuickView === 'function') openQuickView(p.id);
    });
  });
  qsa('[data-action="wishlist"]').forEach((btn) => {
    btn.addEventListener('click', () => {
      if (typeof toggleWishlist === 'function') toggleWishlist(p.id);
      syncFavCompareUI();
    });
  });
  qsa('[data-action="compare"]').forEach((btn) => {
    btn.addEventListener('click', () => {
      if (typeof toggleCompare === 'function') toggleCompare(p.id);
      syncFavCompareUI();
    });
  });

  // ===================== Carousels (بدون تحريك تلقائي) =====================
  const initHorizontalNav = (trackEl, prevBtn, nextBtn) => {
    if (!trackEl) return;
    // smooth + snap
    trackEl.style.scrollBehavior = 'smooth';

    const step = () => {
      const firstCard = trackEl.querySelector(':scope > *');
      const w = firstCard ? (firstCard.getBoundingClientRect().width + 16) : Math.max(280, trackEl.clientWidth * 0.8);
      return Math.round(w);
    };

    prevBtn?.addEventListener('click', () => {
      trackEl.scrollBy({ left: -step(), behavior: 'smooth' });
    });
    nextBtn?.addEventListener('click', () => {
      trackEl.scrollBy({ left: step(), behavior: 'smooth' });
    });
  };

  // Related products
  const relatedWrap = byId('relatedProductsWrap');
  if (relatedWrap && typeof renderProductCard === 'function') {
    const sameCategory = products.filter(x => x?.category && p?.category && x.category === p.category && x.id !== p.id);
    const sameBrand = products.filter(x => x?.brand && p?.brand && x.brand === p.brand && x.id !== p.id);
    const fallback = products.filter(x => x.id !== p.id);

    const related = (sameCategory.length ? sameCategory : (sameBrand.length ? sameBrand : fallback)).slice(0, 12);

    relatedWrap.innerHTML = related.length
      ? related.map(x => renderProductCard(x)).join('')
      : `<div class="text-gray-400 text-sm py-6">لا توجد منتجات مشابهة حالياً.</div>`;

    initHorizontalNav(relatedWrap, byId('relatedPrev'), byId('relatedNext'));
  }

  // Reviews
  initHorizontalNav(byId('reviewsTrack'), byId('reviewsPrev'), byId('reviewsNext'));

  // Keep UI synced when drawers update counts
  window.addEventListener('storage', () => syncFavCompareUI());
})();
