/* Categories Page (Filters + Sort + Grid/List)
   Depends on: products[], renderGridInto(), formatIQD(), normalizeIQD() from app.js
   Notes:
   - لا يوجد بحث داخل صفحة الفئات (حسب طلبك)
   - لا يوجد Pagination أو عدّاد نتائج
*/

(function () {
  const el = (id) => document.getElementById(id);
// Electric Spark (GSAP) - subtle, fast, and theme-aligned
const reduceMotion = window.matchMedia && window.matchMedia('(prefers-reduced-motion: reduce)').matches;

function prepSpark(btn) {
  const path = btn?.querySelector?.('.spark-path');
  if (!path || path.__sparkReady) return;
  try {
    const len = path.getTotalLength();
    path.style.strokeDasharray = String(len);
    path.style.strokeDashoffset = String(len);
    path.__sparkLen = len;
    path.__sparkReady = true;
  } catch (e) {
    // ignore if browser doesn't support getTotalLength for some reason
  }
}

function electricSpark(btn) {
  if (reduceMotion) return;
  if (!window.gsap) return;

  prepSpark(btn);

  const spark = btn.querySelector('.spark');
  const path = btn.querySelector('.spark-path');
  if (!spark || !path) return;

  const len = path.__sparkLen || 120;

  // kill previous animations on same elements
  gsap.killTweensOf([btn, spark, path]);

  // quick "electric" pulse + spark draw
  const tl = gsap.timeline({ defaults: { ease: 'power2.out' } });

  tl.set(spark, { opacity: 0, rotate: 0, scale: 1 })
    .set(path, { strokeDasharray: len, strokeDashoffset: len, opacity: 1 })
    .to(btn, { scale: 1.06, duration: 0.12, yoyo: true, repeat: 1 }, 0)
    .to(btn, { filter: 'drop-shadow(0 0 14px rgba(212, 163, 86, .65))', duration: 0.14, yoyo: true, repeat: 2, ease: 'power1.inOut' }, 0)
    .to(spark, { opacity: 1, duration: 0.05 }, 0.02)
    .to(spark, { rotate: gsap.utils.random(-12, 12), duration: 0.10 }, 0.02)
    .to(path, { strokeDashoffset: 0, duration: 0.22, ease: 'power1.out' }, 0.04)
    .to(spark, { opacity: 0, duration: 0.12 }, 0.22)
    .set(btn, { filter: 'none' }, 0.36);
}

function bindSpark(btn) {
  if (!btn) return;
  prepSpark(btn);
  btn.addEventListener('mouseenter', () => electricSpark(btn), { passive: true });
  btn.addEventListener('click', () => electricSpark(btn));
}


// Hero Lightning (GSAP) - محاكاة صاعقة حقيقية على أيقونة الهيرو

function initHeroLightning() {
  if (!window.gsap) return;

  const hero = document.getElementById('categoriesHero');
  const box = document.getElementById('heroLightning');
  const heroSectionFx = document.getElementById('heroSectionFx');
  const breadcrumbsWrap = document.getElementById('breadcrumbsWrap');
  const breadcrumbsFx = document.getElementById('breadcrumbsFx');
  const boltSvg = document.getElementById('heroBoltFxSvg');
  const boltPath = document.getElementById('heroBoltFxPath');

  if (!hero || !box) return;

  // Respect reduce motion
  if (reduceMotion) {
    // ensure fx is hidden
    if (boltPath) boltPath.style.opacity = '0';
    if (breadcrumbsFx) breadcrumbsFx.style.opacity = '0';
    if (heroSectionFx) heroSectionFx.style.opacity = '0';
    return;
  }

  const arcs = Array.from(box.querySelectorAll('.hero-arcs .arc'));
  const sparks = Array.from(box.querySelectorAll('.hero-sparks .spark-line'));

  // Prepare stroke-dash for arcs/sparks
  const prepStroke = (p) => {
    try {
      const len = p.getTotalLength();
      p.style.strokeDasharray = String(len);
      p.style.strokeDashoffset = String(len);
      p.dataset.len = String(len);
      p.style.opacity = '0';
    } catch (_) {}
  };

  arcs.forEach(prepStroke);
  sparks.forEach(prepStroke);

  const pick = (arr, n) => {
    const copy = arr.slice();
    const out = [];
    while (copy.length && out.length < n) {
      out.push(copy.splice(Math.floor(Math.random() * copy.length), 1)[0]);
    }
    return out;
  };

  // --- Lightning path builder (DOM -> SVG coords) ---
  const viewW = 1000;
  const viewH = 360;

  const toSvgPoint = (clientX, clientY) => {
    if (!boltSvg) return { x: 0, y: 0 };
    const r = boltSvg.getBoundingClientRect();
    const x = ((clientX - r.left) / Math.max(1, r.width)) * viewW;
    const y = ((clientY - r.top) / Math.max(1, r.height)) * viewH;
    return { x, y };
  };

  const jitter = (v, amt) => v + (Math.random() * 2 - 1) * amt;

  function buildBoltD(from, to) {
    // segmented polyline with slight chaos (looks like real electricity)
    const segs = 8;
    const pts = [];
    for (let i = 0; i <= segs; i++) {
      const t = i / segs;
      const x = from.x + (to.x - from.x) * t;
      const y = from.y + (to.y - from.y) * t;

      const j = (i === 0 || i === segs) ? 0 : 14 + Math.random() * 10;
      pts.push({ x: jitter(x, j), y: jitter(y, j) });
    }

    // Smooth-ish path with Q curves between points
    let d = `M ${pts[0].x.toFixed(1)} ${pts[0].y.toFixed(1)}`;
    for (let i = 1; i < pts.length; i++) {
      const p0 = pts[i - 1];
      const p1 = pts[i];
      const cx = (p0.x + p1.x) / 2 + (Math.random() * 2 - 1) * 8;
      const cy = (p0.y + p1.y) / 2 + (Math.random() * 2 - 1) * 8;
      d += ` Q ${cx.toFixed(1)} ${cy.toFixed(1)} ${p1.x.toFixed(1)} ${p1.y.toFixed(1)}`;
    }
    return d;
  }

  function zapToBreadcrumbs() {
    if (!boltSvg || !boltPath || !breadcrumbsWrap) return;

    // تأثير على خلفية الـ Hero بالكامل (categoriesHero)
    heroSectionPulse();

    // Start at bolt icon center
    const boltIcon = box.querySelector('svg');
    const rFrom = (boltIcon || box).getBoundingClientRect();
    const rTo = breadcrumbsWrap.getBoundingClientRect();

    const from = toSvgPoint(rFrom.left + rFrom.width * 0.50, rFrom.top + rFrom.height * 0.42);
    const to = toSvgPoint(rTo.left + rTo.width * 0.20, rTo.top + rTo.height * 0.55); // lands on left of breadcrumbs pill

    const d = buildBoltD(from, to);
    boltPath.setAttribute('d', d);

    // Stroke dash animation
    let len = 420;
    try { len = boltPath.getTotalLength(); } catch (_) {}
    boltPath.style.strokeDasharray = String(len);
    boltPath.style.strokeDashoffset = String(len);

    gsap.killTweensOf(boltPath);
    gsap.set(boltPath, { opacity: 0 });

    const tl = gsap.timeline({ defaults: { ease: 'power2.out' } });
    tl.to(boltPath, { opacity: 1, duration: 0.05 }, 0)
      .to(boltPath, { strokeDashoffset: 0, duration: 0.22, ease: 'power1.out' }, 0.03)
      .to(boltPath, { opacity: 0, duration: 0.18, ease: 'power1.inOut' }, 0.20);

    // Breadcrumbs background electricity pulse
    if (breadcrumbsFx) {
      gsap.killTweensOf([breadcrumbsFx, breadcrumbsWrap]);
      gsap.set(breadcrumbsFx, { opacity: 0, scale: 0.92 });

      gsap.timeline({ defaults: { ease: 'power2.out' } })
        .to(breadcrumbsFx, { opacity: 1, scale: 1.02, duration: 0.10 }, 0.06)
        .to(breadcrumbsFx, { opacity: 0, scale: 0.96, duration: 0.18 }, 0.18);

      gsap.fromTo(
        breadcrumbsWrap,
        { boxShadow: '0 0 0px rgba(212,163,86,0)' },
        { boxShadow: '0 0 28px rgba(212,163,86,.22)', duration: 0.12, yoyo: true, repeat: 1, ease: 'power1.inOut' }
      );
    }

    // Full section background electricity sweep (categoriesHero)
    if (heroSectionFx) {
      gsap.killTweensOf(heroSectionFx);
      gsap.set(heroSectionFx, { opacity: 0, scale: 0.98, backgroundPosition: '50% 50%, 50% 50%, 50% 50%' });

      gsap.timeline({ defaults: { ease: 'power2.out' } })
        .to(heroSectionFx, { opacity: 1, scale: 1.02, duration: 0.10 }, 0.06)
        .to(heroSectionFx, { backgroundPosition: '42% 44%, 58% 54%, 46% 52%', duration: 0.16, ease: 'power1.inOut' }, 0.06)
        .to(heroSectionFx, { opacity: 0, scale: 0.99, duration: 0.22, ease: 'power1.inOut' }, 0.22);
    }
  }

  function heroSectionPulse() {
    // Electricity/light sweep over the entire hero section background
    if (!heroSectionFx) return;

    gsap.killTweensOf(heroSectionFx);
    gsap.set(heroSectionFx, {
      opacity: 0,
      scale: 0.985,
      backgroundPosition: '50% 50%, 50% 50%, 50% 50%'
    });

    const dx1 = gsap.utils.random(-12, 12);
    const dy1 = gsap.utils.random(-10, 10);
    const dx2 = gsap.utils.random(-14, 14);
    const dy2 = gsap.utils.random(-12, 12);
    const dx3 = gsap.utils.random(-10, 10);
    const dy3 = gsap.utils.random(-10, 10);

    gsap.timeline({ defaults: { ease: 'power2.out' } })
      .to(heroSectionFx, { opacity: 0.85, duration: 0.10, scale: 1.02 }, 0)
      .to(
        heroSectionFx,
        {
          backgroundPosition: `${50 + dx1}% ${50 + dy1}%, ${50 + dx2}% ${50 + dy2}%, ${50 + dx3}% ${50 + dy3}%`,
          duration: 0.22,
          ease: 'sine.inOut'
        },
        0.02
      )
      .to(heroSectionFx, { opacity: 0, duration: 0.22, scale: 0.99, ease: 'power1.inOut' }, 0.22);
  }

  function framePulse() {
    // electricity travels "from icon to frame"
    gsap.killTweensOf(box);
    gsap.timeline({ defaults: { ease: 'power2.out' } })
      .to(box, { scale: 1.04, duration: 0.12, yoyo: true, repeat: 1 }, 0)
      .to(box, { '--frameGlow': 1, duration: 0.08 }, 0.02)
      .to(box, { '--frameGlow': 0, duration: 0.22, ease: 'power1.inOut' }, 0.18)
      .fromTo(
        box,
        { boxShadow: 'inset 0 0 0 1px rgba(255,255,255,.10), 0 0 0 rgba(212,163,86,0)' },
        { boxShadow: 'inset 0 0 0 1px rgba(212,163,86,.35), 0 0 28px rgba(212,163,86,.18)', duration: 0.14, yoyo: true, repeat: 1, ease: 'power1.inOut' },
        0.04
      );
  }

  function arcsBurst() {
    if (!arcs.length) return;

    const chosenArcs = pick(arcs, gsap.utils.random(4, 7, 1));
    const chosenSparks = pick(sparks, gsap.utils.random(3, 6, 1));

    chosenArcs.forEach((p, i) => {
      const len = Number(p.dataset.len || 120);
      gsap.set(p, { strokeDasharray: len, strokeDashoffset: len, opacity: 0 });
      gsap.to(p, { opacity: 1, duration: 0.05, delay: i * 0.02 });
      gsap.to(p, { strokeDashoffset: 0, duration: 0.22, delay: i * 0.02, ease: 'power1.out' });
      gsap.to(p, { opacity: 0, duration: 0.18, delay: 0.18 + i * 0.03, ease: 'power1.inOut' });
    });

    chosenSparks.forEach((p, i) => {
      const len = Number(p.dataset.len || 40);
      gsap.set(p, { strokeDasharray: len, strokeDashoffset: len, opacity: 0 });
      gsap.to(p, { opacity: 1, duration: 0.04, delay: 0.03 + i * 0.03 });
      gsap.to(p, { strokeDashoffset: 0, duration: 0.14, delay: 0.03 + i * 0.03, ease: 'power1.out' });
      gsap.to(p, { opacity: 0, duration: 0.12, delay: 0.14 + i * 0.03, ease: 'power1.inOut' });
    });
  }

  // Idle loop (subtle) + occasional zap to breadcrumbs
  const idle = gsap.timeline({ repeat: -1, repeatDelay: 0.8 });
  idle.add(() => {
    arcsBurst();
    framePulse();
    // occasional travel to breadcrumbs
    if (breadcrumbsWrap && Math.random() > 0.35) zapToBreadcrumbs();
  })
  .to(box, { scale: 1.015, duration: 0.36, yoyo: true, repeat: 1, ease: 'sine.inOut' }, 0);

  // Stronger burst on interaction
  const burst = () => {
    heroSectionPulse();
    arcsBurst();
    framePulse();
    zapToBreadcrumbs();
  };

  box.addEventListener('mouseenter', burst, { passive: true });
  box.addEventListener('click', burst);

  // Recompute path on resize
  window.addEventListener('resize', () => {
    if (boltPath) boltPath.style.opacity = '0';
  }, { passive: true });
}




  // Filters lists (Drawer)
  const catListEl = el('catList');
  const subcatListEl = el('subcatList');
  const brandListEl = el('brandList');
  const originListEl = el('originList');

  const priceMinEl = el('priceMin');
  const priceMaxEl = el('priceMax');
  const priceMinLabelEl = el('priceMinLabel');
  const priceMaxLabelEl = el('priceMaxLabel');

  // Toolbar
  const sortSelect = el('sortSelect');
  const saleOnlyToggle = el('saleOnlyToggle');

  const viewGridBtn = el('viewGrid');
  const viewListBtn = el('viewList');

  // Grid columns controls (3/4/5)
  const gridColsBtns = document.querySelectorAll('.grid-btn');

  
  // Bind electric spark effect to grid buttons
  if (gridColsBtns?.length) gridColsBtns.forEach((b) => bindSpark(b));

  // Hero lightning electricity effect
  initHeroLightning();
// Pagination target
  const paginationEl = el('catPagination');

  // Grid target
  const gridEl = el('catProductsGrid');

  // Drawer
  const openFiltersBtn = el('openFilters');
  const filterOverlay = el('filterOverlay');
  const filterDrawer = el('filterDrawer');
  const closeFiltersBtn = el('closeFilters');
  const applyFiltersBtn = el('applyFilters');
  const clearBtn = el('clearFilters');

  if (!gridEl || typeof products === 'undefined') return;

  const LABELS = {
    led: 'إضاءة LED',
    chandeliers: 'ثريات وإنارة',
    switches: 'مفاتيح ومقابس',
    outdoor: 'إضاءة خارجية',
    tools: 'مستلزمات كهرباء',
  };

  const uniq = (arr) => [...new Set(arr.filter(Boolean))];

  const escapeHtml = (str) =>
    String(str)
      .replaceAll('&', '&amp;')
      .replaceAll('<', '&lt;')
      .replaceAll('>', '&gt;')
      .replaceAll('"', '&quot;')
      .replaceAll("'", '&#039;');

  const toIntMoney = (v) =>
    typeof normalizeIQD === 'function'
      ? normalizeIQD(v)
      : Number(String(v).replace(/[^0-9]/g, '') || 0);

  const fmtInt = (n) => Number(n || 0).toLocaleString('en-US');

  const state = {
    category: 'all',
    subcats: new Set(),
    brands: new Set(),
    origins: new Set(),
    priceMin: 0,
    priceMax: 0,
    sort: 'newest',
    view: 'grid',
    gridCols: 3,
    saleOnly: false,
    page: 1,
    perPage: 12,
    _maxPrice: 0,
  };

  function countForCategory(key) {
    return products.filter((p) => key === 'all' || String(p.category) === String(key)).length;
  }

  function listForCategory(catKey) {
    return products.filter((p) => catKey === 'all' || String(p.category) === String(catKey));
  }

  function subcatsForCategory(catKey) {
    return uniq(listForCategory(catKey).map((p) => p.subcategory)).sort((a, b) => a.localeCompare(b, 'ar'));
  }

  function brandsForCategory(catKey) {
    return uniq(listForCategory(catKey).map((p) => p.brand)).sort((a, b) => a.localeCompare(b, 'en'));
  }

  function originsForCategory(catKey) {
    return uniq(listForCategory(catKey).map((p) => p.originCountry)).sort((a, b) => a.localeCompare(b, 'ar'));
  }

  function renderRadioList(targetEl, items, activeValue, onChange, { labelFn, countFn } = {}) {
    if (!targetEl) return;
    if (!items.length) {
      targetEl.innerHTML = `<div class="text-gray-500 text-sm">لا توجد خيارات حالياً.</div>`;
      return;
    }

    targetEl.innerHTML = items
      .map((v, idx) => {
        const id = `r_${targetEl.id}_${idx}`;
        const label = labelFn ? labelFn(v) : v;
        const cnt = countFn ? countFn(v) : null;
        return `
          <label class="filter-item" for="${id}">
            <span class="filter-left">
              <input id="${id}" type="radio" name="${targetEl.id}_radio" value="${escapeHtml(v)}" ${String(v) === String(activeValue) ? 'checked' : ''}>
              <span class="filter-text">${escapeHtml(label)}</span>
            </span>
            ${cnt !== null ? `<span class="filter-count">${fmtInt(cnt)}</span>` : ''}
          </label>
        `;
      })
      .join('');

    targetEl.querySelectorAll('input[type="radio"]').forEach((inp) => {
      inp.addEventListener('change', () => onChange(inp.value));
    });
  }

  function renderCheckboxList(targetEl, items, activeSet, onToggle, { countFn } = {}) {
    if (!targetEl) return;
    if (!items.length) {
      targetEl.innerHTML = `<div class="text-gray-500 text-sm">لا توجد خيارات حالياً.</div>`;
      return;
    }

    targetEl.innerHTML = items
      .map((v, idx) => {
        const id = `c_${targetEl.id}_${idx}`;
        const checked = activeSet?.has(String(v)) ? 'checked' : '';
        const cnt = countFn ? countFn(v) : null;
        return `
          <label class="filter-item" for="${id}">
            <span class="filter-left">
              <input id="${id}" type="checkbox" value="${escapeHtml(v)}" ${checked}>
              <span class="filter-text">${escapeHtml(v)}</span>
            </span>
            ${cnt !== null ? `<span class="filter-count">${fmtInt(cnt)}</span>` : ''}
          </label>
        `;
      })
      .join('');

    targetEl.querySelectorAll('input[type="checkbox"]').forEach((inp) => {
      inp.addEventListener('change', () => onToggle(inp.value, inp.checked));
    });
  }

  function renderCategories() {
    const keys = ['all', ...uniq(products.map((p) => p.category))];

    renderRadioList(
      catListEl,
      keys,
      state.category,
      (val) => {
        state.category = val || 'all';
        state.subcats = new Set();
        state.brands = new Set();
        state.origins = new Set();
        state.page = 1;
        renderAllFilters();
        renderResults();
      },
      {
        labelFn: (key) => (key === 'all' ? 'كل الفئات' : LABELS[key] || key),
        countFn: (key) => countForCategory(key),
      }
    );
  }

  function renderAllFilters() {
    renderCategories();

    // Subcategories
    const subcats = subcatsForCategory(state.category);
    renderCheckboxList(
      subcatListEl,
      subcats,
      state.subcats,
      (val, checked) => {
        if (!val) return;
        if (checked) state.subcats.add(String(val));
        else state.subcats.delete(String(val));
        state.page = 1;
        renderResults();
      },
      {
        countFn: (s) =>
          products.filter(
            (p) =>
              (state.category === 'all' || String(p.category) === String(state.category)) &&
              String(p.subcategory) === String(s)
          ).length,
      }
    );

    // Brands
    const brands = brandsForCategory(state.category);
    renderCheckboxList(
      brandListEl,
      brands,
      state.brands,
      (val, checked) => {
        if (!val) return;
        if (checked) state.brands.add(String(val));
        else state.brands.delete(String(val));
        state.page = 1;
        renderResults();
      },
      {
        countFn: (b) =>
          products.filter(
            (p) =>
              (state.category === 'all' || String(p.category) === String(state.category)) &&
              String(p.brand) === String(b)
          ).length,
      }
    );

    // Origins
    const origins = originsForCategory(state.category);
    renderCheckboxList(
      originListEl,
      origins,
      state.origins,
      (val, checked) => {
        if (!val) return;
        if (checked) state.origins.add(String(val));
        else state.origins.delete(String(val));
        state.page = 1;
        renderResults();
      },
      {
        countFn: (o) =>
          products.filter(
            (p) =>
              (state.category === 'all' || String(p.category) === String(state.category)) &&
              String(p.originCountry) === String(o)
          ).length,
      }
    );

    // Price labels
    if (priceMinLabelEl) priceMinLabelEl.textContent = fmtInt(state.priceMin);
    if (priceMaxLabelEl) priceMaxLabelEl.textContent = fmtInt(state.priceMax);
  }

  function computeFiltered() {
    let list = [...products];

    if (state.category !== 'all') list = list.filter((p) => String(p.category) === String(state.category));
    if (state.subcats.size) list = list.filter((p) => state.subcats.has(String(p.subcategory || '')));
    if (state.brands.size) list = list.filter((p) => state.brands.has(String(p.brand || '')));
    if (state.origins.size) list = list.filter((p) => state.origins.has(String(p.originCountry || '')));

    // Price range
    if (state.priceMax > 0) {
      list = list.filter((p) => {
        const price = toIntMoney(p.price);
        return price >= state.priceMin && price <= state.priceMax;
      });
    }

    // Sale only (يعتبر المنتج ضمن العروض إذا كان originalPrice أكبر من السعر)
    if (state.saleOnly) {
      list = list.filter((p) => {
        const now = toIntMoney(p.price);
        const old = toIntMoney(p.originalPrice);
        return old > 0 && old > now;
      });
    }

    // Sort
    const byPrice = (p) => toIntMoney(p.price);
    const byDate = (p) => Date.parse(p.createdAt || '1970-01-01');
    const byRating = (p) => Number(p.rating || 0);

    switch (state.sort) {
      case 'price_asc':
        list.sort((a, b) => byPrice(a) - byPrice(b));
        break;
      case 'price_desc':
        list.sort((a, b) => byPrice(b) - byPrice(a));
        break;
      case 'rating_desc':
        list.sort((a, b) => byRating(b) - byRating(a));
        break;
      case 'newest':
      default:
        list.sort((a, b) => byDate(b) - byDate(a));
        break;
    }

    return list;
  }

  function applyView() {
    if (!gridEl) return;
    if (state.view === 'list') {
      gridEl.className = 'grid grid-cols-1 gap-4';
      viewListBtn?.classList.add('active');
      viewGridBtn?.classList.remove('active');
    } else {
      const cols = Number(state.gridCols || 3);
      // افتراضي: 3 أعمدة على الشاشات الكبيرة
      // ملاحظة: Tailwind CDN يدعم الكلاسات الديناميكية هنا لأنها موجودة كنص في الملفات (3/4/5)
      gridEl.className = `grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-${cols} gap-6`;
      viewGridBtn?.classList.add('active');
      viewListBtn?.classList.remove('active');
    }
  }

  function setPage(p) {
    const next = Math.max(1, Number(p || 1));
    state.page = next;
    renderResults();
    // scroll to products (soft)
    gridEl?.scrollIntoView({ behavior: 'smooth', block: 'start' });
  }

  function renderPagination(totalItems) {
    if (!paginationEl) return;

    const perPage = Math.max(1, Number(state.perPage || 12));
    const totalPages = Math.max(1, Math.ceil(totalItems / perPage));
    state.page = Math.min(state.page, totalPages);

    // لا نعرض Pagination إذا صفحة واحدة فقط
    if (totalPages <= 1) {
      paginationEl.innerHTML = '';
      return;
    }

    const current = state.page;

    // نافذة صفحات بسيطة حول الصفحة الحالية
    const windowSize = 5;
    const half = Math.floor(windowSize / 2);
    let start = Math.max(1, current - half);
    let end = Math.min(totalPages, start + windowSize - 1);
    start = Math.max(1, end - windowSize + 1);

    let pagesHtml = '';
    for (let i = start; i <= end; i++) {
      const isActive = i === current;
      pagesHtml += `
        <button
          class="w-10 h-10 rounded-full ${isActive ? 'bg-[var(--primary-gold)] text-black' : 'border border-white/10 text-white/70 hover:bg-white/10'} font-black transition"
          data-page="${i}"
          type="button"
          aria-current="${isActive ? 'page' : 'false'}"
        >${i}</button>
      `;
    }

    paginationEl.innerHTML = `
      <nav class="flex items-center gap-2 text-sm" dir="rtl" aria-label="Pagination">
        <button
          class="px-4 h-10 rounded-full border border-white/10 text-white/70 hover:bg-white/10 disabled:opacity-40 disabled:cursor-not-allowed transition"
          ${current === 1 ? 'disabled' : ''}
          data-prev
          type="button"
        >السابق</button>

        <div class="flex items-center gap-2">${pagesHtml}</div>

        <button
          class="px-4 h-10 rounded-full border border-white/10 text-white/70 hover:bg-white/10 disabled:opacity-40 disabled:cursor-not-allowed transition"
          ${current === totalPages ? 'disabled' : ''}
          data-next
          type="button"
        >التالي</button>
      </nav>
    `;

    paginationEl.querySelector('[data-prev]')?.addEventListener('click', () => setPage(current - 1));
    paginationEl.querySelector('[data-next]')?.addEventListener('click', () => setPage(current + 1));
    paginationEl.querySelectorAll('[data-page]')?.forEach((btn) => {
      btn.addEventListener('click', () => setPage(Number(btn.getAttribute('data-page'))));
    });
  }

  function renderListInto(targetId, list) {
    const wrap = el(targetId);
    if (!wrap) return;

    if (!list.length) {
      wrap.innerHTML = `<div class="text-center glass rounded-3xl p-10 text-gray-300">لا توجد منتجات حالياً.</div>`;
      return;
    }

    wrap.innerHTML = list
      .map((p) => {
        const badge = p.badge
          ? `<span class="badge" style="background: rgba(212,163,86,.18); border:1px solid rgba(212,163,86,.30)">${escapeHtml(
              p.badge
            )}</span>`
          : '';
        const old = p.originalPrice ? `<del>${typeof formatIQD === 'function' ? formatIQD(p.originalPrice) : p.originalPrice}</del>` : '';
        const price = typeof formatIQD === 'function' ? formatIQD(p.price) : p.price;

        return `
          <div class="glass rounded-3xl p-4 sm:p-5 border border-white/10 flex gap-4 items-center relative">
            <a href="product.html?id=${p.id}" aria-label="فتح ${escapeHtml(p.name)}" class="absolute inset-0"></a>
            <div class="relative w-24 h-24 rounded-2xl overflow-hidden bg-white/5 shrink-0">
              ${badge}
              <img src="${p.image}" alt="${escapeHtml(p.name)}" class="w-full h-full object-cover opacity-90">
            </div>

            <div class="flex-1 min-w-0">
              <div class="flex items-center justify-between gap-3">
                <h3 class="font-black text-lg truncate">${escapeHtml(p.name)}</h3>
                <div class="text-xs text-gray-400 font-bold">${escapeHtml(p.brand || '')} • ${escapeHtml(p.originCountry || '')}</div>
              </div>
              <p class="mt-1 text-sm text-gray-300 line-clamp-2">${escapeHtml(p.description || '')}</p>

              <div class="mt-3 flex items-center justify-between gap-3">
                <div class="flex items-center gap-3">
                  <div class="font-black text-[#d4a356]">${price}</div>
                  <div class="text-xs text-gray-500">${old}</div>
                  <div class="text-xs text-gray-400 font-bold">⭐ ${Number(p.rating || 0).toFixed(1)}</div>
                </div>

                <div class="flex items-center gap-2 relative z-10">
                  <button class="action-btn" title="عرض سريع" data-qv="${p.id}">
                    <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.477 0 8.268 2.943 9.542 7-1.274 4.057-5.065 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                    </svg>
                  </button>

                  <button class="btn-primary h-10 px-4 rounded-2xl font-black" data-add="${p.id}">
                    إضافة إلى السلة
                  </button>
                </div>
              </div>
            </div>
          </div>
        `;
      })
      .join('');

    wrap.querySelectorAll('[data-add]').forEach((b) => {
      b.addEventListener('click', () => window.addToCart && window.addToCart(Number(b.getAttribute('data-add'))));
    });
    wrap.querySelectorAll('[data-qv]').forEach((b) => {
      b.addEventListener('click', () => window.openQuickView && window.openQuickView(Number(b.getAttribute('data-qv'))));
    });
  }

  function renderResults() {
    const list = computeFiltered();

    // Pagination slice
    const perPage = Math.max(1, Number(state.perPage || 12));
    const total = list.length;
    const totalPages = Math.max(1, Math.ceil(total / perPage));
    state.page = Math.min(Math.max(1, state.page), totalPages);
    const start = (state.page - 1) * perPage;
    const pageList = list.slice(start, start + perPage);

    applyView();

    if (state.view === 'list') {
      renderListInto('catProductsGrid', pageList);
    } else {
      if (typeof renderGridInto === 'function') renderGridInto('catProductsGrid', pageList);
    }

    renderPagination(total);
  }

  // Price range setup
  function clampPriceState() {
    const minV = Number(priceMinEl?.value || 0);
    const maxV = Number(priceMaxEl?.value || 0);
    const a = Math.min(minV, maxV);
    const b = Math.max(minV, maxV);
    state.priceMin = a;
    state.priceMax = b;
    if (priceMinEl) priceMinEl.value = String(a);
    if (priceMaxEl) priceMaxEl.value = String(b);
    if (priceMinLabelEl) priceMinLabelEl.textContent = fmtInt(a);
    if (priceMaxLabelEl) priceMaxLabelEl.textContent = fmtInt(b);
  }

  if (priceMinEl && priceMaxEl) {
    const maxPrice = Math.max(...products.map((p) => toIntMoney(p.price)), 0);
    const roundedMax = Math.ceil(maxPrice / 5000) * 5000 || 300000;
    state._maxPrice = roundedMax;

    [priceMinEl, priceMaxEl].forEach((r) => {
      r.min = '0';
      r.max = String(roundedMax);
      r.step = '5000';
    });

    state.priceMin = 0;
    state.priceMax = roundedMax;
    priceMinEl.value = '0';
    priceMaxEl.value = String(roundedMax);
    clampPriceState();

    const onPriceChange = () => {
      clampPriceState();
      state.page = 1;
      renderResults();
    };
    priceMinEl.addEventListener('input', onPriceChange);
    priceMaxEl.addEventListener('input', onPriceChange);
  }

  // Toolbar events
  sortSelect?.addEventListener('change', () => {
    state.sort = sortSelect.value || 'newest';
    state.page = 1;
    renderResults();
  });

  saleOnlyToggle?.addEventListener('change', () => {
    state.saleOnly = !!saleOnlyToggle.checked;
    state.page = 1;
    renderResults();
  });

  viewGridBtn?.addEventListener('click', () => {
    state.view = 'grid';
    renderResults();
  });

  viewListBtn?.addEventListener('click', () => {
    state.view = 'list';
    renderResults();
  });

  // Grid columns events
  if (gridColsBtns?.length) {
    gridColsBtns.forEach((btn) => {
      btn.addEventListener('click', () => {
        const cols = Number(btn.getAttribute('data-cols') || 3);
        state.gridCols = [3, 4, 5].includes(cols) ? cols : 3;
        gridColsBtns.forEach((b) => b.classList.remove('active'));
        btn.classList.add('active');
        if (state.view !== 'grid') state.view = 'grid';
        renderResults();
      });
    });
  }

  if (clearBtn) {
    clearBtn.addEventListener('click', () => {
      state.category = 'all';
      state.subcats = new Set();
      state.brands = new Set();
      state.origins = new Set();
      state.priceMin = 0;
      state.priceMax = state._maxPrice || 0;
      state.sort = 'newest';
      state.saleOnly = false;
      state.page = 1;
      state.gridCols = 3;

      if (sortSelect) sortSelect.value = 'newest';
      if (saleOnlyToggle) saleOnlyToggle.checked = false;

      // Reset columns UI
      if (gridColsBtns?.length) {
        gridColsBtns.forEach((b) => b.classList.remove('active'));
        const first = Array.from(gridColsBtns).find((b) => String(b.getAttribute('data-cols')) === '3');
        first?.classList.add('active');
      }

      if (priceMinEl) priceMinEl.value = String(state.priceMin);
      if (priceMaxEl) priceMaxEl.value = String(state.priceMax);

      renderAllFilters();
      renderResults();
    });
  }

  // Drawer open/close
  function openDrawer() {
    filterOverlay?.classList.add('open');
    filterDrawer?.classList.add('open');
  }
  function closeDrawer() {
    filterOverlay?.classList.remove('open');
    filterDrawer?.classList.remove('open');
  }

  openFiltersBtn?.addEventListener('click', openDrawer);
  closeFiltersBtn?.addEventListener('click', closeDrawer);
  filterOverlay?.addEventListener('click', closeDrawer);
  applyFiltersBtn?.addEventListener('click', closeDrawer);

  document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') closeDrawer();
  });

  // Init
  state.sort = sortSelect?.value || 'newest';
  state.saleOnly = !!saleOnlyToggle?.checked;

  // Default columns = 3
  state.gridCols = 3;
  if (gridColsBtns?.length) {
    gridColsBtns.forEach((b) => b.classList.remove('active'));
    const first = Array.from(gridColsBtns).find((b) => String(b.getAttribute('data-cols')) === '3');
    first?.classList.add('active');
  }

  renderAllFilters();
  renderResults();
})();
