/* Different Store - app.js (Vanilla JS, RTL, IQD int amounts)
   - Cart Drawer (Right)
   - Wishlist Drawer (Right)
   - Compare Drawer (Left)
   - Quick View Modal
   - Products loaded from database API
*/

// Products array - will be populated from API
let products = [];

// Flag to track if products have been loaded
let productsLoaded = false;

// Fetch products from database API
async function fetchProducts(params = {}) {
    try {
        const queryString = new URLSearchParams(params).toString();
        const url = `/api/store/products${queryString ? '?' + queryString : ''}`;
        const response = await fetch(url);
        const data = await response.json();

        if (data.success && Array.isArray(data.products)) {
            products = data.products;
            productsLoaded = true;
            return { products: data.products, pagination: data.pagination };
        }
        return { products: [], pagination: null };
    } catch (error) {
        console.error('Error fetching products:', error);
        return { products: [], pagination: null };
    }
}

// Fetch categories for filters
async function fetchCategories() {
    try {
        const response = await fetch('/api/store/categories');
        const data = await response.json();
        return data.success ? data.categories : [];
    } catch (error) {
        console.error('Error fetching categories:', error);
        return [];
    }
}

// Fetch brands for filters
async function fetchBrands() {
    try {
        const response = await fetch('/api/store/brands');
        const data = await response.json();
        return data.success ? data.brands : [];
    } catch (error) {
        console.error('Error fetching brands:', error);
        return [];
    }
}

// Fallback mock products for development (will be removed when API is ready)
const mockProducts = [
    {
        id: 1,
        category: 'led',
        subcategory: 'مصابيح ذكية',
        brand: 'Schneider',
        originCountry: 'ألمانيا',
        createdAt: '2026-01-10',
        rating: 3.9,
        name: 'مصباح LED ذكي',
        description: 'إضاءة قابلة للتحكم عن بعد مع إمكانية تغيير الألوان',
        price: 299,
        originalPrice: 399,
        image: 'https://images.unsplash.com/photo-1550985616-10810253b84d?w=600&h=600&fit=crop',
        badge: 'جديد',
        badgeColor: 'bg-[#baa263]',
        options: {
            color: [
                { name: 'أبيض دافئ', value: '#FFF4E6', code: 'warm-white' },
                { name: 'أبيض بارد', value: '#E8F4F8', code: 'cool-white' },
                { name: 'RGB', value: 'linear-gradient(135deg, #FF6B6B, #4ECDC4, #45B7D1)', code: 'rgb' },
            ],
            power: ['9W', '12W', '18W'],
        },
        inStock: true,
    },
    {
        id: 2,
        category: 'chandeliers',
        subcategory: 'ثريات كريستال',
        brand: 'Legrand',
        originCountry: 'فرنسا',
        createdAt: '2026-01-19',
        rating: 4.2,
        name: 'ثريا كريستال فاخرة',
        description: 'تصميم عصري وأنيق من الكريستال الأصلي',
        price: 1499,
        originalPrice: 2150,
        image: 'https://images.unsplash.com/photo-1540932239986-30128078f3c5?w=600&h=600&fit=crop',
        badge: 'خصم 30%',
        badgeColor: 'bg-red-500',
        options: {
            color: [
                { name: 'ذهبي', value: '#D4AF37', code: 'gold' },
                { name: 'فضي', value: '#C0C0C0', code: 'silver' },
                { name: 'برونزي', value: '#CD7F32', code: 'bronze' },
            ],
            size: ['60 سم', '80 سم', '100 سم'],
        },
        inStock: true,
    },
    {
        id: 3,
        category: 'switches',
        subcategory: 'مفاتيح ذكية',
        brand: 'Philips',
        originCountry: 'الصين',
        createdAt: '2026-01-28',
        rating: 4.5,
        name: 'مفتاح كهربائي ذكي',
        description: 'تحكم باللمس والصوت مع دعم Alexa و Google Home',
        price: 179,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1621905252507-b35492cc74b4?w=600&h=600&fit=crop',
        badge: null,
        badgeColor: null,
        options: {
            color: [
                { name: 'أبيض', value: '#FFFFFF', code: 'white' },
                { name: 'أسود', value: '#1a1a1a', code: 'black' },
                { name: 'رمادي', value: '#808080', code: 'gray' },
            ],
            size: ['مفتاح واحد', 'مفتاحين', '3 مفاتيح'],
        },
        inStock: true,
    },
    {
        id: 4,
        category: 'led',
        subcategory: 'مصابيح ليلية',
        brand: 'Osram',
        originCountry: 'تركيا',
        createdAt: '2026-02-06',
        rating: 4.8,
        name: 'مصباح ليلي LED',
        description: 'إضاءة هادئة وموفرة للطاقة مع مستشعر حركة',
        price: 89,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1507473885765-e6ed057f782c?w=600&h=600&fit=crop',
        badge: 'الأكثر مبيعاً',
        badgeColor: 'bg-[#baa263]',
        options: {
            color: [
                { name: 'أبيض دافئ', value: '#FFF4E6', code: 'warm' },
                { name: 'أصفر', value: '#FFD700', code: 'yellow' },
            ],
            power: ['1.5W', '3W'],
        },
        inStock: true,
    },
    {
        id: 5,
        category: 'chandeliers',
        subcategory: 'إنارة معلّقة',
        brand: 'Panasonic',
        originCountry: 'إيطاليا',
        createdAt: '2026-02-15',
        rating: 3.6,
        name: 'مصباح معلق حديث',
        description: 'تصميم معاصر مثالي للمطابخ وغرف الطعام',
        price: 449,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1534105615256-13940a6b3a5e?w=600&h=600&fit=crop',
        badge: null,
        badgeColor: null,
        options: {
            color: [
                { name: 'أسود مات', value: '#2C2C2C', code: 'black-matte' },
                { name: 'ذهبي', value: '#D4AF37', code: 'gold' },
                { name: 'نحاسي', value: '#B87333', code: 'copper' },
            ],
            size: ['25 سم', '35 سم', '45 سم'],
        },
        inStock: true,
    },
    {
        id: 6,
        category: 'led',
        subcategory: 'إنارة جدارية',
        brand: 'Hager',
        originCountry: 'اليابان',
        createdAt: '2026-02-24',
        rating: 3.9,
        name: 'مصباح جداري',
        description: 'إضاءة ديكورية للجدران بتصميم عصري',
        price: 239,
        originalPrice: 299,
        image: 'https://images.unsplash.com/photo-1517467139951-f5a925c9f9de?w=600&h=600&fit=crop',
        badge: 'خصم 20%',
        badgeColor: 'bg-red-500',
        options: {
            color: [
                { name: 'أبيض', value: '#FFFFFF', code: 'white' },
                { name: 'أسود', value: '#1a1a1a', code: 'black' },
            ],
            power: ['12W', '18W'],
        },
        inStock: true,
    },
    {
        id: 7,
        category: 'led',
        subcategory: 'شرائط LED',
        brand: 'Schneider',
        originCountry: 'ألمانيا',
        createdAt: '2026-03-05',
        rating: 4.2,
        name: 'شريط LED ملون',
        description: 'RGB قابل للتحكم بالتطبيق مع 16 مليون لون',
        price: 199,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1558618666-fcd25c85cd64?w=600&h=600&fit=crop',
        badge: 'جديد',
        badgeColor: 'bg-[#baa263]',
        options: {
            color: [{ name: 'RGB', value: 'linear-gradient(135deg, #FF0000, #00FF00, #0000FF)', code: 'rgb' }],
            size: ['3 متر', '5 متر', '10 متر'],
        },
        inStock: true,
    },
    {
        id: 8,
        category: 'chandeliers',
        subcategory: 'إنارة أرضية',
        brand: 'Legrand',
        originCountry: 'فرنسا',
        createdAt: '2026-03-14',
        rating: 4.5,
        name: 'مصباح أرضي حديث',
        description: 'تصميم أنيق مثالي للصالات وغرف المعيشة',
        price: 599,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1545042746-31d37e738458?w=600&h=600&fit=crop',
        badge: null,
        badgeColor: null,
        options: {
            color: [
                { name: 'أسود', value: '#1a1a1a', code: 'black' },
                { name: 'ذهبي', value: '#D4AF37', code: 'gold' },
                { name: 'فضي', value: '#C0C0C0', code: 'silver' },
            ],
            size: ['150 سم', '170 سم', '190 سم'],
        },
        inStock: true,
    },

    /* عناصر إضافية للمتجر */
    {
        id: 9,
        category: 'outdoor',
        subcategory: 'كشافات خارجية',
        brand: 'Philips',
        originCountry: 'الصين',
        createdAt: '2026-03-23',
        rating: 4.8,
        name: 'كشاف خارجي ضد الماء',
        description: 'IP65 مناسب للحدائق والواجهات',
        price: 259,
        originalPrice: 329,
        image: 'https://images.unsplash.com/photo-1524484485831-a92ffc0de03f?w=600&h=600&fit=crop',
        badge: 'عرض',
        badgeColor: 'bg-red-500',
        options: {
            power: ['30W', '50W', '100W'],
            color: [
                { name: 'أبيض بارد', value: '#E8F4F8', code: 'cool' },
                { name: 'أبيض دافئ', value: '#FFF4E6', code: 'warm' },
            ],
        },
        inStock: true,
    },
    {
        id: 10,
        category: 'switches',
        subcategory: 'مقابس',
        brand: 'Osram',
        originCountry: 'تركيا',
        createdAt: '2026-04-01',
        rating: 3.6,
        name: 'مقبس جداري USB',
        description: 'مقبس 220V مع منفذي USB للشحن السريع',
        price: 119,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1581092160562-40aa08e78837?w=600&h=600&fit=crop',
        badge: null,
        badgeColor: null,
        options: {
            color: [
                { name: 'أبيض', value: '#FFFFFF', code: 'white' },
                { name: 'أسود', value: '#1a1a1a', code: 'black' },
            ],
            size: ['مقبس واحد', 'مقبسين'],
        },
        inStock: true,
    },
    {
        id: 11,
        category: 'tools',
        subcategory: 'قواطع وحماية',
        brand: 'Panasonic',
        originCountry: 'إيطاليا',
        createdAt: '2026-04-10',
        rating: 3.9,
        name: 'قاطع كهربائي (MCB)',
        description: 'قاطع حماية 1P/2P مع تيارات متعددة',
        price: 39,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1581092580497-e0d23cbdf1dc?w=600&h=600&fit=crop',
        badge: 'أساسي',
        badgeColor: 'bg-[#baa263]',
        options: {
            size: ['1P', '2P'],
            power: ['10A', '16A', '25A', '32A'],
        },
        inStock: true,
    },
    {
        id: 12,
        category: 'tools',
        subcategory: 'أسلاك وكابلات',
        brand: 'Hager',
        originCountry: 'اليابان',
        createdAt: '2026-04-19',
        rating: 4.2,
        name: 'سلك نحاس معزول',
        description: 'جودة عالية للأعمال المنزلية والمشاريع',
        price: 59,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1541534401786-2077eed87a72?w=600&h=600&fit=crop',
        badge: null,
        badgeColor: null,
        options: {
            size: ['1.5mm', '2.5mm', '4mm'],
            color: [
                { name: 'أحمر', value: '#ff3b30', code: 'red' },
                { name: 'أزرق', value: '#007aff', code: 'blue' },
                { name: 'أصفر', value: '#ffcc00', code: 'yellow' },
            ],
        },
        inStock: true,
    },
    {
        id: 13,
        category: 'outdoor',
        subcategory: 'إنارة شوارع',
        brand: 'Schneider',
        originCountry: 'ألمانيا',
        createdAt: '2026-04-28',
        rating: 4.5,
        name: 'إنارة شارع LED',
        description: 'مناسبة للأزقة والمداخل مع عدسة تركيز',
        price: 899,
        originalPrice: 1099,
        image: 'https://images.unsplash.com/photo-1484704849700-f032a568e944?w=600&h=600&fit=crop',
        badge: 'خصم',
        badgeColor: 'bg-red-500',
        options: {
            power: ['100W', '150W', '200W'],
            size: ['عدسة 90°', 'عدسة 120°'],
        },
        inStock: true,
    },
    {
        id: 14,
        category: 'led',
        subcategory: 'سبوت لايت',
        brand: 'Legrand',
        originCountry: 'فرنسا',
        createdAt: '2026-05-07',
        rating: 4.8,
        name: 'سبوت لايت سقفي',
        description: 'إضاءة موجهة للديكور (سبوت)',
        price: 69,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1519710164239-da123dc03ef4?w=600&h=600&fit=crop',
        badge: null,
        badgeColor: null,
        options: {
            power: ['7W', '10W', '12W'],
            color: [
                { name: 'أبيض دافئ', value: '#FFF4E6', code: 'warm' },
                { name: 'أبيض بارد', value: '#E8F4F8', code: 'cool' },
            ],
        },
        inStock: true,
    },
    {
        id: 15,
        category: 'chandeliers',
        subcategory: 'ثريات مودرن',
        brand: 'Philips',
        originCountry: 'الصين',
        createdAt: '2026-05-16',
        rating: 3.6,
        name: 'ثريا حلقات LED',
        description: 'تصميم حلقي مودرن مع تحكم بالسطوع',
        price: 1299,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1519710164239-da123dc03ef4?w=600&h=600&fit=crop',
        badge: 'مودرن',
        badgeColor: 'bg-[#baa263]',
        options: {
            size: ['حلقة واحدة', 'حلقتين', '3 حلقات'],
            color: [
                { name: 'ذهبي', value: '#D4AF37', code: 'gold' },
                { name: 'أسود', value: '#1a1a1a', code: 'black' },
            ],
        },
        inStock: true,
    },
    {
        id: 16,
        category: 'switches',
        subcategory: 'قواطع ذكية',
        brand: 'Osram',
        originCountry: 'تركيا',
        createdAt: '2026-05-25',
        rating: 3.9,
        name: 'قاطع ذكي WiFi',
        description: 'تحكم عبر التطبيق ومراقبة الاستهلاك',
        price: 229,
        originalPrice: null,
        image: 'https://images.unsplash.com/photo-1621905251189-08b45d6a269e?w=600&h=600&fit=crop',
        badge: 'ذكي',
        badgeColor: 'bg-[#baa263]',
        options: {
            size: ['1 قناة', '2 قناة', '4 قنوات'],
            power: ['10A', '16A'],
        },
        inStock: true,
    },
];

// ----------------------------
// State (LocalStorage)
// ----------------------------
const LS = {
    cart: 'different_cart_v1',
    wishlist: 'different_wishlist_v1',
    compare: 'different_compare_v1',
};

let cart = loadState(LS.cart, []);
let wishlist = loadState(LS.wishlist, []);
let compare = loadState(LS.compare, []);

// Products filter
let currentFilter = 'all';

// Quick View selection (يدعم عدة متغيرات: لون/مقاس/قدرة/..)
let quickViewSelection = {
    productId: null,
    options: {},
};

// ----------------------------
// Helpers
// ----------------------------
function loadState(key, fallback) {
    try {
        const raw = localStorage.getItem(key);
        return raw ? JSON.parse(raw) : fallback;
    } catch (e) {
        return fallback;
    }
}

function saveState(key, value) {
    try {
        localStorage.setItem(key, JSON.stringify(value));
    } catch (e) { }
}

function normalizeIQD(amount) {
    const n = Number(amount || 0);
    if (!Number.isFinite(n)) return 0;
    // بعض الداتا القديمة تكون "بالألف" (مثلاً 179 يعني 179,000)
    return n < 1000 ? Math.round(n * 1000) : Math.round(n);
}

function formatIQD(amount) {
    const v = normalizeIQD(amount);
    try {
        // ✅ الأرقام الإنجليزية (حسب طلبك)
        return new Intl.NumberFormat('en-US', { maximumFractionDigits: 0 }).format(v) + ' د.ع';
    } catch (e) {
        return String(v) + ' د.ع';
    }
}


// ===================== Pricing Helpers (Options + Qty) =====================
function computeUnitPrice(product, selectedOptions = {}) {
    const base = normalizeIQD(product?.price ?? 0);

    // 0) Variant price table (سعر نهائي حسب تركيبة المتغيرات)
    // شكل الداتا المقترح:
    // variantPrices: [
    //   { options: { size: '80 سم', color: 'gold' }, price: 185000 },
    //   { options: { power: '16A' }, price: 259000 },
    // ]
    // إذا وجدنا تطابق كامل نستخدم السعر النهائي مباشرة.
    if (Array.isArray(product?.variantPrices) && product.variantPrices.length) {
        const match = product.variantPrices.find((row) => {
            const opts = row?.options && typeof row.options === 'object' ? row.options : {};
            // يجب أن تطابق كل مفاتيح row.options قيم الاختيار الحالية
            for (const [k, v] of Object.entries(opts)) {
                if (String(selectedOptions?.[k] ?? '') !== String(v)) return false;
            }
            return true;
        });
        if (match && match.price != null) {
            return Math.max(0, normalizeIQD(match.price));
        }
    }

    let delta = 0;

    // 1) explicit mapping
    const pricing = product?.optionPricing || {};
    for (const [groupKey, mapping] of Object.entries(pricing)) {
        const chosen = selectedOptions[groupKey];
        if (chosen != null && Object.prototype.hasOwnProperty.call(mapping, chosen)) {
            delta += normalizeIQD(mapping[chosen]);
        }
    }

    // 2) heuristics fallback (إن لم يتم تعريف optionPricing)
    // power: 9W / 12W / 18W => +0 / +40k / +80k
    if (!Object.keys(pricing).length && selectedOptions.power) {
        const w = parseInt(String(selectedOptions.power).replace(/\D+/g, ''), 10);
        if (Number.isFinite(w)) {
            if (w >= 18) delta += 80000;
            else if (w >= 12) delta += 40000;
        }
    }
    // size: 60/80/100 سم => +0 / +50k / +90k
    if (!Object.keys(pricing).length && selectedOptions.size) {
        const s = parseInt(String(selectedOptions.size).replace(/\D+/g, ''), 10);
        if (Number.isFinite(s)) {
            if (s >= 100) delta += 90000;
            else if (s >= 80) delta += 50000;
        }
        // حلقات (حلقة واحدة / حلقتين / 3 حلقات) => +0 / +60k / +110k
        if (!Number.isFinite(s)) {
            const raw = String(selectedOptions.size);
            const n = raw.includes('حلقتين') ? 2 : (raw.includes('واحدة') ? 1 : parseInt(raw.replace(/\D+/g, ''), 10));
            if (Number.isFinite(n)) {
                if (n >= 3) delta += 110000;
                else if (n >= 2) delta += 60000;
            }
        }
    }

    // قنوات (1 قناة / 2 قناة / 4 قنوات) => +0 / +40k / +90k
    if (!Object.keys(pricing).length && selectedOptions.size) {
        const raw = String(selectedOptions.size);
        if (raw.includes('قناة')) {
            const n = parseInt(raw.replace(/\D+/g, ''), 10);
            if (Number.isFinite(n)) {
                if (n >= 4) delta += 90000;
                else if (n >= 2) delta += 40000;
            }
        }
    }
    // color: RGB => +20k
    if (!Object.keys(pricing).length && selectedOptions.color) {
        const c = String(selectedOptions.color).toLowerCase();
        if (c.includes('rgb')) delta += 20000;
    }

    return Math.max(0, base + delta);
}

function computeTotalPrice(product, selectedOptions = {}, qty = 1) {
    const q = Math.max(1, Math.min(99, parseInt(qty || 1, 10) || 1));
    return computeUnitPrice(product, selectedOptions) * q;
}

// شحن: مجاني عند 100,000 د.ع فأكثر
const FREE_SHIPPING_THRESHOLD = 100000;
const SHIPPING_FEE = 5000;

function byId(id) {
    return document.getElementById(id);
}

function findProduct(productId) {
    return products.find(p => Number(p.id) === Number(productId));
}

function isInList(list, productId) {
    return list.some(x => Number(x.productId) === Number(productId));
}

function toast(msg) {
    const el = document.createElement('div');
    el.className = 'toast';
    el.textContent = msg;
    document.body.appendChild(el);
    setTimeout(() => {
        el.style.opacity = '0';
        el.style.transform = 'translateY(6px)';
        el.style.transition = 'all .25s ease';
    }, 1800);
    setTimeout(() => el.remove(), 2200);
}

// ----------------------------
// Drawer Controls
// ----------------------------
function openCart() {
    byId('cartOverlay')?.classList.add('open');
    byId('cartDrawer')?.classList.add('open');
    document.body.style.overflow = 'hidden';
}
function closeCart() {
    byId('cartOverlay')?.classList.remove('open');
    byId('cartDrawer')?.classList.remove('open');
    document.body.style.overflow = '';
}

function openWishlist() {
    byId('wishlistOverlay')?.classList.add('open');
    byId('wishlistDrawer')?.classList.add('open');
    document.body.style.overflow = 'hidden';
}
function closeWishlist() {
    byId('wishlistOverlay')?.classList.remove('open');
    byId('wishlistDrawer')?.classList.remove('open');
    document.body.style.overflow = '';
}

function openCompare() {
    byId('compareOverlay')?.classList.add('open');
    byId('compareDrawer')?.classList.add('open');
    document.body.style.overflow = 'hidden';
}
function closeCompare() {
    byId('compareOverlay')?.classList.remove('open');
    byId('compareDrawer')?.classList.remove('open');
    document.body.style.overflow = '';
}

function openSearch() {
    window.dispatchEvent(new CustomEvent('search-open'));
}

function closeSearch() {
    window.dispatchEvent(new CustomEvent('search-close'));
}

// ESC closes all
document.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') {
        closeCart();
        closeWishlist();
        closeCompare();
        closeQuickView();
        closeSearch();
    }
});

// ----------------------------
// Cart Logic
// item shape: { productId, variantId, qty, color, size, unitPrice }
// ----------------------------
async function addToCart(productId, opts = {}) {
    const p = findProduct(productId);
    if (!p) return;

    // Get variant ID - required for DB storage
    const variantId = opts.variantId || opts.variant_id || null;

    // Use price from product API (already calculated by backend based on client type)
    const unitPrice = normalizeIQD(opts.unitPrice ?? p.price ?? 0);

    const qtyToAdd = Math.max(1, Math.min(99, parseInt(opts.qty ?? 1, 10) || 1));

    // خيارات المتغيرات (لون/مقاس/قدرة/..)
    const options = normalizeOptions(opts.options ?? {});

    // إن كان اللون مُخزن بالكود، حوّله لاسم واضح (للعرض داخل السلة)
    try {
        const colorCode = options.color;
        const colors = p?.options?.color;
        if (colorCode && Array.isArray(colors)) {
            const found = colors.find(c => String(c.code) === String(colorCode));
            if (found?.name) options.color = String(found.name);
        }
    } catch (e) { }
    // للحفاظ على واجهة الدروار القديمة: نُظهر اللون والمقاس إن وُجدا
    const color = options.color ?? (opts.color ?? null);
    const size = options.size ?? (opts.size ?? null);
    const optionsKey = stableOptionsKey(options);

    // دمج العناصر المتشابهة (نفس المنتج + نفس الخيارات)
    const existing = cart.find(i =>
        Number(i.productId) === Number(productId) &&
        String(i.optionsKey || '') === String(optionsKey || '')
    );

    if (existing) {
        existing.qty = Math.min(99, (Number(existing.qty) || 1) + qtyToAdd);
    } else {
        cart.push({
            productId: Number(productId),
            variantId: variantId ? Number(variantId) : null,
            qty: qtyToAdd,
            color,
            size,
            options,
            optionsKey,
            unitPrice,
            name: p.name || '',
            image: p.image || ''
        });
    }

    saveState(LS.cart, cart);
    updateUI(false);
    toast('تمت الإضافة إلى السلة');

    // Sync with backend API (if variant_id is available)
    if (variantId) {
        try {
            await fetch('/cart/add', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json',
                },
                body: JSON.stringify({
                    variant_id: variantId,
                    quantity: qtyToAdd,
                }),
            });
        } catch (e) {
            console.warn('Failed to sync cart with backend:', e);
        }
    }
}

function normalizeOptions(obj) {
    const out = {};
    try {
        for (const [k, v] of Object.entries(obj || {})) {
            if (v === null || typeof v === 'undefined' || String(v).trim() === '') continue;
            out[String(k)] = String(v);
        }
    } catch (e) { }
    return out;
}

function stableOptionsKey(options) {
    try {
        const keys = Object.keys(options || {}).sort();
        const stable = {};
        for (const k of keys) stable[k] = options[k];
        return JSON.stringify(stable);
    } catch (e) {
        return '';
    }
}

async function removeFromCart(index) {
    const item = cart[index];
    const variantId = item?.variantId;

    cart.splice(index, 1);
    saveState(LS.cart, cart);
    updateUI(false);

    // Sync with backend
    if (variantId) {
        try {
            await fetch('/cart/remove', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json',
                },
                body: JSON.stringify({ variant_id: variantId }),
            });
        } catch (e) {
            console.warn('Failed to remove from cart on backend:', e);
        }
    }
}

async function setCartQty(index, qty) {
    const q = Math.max(1, Math.min(99, Math.round(Number(qty) || 1)));
    if (!cart[index]) return;

    const variantId = cart[index].variantId;
    cart[index].qty = q;
    saveState(LS.cart, cart);
    updateUI(false);

    // Sync with backend
    if (variantId) {
        try {
            await fetch('/cart/update', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json',
                },
                body: JSON.stringify({ variant_id: variantId, quantity: q }),
            });
        } catch (e) {
            console.warn('Failed to update cart on backend:', e);
        }
    }
}

function cartTotals() {
    let subtotal = 0;
    for (const item of cart) {
        const unit = normalizeIQD(item.unitPrice);
        const qty = Math.max(1, Math.round(Number(item.qty) || 1));
        subtotal += unit * qty;
    }
    const isFreeShipping = subtotal >= FREE_SHIPPING_THRESHOLD;
    const shipping = isFreeShipping ? 0 : SHIPPING_FEE;
    return {
        subtotal,
        shipping,
        isFreeShipping,
        total: subtotal + shipping,
        count: cart.reduce((s, i) => s + Math.max(1, Number(i.qty) || 1), 0),
    };
}

async function clearCart() {
    cart.splice(0, cart.length);
    saveState(LS.cart, cart);
    updateUI(false);
    toast('تم إفراغ السلة');

    // Sync with backend
    try {
        await fetch('/cart/clear', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                'Accept': 'application/json',
            },
        });
    } catch (e) {
        console.warn('Failed to clear cart on backend:', e);
    }
}

// ----------------------------
// Wishlist / Compare
// ----------------------------
async function toggleWishlist(productId, variantId = null) {
    productId = Number(productId);
    const idx = wishlist.findIndex(x => Number(x.productId) === productId);
    if (idx >= 0) {
        wishlist.splice(idx, 1);
        toast('تمت الإزالة من المفضلة');
    } else {
        wishlist.push({ productId, variantId });
        toast('تمت الإضافة إلى المفضلة');
    }
    saveState(LS.wishlist, wishlist);
    updateUI(true);

    // Sync with backend API
    try {
        await fetch('/wishlist/toggle', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                'Accept': 'application/json',
            },
            body: JSON.stringify({
                product_id: productId,
                variant_id: variantId,
            }),
        });
    } catch (e) {
        console.warn('Failed to sync wishlist with backend:', e);
    }
}

async function toggleCompare(productId, variantId = null) {
    productId = Number(productId);
    const idx = compare.findIndex(x => Number(x.productId) === productId);
    if (idx >= 0) {
        compare.splice(idx, 1);
        toast('تمت الإزالة من المقارنة');
    } else {
        if (compare.length >= 4) {
            toast('المقارنة تسمح بـ 4 منتجات فقط');
            return;
        }
        compare.push({ productId, variantId });
        toast('تمت الإضافة للمقارنة');
    }
    saveState(LS.compare, compare);
    updateUI(true);

    // Sync with backend API
    try {
        await fetch('/compare/toggle', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                'Accept': 'application/json',
            },
            body: JSON.stringify({
                product_id: productId,
                variant_id: variantId,
            }),
        });
    } catch (e) {
        console.warn('Failed to sync compare with backend:', e);
    }
}

function shareProduct(productId) {
    const p = findProduct(productId);
    if (!p) return;

    const url = new URL(`product.html?id=${p.id}`, window.location.href).href;
    const text = `${p.name} — ${formatIQD(p.price)}`;

    // Native share (mobile / supported browsers)
    if (navigator.share) {
        navigator.share({ title: p.name, text, url }).catch(() => { });
        return;
    }

    // Fallback: copy link
    if (navigator.clipboard?.writeText) {
        navigator.clipboard.writeText(url).then(() => toast('تم نسخ رابط المنتج')).catch(() => toast('تعذر نسخ الرابط'));
        return;
    }

    // Last resort
    prompt('انسخ الرابط:', url);
}

async function removeFromWishlist(index) {
    const item = wishlist[index];
    const productId = item?.productId;

    wishlist.splice(index, 1);
    saveState(LS.wishlist, wishlist);
    updateUI(true);

    // Sync with backend
    if (productId) {
        try {
            await fetch('/wishlist/remove', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json',
                },
                body: JSON.stringify({ product_id: productId }),
            });
        } catch (e) {
            console.warn('Failed to remove from wishlist on backend:', e);
        }
    }
}

async function removeFromCompare(index) {
    const item = compare[index];
    const productId = item?.productId;

    compare.splice(index, 1);
    saveState(LS.compare, compare);
    updateUI(true);

    // Sync with backend
    if (productId) {
        try {
            await fetch('/compare/remove', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json',
                },
                body: JSON.stringify({ product_id: productId }),
            });
        } catch (e) {
            console.warn('Failed to remove from compare on backend:', e);
        }
    }
}

// ----------------------------
// Quick View (يدعم المتغيرات: لون/قياس/قدرة/..)
// ----------------------------
function getProductOptionGroups(p) {
    const opts = p?.options && typeof p.options === 'object' ? p.options : {};
    const groups = [];

    // لون (أزرار دائرية)
    if (Array.isArray(opts.color) && opts.color.length) {
        groups.push({ key: 'color', label: 'اللون', type: 'color', items: opts.color });
    }

    // قياس (أزرار)
    if (Array.isArray(opts.size) && opts.size.length) {
        groups.push({ key: 'size', label: 'المقاس', type: 'chips', items: opts.size });
    }

    // قدرة/أمبير/... (أزرار)
    if (Array.isArray(opts.power) && opts.power.length) {
        groups.push({ key: 'power', label: 'القدرة', type: 'chips', items: opts.power });
    }

    // أي مفاتيح أخرى ديناميكية (احتياط)
    for (const [k, v] of Object.entries(opts)) {
        if (k === 'color' || k === 'size' || k === 'power') continue;
        if (Array.isArray(v) && v.length) {
            groups.push({ key: k, label: k, type: 'chips', items: v });
        }
    }

    return groups;
}

function openQuickView(productId) {
    const p = findProduct(productId);
    if (!p) return;

    quickViewSelection.productId = Number(productId);
    quickViewSelection.options = {};
    quickViewSelection.qty = 1;

    // default selections
    for (const g of getProductOptionGroups(p)) {
        if (g.key === 'color') {
            quickViewSelection.options[g.key] = g.items?.[0]?.code ?? null;
        } else {
            quickViewSelection.options[g.key] = g.items?.[0] ?? null;
        }
    }

    const modal = byId('quickViewModal');
    const content = byId('quickViewContent');
    if (!modal || !content) return;

    content.innerHTML = renderQuickViewHtml(p);
    modal.classList.add('open');
    document.body.style.overflow = 'hidden';


    const updateQv = () => {
        const unit = computeUnitPrice(p, quickViewSelection.options);
        const total = computeTotalPrice(p, quickViewSelection.options, quickViewSelection.qty);

        const unitEl = content.querySelector('#qvUnitPrice');
        const totalEl = content.querySelector('#qvTotal');
        const totalInlineEl = content.querySelector('#qvTotalInline');
        const qtyEl = content.querySelector('#qvQty');

        if (unitEl) unitEl.textContent = formatIQD(unit);
        if (totalEl) totalEl.textContent = formatIQD(total);
        if (totalInlineEl) totalInlineEl.textContent = formatIQD(total);
        if (qtyEl) qtyEl.textContent = String(quickViewSelection.qty);
    };

    updateQv();


    // Bind option clicks (general)
    content.querySelectorAll('[data-qv-opt]').forEach(btn => {
        btn.addEventListener('click', () => {
            const key = btn.getAttribute('data-qv-key');
            const val = btn.getAttribute('data-qv-opt');
            quickViewSelection.options[key] = val;

            // active state for this group
            content.querySelectorAll(`[data-qv-key="${CSS.escape(key)}"]`).forEach(x => x.classList.remove('active'));
            btn.classList.add('active');
            updateQv();
        });
    });


    content.querySelectorAll('[data-qv-qty]').forEach(b => {
        b.addEventListener('click', () => {
            const step = b.getAttribute('data-qv-qty');
            if (step === '+1') quickViewSelection.qty = Math.min(99, (quickViewSelection.qty || 1) + 1);
            if (step === '-1') quickViewSelection.qty = Math.max(1, (quickViewSelection.qty || 1) - 1);
            updateQv();
        });
    });

    // Toggle active state for wishlist/compare buttons inside quick view
    const qvWish = content.querySelector('[data-wish]');
    const qvCmp = content.querySelector('[data-compare]');
    qvWish?.addEventListener('click', () => qvWish.classList.toggle('active'));
    qvCmp?.addEventListener('click', () => qvCmp.classList.toggle('active'));

    const addBtn = content.querySelector('[data-qv-add]');

    addBtn?.addEventListener('click', () => {
        const unitPrice = computeUnitPrice(p, quickViewSelection.options);
        addToCart(p.id, {
            variantId: p.variant_id,
            unitPrice,
            qty: quickViewSelection.qty,
            options: quickViewSelection.options,
        });
        closeQuickView();
        openCart();
    });
}

function closeQuickView() {
    byId('quickViewModal')?.classList.remove('open');
    document.body.style.overflow = '';
}

function renderQuickViewHtml(p) {
    const unit = computeUnitPrice(p, quickViewSelection.options);
    const total = computeTotalPrice(p, quickViewSelection.options, quickViewSelection.qty);
    const price = formatIQD(unit);
    const totalStr = formatIQD(total);
    const old = p.originalPrice ? `<del>${formatIQD(p.originalPrice)}</del>` : '';

    const groupsHtml = getProductOptionGroups(p).map((g) => {
        if (g.type === 'color') {
            const colors = g.items.map((c, i) => {
                const val = String(c.code);
                const active = (quickViewSelection.options[g.key] ? val === String(quickViewSelection.options[g.key]) : i === 0) ? 'active' : '';
                return `<button class="color-option ${active}" style="background:${c.value}" data-qv-key="${g.key}" data-qv-opt="${val}" title="${c.name}"></button>`;
            }).join('');
            return `
        <div class="mt-6">
          <div class="text-gray-300 font-bold mb-3">${g.label}</div>
          <div class="flex flex-wrap gap-3">${colors}</div>
        </div>`;
        }

        // chips (size/power/...)
        const chips = g.items.map((s, i) => {
            const val = String(s);
            const active = (quickViewSelection.options[g.key] ? val === String(quickViewSelection.options[g.key]) : i === 0) ? 'active' : '';
            return `<button class="size-option ${active}" data-qv-key="${g.key}" data-qv-opt="${val}">${val}</button>`;
        }).join('');

        return `
      <div class="mt-6">
        <div class="text-gray-300 font-bold mb-3">${g.label}</div>
        <div class="flex flex-wrap gap-3">${chips}</div>
      </div>`;
    }).join('');

    return `
    <div class="p-6 md:p-8">
      <div class="flex items-center justify-between mb-6">
        <h3 class="cairo text-2xl md:text-3xl font-black text-white">عرض سريع</h3>
        <button onclick="closeQuickView()" class="p-2 hover:bg-white/5 rounded-full transition-colors">
          <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
          </svg>
        </button>
      </div>

      <div class="grid md:grid-cols-2 gap-8">
        <div class="glass rounded-3xl overflow-hidden">
          <div class="aspect-[4/5] overflow-hidden">
            <img src="${p.image}" alt="${p.name}" class="w-full h-full object-cover">
          </div>
        </div>

        <div>
          <h4 class="cairo text-3xl font-black text-white leading-tight">${p.name}</h4>


          <div class="mt-4">
            <div class="flex items-end gap-3">
              <div class="cairo text-3xl font-black" style="color: var(--primary-gold)" id="qvUnitPrice">${price}</div>
              ${old}
            </div>
          </div>


          ${groupsHtml}

          <div class="mt-8">
            <div class="qv-actions-row">
              <!-- Qty pill ( + / number / - ) -->
              <div class="qv-qty-pill">
                <button type="button" data-qv-qty="+1" class="qv-step" aria-label="زيادة">+</button>
                <span id="qvQty" class="qv-qty-num">1</span>
                <button type="button" data-qv-qty="-1" class="qv-step" aria-label="نقص">-</button>
              </div>
              <!-- Icons: Share / Compare / Wishlist -->
              <div class="qv-icons">
                <button class="qv-icon-btn" title="مشاركة" data-share="${p.id}" aria-label="مشاركة">
                  <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 8a3 3 0 10-6 0 3 3 0 006 0z"></path>
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M18 21a3 3 0 10-6 0 3 3 0 006 0z"></path>
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 21a3 3 0 10-6 0 3 3 0 006 0z"></path>
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8.59 13.51l6.83 3.98"></path>
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15.42 6.51L8.59 10.49"></path>
                  </svg>
                </button>

                <button class="qv-icon-btn ${isInList(compare, p.id) ? 'active' : ''}" title="مقارنة" data-compare="${p.id}" aria-label="مقارنة">
                  <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"></path>
                  </svg>
                </button>

                <button class="qv-icon-btn ${isInList(wishlist, p.id) ? 'active' : ''}" title="مفضلة" data-wish="${p.id}" aria-label="مفضلة">
                  <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"></path>
                  </svg>
                </button>
              </div>
            </div>

            <button data-qv-add class="qv-add-btn gold-gradient text-black">
              <span class="qv-add-label">إضافة إلى السلة</span>
              <span class="qv-add-sep">-</span>
              <span class="qv-add-price" id="qvTotalInline">${totalStr}</span>
              <span class="sr-only" id="qvTotal">${totalStr}</span>
            </button>
          </div></div>

          <p class="mt-6 text-gray-400">
            تم حذف التفاصيل داخل الكارت (سيتم تعويضها بأيقونة العرض السريع).
          </p>
        </div>
      </div>
    </div>
  `;
}

// ----------------------------
// Render Products Grid (Product Card v2)
// ----------------------------
function renderProducts() {
    const grid = byId('productsGrid');
    if (!grid) return;

    const list = products.filter(p => currentFilter === 'all' || String(p.category) === String(currentFilter));

    if (!list.length) {
        grid.innerHTML = `<div class="col-span-full text-center glass rounded-3xl p-10 text-gray-300">
        لا توجد منتجات ضمن هذا القسم حالياً.
      </div>`;
        return;
    }

    grid.innerHTML = list.map(p => {
        const badge = p.badge ? `<span class="badge" style="background: rgba(212,163,86,.18); border:1px solid rgba(212,163,86,.30)">${p.badge}</span>` : '';
        const old = p.originalPrice ? `<del>${formatIQD(p.originalPrice)}</del>` : '';
        const wishActive = isInList(wishlist, p.id) ? 'active' : '';
        const compareActive = isInList(compare, p.id) ? 'active' : '';

        return `
      <div class="product-card v2 group relative">
        <a class="card-hit" href="product.html?id=${p.id}" aria-label="فتح ${p.name}"></a>
        <div class="media">
          ${badge}
          <img src="${p.image}" alt="${p.name}">
          <div class="hover-actions">
            <button class="action-btn" title="عرض سريع" data-qv="${p.id}">
              <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.477 0 8.268 2.943 9.542 7-1.274 4.057-5.065 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path>
              </svg>
            </button>
            <button class="action-btn ${wishActive}" title="مفضلة" data-wish="${p.id}">
              <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"></path>
              </svg>
            </button>
            <button class="action-btn ${compareActive}" title="مقارنة" data-compare="${p.id}">
              <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"></path>
              </svg>
            </button>
          </div>
        </div>

        <div class="content">
          <div class="title">${p.name}</div>
          <div class="price-row">
            <div class="price">${old} ${formatIQD(p.price)}</div>
          </div>

          <button class="add-btn gold-gradient text-black" data-add="${p.id}">
            إضافة إلى السلة
          </button>
        </div>
      </div>
    `;
    }).join('');

    // Bind actions
    grid.querySelectorAll('[data-add]').forEach(btn => {
        btn.addEventListener('click', () => addToCart(btn.getAttribute('data-add')));
    });

    grid.querySelectorAll('[data-qv]').forEach(btn => {
        btn.addEventListener('click', () => openQuickView(btn.getAttribute('data-qv')));
    });

    grid.querySelectorAll('[data-wish]').forEach(btn => {
        btn.addEventListener('click', () => toggleWishlist(btn.getAttribute('data-wish')));
    });

    grid.querySelectorAll('[data-compare]').forEach(btn => {
        btn.addEventListener('click', () => toggleCompare(btn.getAttribute('data-compare')));
    });
}

function renderGridInto(targetId, list) {
    const grid = byId(targetId);
    if (!grid) return;

    if (!list.length) {
        grid.innerHTML = `<div class="col-span-full text-center glass rounded-3xl p-10 text-gray-300">لا توجد منتجات حالياً.</div>`;
        return;
    }

    grid.innerHTML = list.map(p => {
        const badge = p.badge ? `<span class="badge" style="background: rgba(212,163,86,.18); border:1px solid rgba(212,163,86,.30)">${p.badge}</span>` : '';
        const old = p.originalPrice ? `<del>${formatIQD(p.originalPrice)}</del>` : '';
        const wishActive = isInList(wishlist, p.id) ? 'active' : '';
        const compareActive = isInList(compare, p.id) ? 'active' : '';

        return `
      <div class="product-card v2 group relative">
        <a class="card-hit" href="product.html?id=${p.id}" aria-label="فتح ${p.name}"></a>
        <div class="media">
          ${badge}
          <img src="${p.image}" alt="${p.name}">
          <div class="hover-actions">
            <button class="action-btn" title="عرض سريع" data-qv="${p.id}">
              <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path>
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.477 0 8.268 2.943 9.542 7-1.274 4.057-5.065 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path>
              </svg>
            </button>
            <button class="action-btn ${wishActive}" title="مفضلة" data-wish="${p.id}">
              <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"></path>
              </svg>
            </button>
            <button class="action-btn ${compareActive}" title="مقارنة" data-compare="${p.id}">
              <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"></path>
              </svg>
            </button>
          </div>
        </div>

        <div class="content">
          <div class="title">${p.name}</div>
          <div class="price-row">
            <div class="price">${old} ${formatIQD(p.price)}</div>
          </div>

          <button class="add-btn gold-gradient text-black" data-add="${p.id}">
            إضافة إلى السلة
          </button>
        </div>
      </div>
    `;
    }).join('');

    // Bind actions inside this grid
    grid.querySelectorAll('[data-add]').forEach(btn => btn.addEventListener('click', () => addToCart(btn.getAttribute('data-add'))));
    grid.querySelectorAll('[data-qv]').forEach(btn => btn.addEventListener('click', () => openQuickView(btn.getAttribute('data-qv'))));
    grid.querySelectorAll('[data-wish]').forEach(btn => btn.addEventListener('click', () => toggleWishlist(btn.getAttribute('data-wish'))));
    grid.querySelectorAll('[data-compare]').forEach(btn => btn.addEventListener('click', () => toggleCompare(btn.getAttribute('data-compare'))));
}

function renderExtraSections() {
    // وصل حديثاً: أحدث IDs
    const arrivals = [...products].sort((a, b) => Number(b.id) - Number(a.id)).slice(0, 4);

    // الأكثر مبيعاً: badge يحتوي "الأكثر" أو منتجات LED الليلية
    const best = products.filter(p => String(p.badge || '').includes('الأكثر')).slice(0, 4);

    // عروض مميزة: المنتجات التي لديها سعر قديم أو badge خصم/عرض
    const deals = products.filter(p => Boolean(p.originalPrice) || String(p.badge || '').includes('خصم') || String(p.badge || '').includes('عرض')).slice(0, 4);

    renderGridInto('arrivalsGrid', arrivals);
    renderGridInto('bestGrid', best);
    renderGridInto('dealsGrid', deals);
}

// ----------------------------
// Render Drawers
// ----------------------------
function renderCartDrawer() {
    const itemsWrap = byId('cartItems');
    if (!itemsWrap) return;

    if (!cart.length) {
        itemsWrap.innerHTML = `
      <div class="text-center py-12 text-gray-400">
        <svg class="w-24 h-24 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4M7 13L5.4 5M7 13l-2.293 2.293c-.63.63-.184 1.707.707 1.707H17m0 0a2 2 0 100 4 2 2 0 000-4zm-8 2a2 2 0 11-4 0 2 2 0 014 0z"/>
        </svg>
        <p class="text-lg">سلة التسوق فارغة</p>
      </div>`;
        return;
    }

    itemsWrap.innerHTML = cart.map((item, idx) => {
        const p = findProduct(item.productId);
        if (!p) return '';
        const qty = Math.max(1, Number(item.qty) || 1);
        const unit = normalizeIQD(item.unitPrice ?? p.price);
        const line = unit * qty;

        const optsObj = item.options && typeof item.options === 'object' ? item.options : {};
        const optsParts = [];
        for (const [k, v] of Object.entries(optsObj)) {
            const label = (k === 'color') ? 'لون' : (k === 'size') ? 'مقاس' : (k === 'power') ? 'قدرة' : k;
            optsParts.push(`${label}: ${v}`);
        }
        // fallback legacy
        if (!optsParts.length) {
            if (item.color) optsParts.push(`لون: ${item.color}`);
            if (item.size) optsParts.push(`مقاس: ${item.size}`);
        }
        const opts = optsParts.join(' • ');

        return `
      <div class="cart-item-card p-4 mb-4">
        <div class="flex gap-4">
          <img src="${p.image}" alt="${p.name}" class="w-20 h-20 rounded-2xl object-cover">
          <div class="flex-1 min-w-0">
            <div class="flex items-start justify-between gap-3">
              <div class="min-w-0">
                <div class="font-black text-white truncate">${p.name}</div>
                ${opts ? `<div class="text-sm text-gray-400 mt-1 truncate">${opts}</div>` : ''}
              </div>
              <button class="text-gray-400 hover:text-white shrink-0" data-cart-remove="${idx}" title="حذف">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6M1 7h22m-5-3H6a2 2 0 00-2 2v1h16V6a2 2 0 00-2-2z"/>
                </svg>
              </button>
            </div>

            <div class="mt-2 flex items-center justify-between">
              <div class="font-black" style="color: var(--primary-gold)">${formatIQD(line)}</div>
              <div class="flex items-center gap-2">
                <button class="qty-btn" data-qty-dec="${idx}">-</button>
                <input class="qty-input" value="${qty}" data-qty-input="${idx}">
                <button class="qty-btn" data-qty-inc="${idx}">+</button>
              </div>
            </div>
          </div>
        </div>
      </div>
    `;
    }).join('');

    // bind cart controls
    itemsWrap.querySelectorAll('[data-cart-remove]').forEach(btn => {
        btn.addEventListener('click', () => removeFromCart(Number(btn.getAttribute('data-cart-remove'))));
    });
    itemsWrap.querySelectorAll('[data-qty-dec]').forEach(btn => {
        btn.addEventListener('click', () => {
            const i = Number(btn.getAttribute('data-qty-dec'));
            setCartQty(i, (cart[i]?.qty || 1) - 1);
        });
    });
    itemsWrap.querySelectorAll('[data-qty-inc]').forEach(btn => {
        btn.addEventListener('click', () => {
            const i = Number(btn.getAttribute('data-qty-inc'));
            setCartQty(i, (cart[i]?.qty || 1) + 1);
        });
    });
    itemsWrap.querySelectorAll('[data-qty-input]').forEach(inp => {
        inp.addEventListener('change', () => {
            const i = Number(inp.getAttribute('data-qty-input'));
            setCartQty(i, inp.value);
        });
    });
}

function renderWishlistDrawer() {
    const wrap = byId('wishlistItems');
    if (!wrap) return;

    if (!wishlist.length) {
        wrap.innerHTML = `
      <div class="text-center py-12 text-gray-400">
        <svg class="w-24 h-24 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/>
        </svg>
        <p class="text-lg">لا توجد منتجات في المفضلة</p>
      </div>`;
        return;
    }

    wrap.innerHTML = wishlist.map((w, idx) => {
        const p = findProduct(w.productId);
        if (!p) return '';
        return `
      <div class="glass rounded-2xl p-4 mb-4">
        <div class="flex gap-4">
          <img src="${p.image}" alt="${p.name}" class="w-20 h-20 rounded-xl object-cover">
          <div class="flex-1">
            <div class="font-black text-white">${p.name}</div>
            <div class="mt-2 flex items-center justify-between">
              <div class="font-black" style="color: var(--primary-gold)">${formatIQD(p.price)}</div>
              <button class="text-gray-400 hover:text-white" data-wish-remove="${idx}">حذف</button>
            </div>
            <button class="mt-3 w-full gold-gradient text-black py-3 rounded-full font-black" data-wish-add="${p.id}">إضافة إلى السلة</button>
          </div>
        </div>
      </div>
    `;
    }).join('');

    wrap.querySelectorAll('[data-wish-remove]').forEach(btn => {
        btn.addEventListener('click', () => removeFromWishlist(Number(btn.getAttribute('data-wish-remove'))));
    });
    wrap.querySelectorAll('[data-wish-add]').forEach(btn => {
        btn.addEventListener('click', () => addToCart(btn.getAttribute('data-wish-add')));
    });
}

function renderCompareDrawer() {
    const wrap = byId('compareItems');
    if (!wrap) return;

    if (!compare.length) {
        wrap.innerHTML = `
      <div class="text-center py-12 text-gray-400">
        <svg class="w-24 h-24 mx-auto mb-4 opacity-50" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
        </svg>
        <p class="text-lg">لا توجد منتجات للمقارنة</p>
      </div>`;
        return;
    }

    wrap.innerHTML = compare.map((c, idx) => {
        const p = findProduct(c.productId);
        if (!p) return '';
        return `
      <div class="glass rounded-2xl p-4 mb-4">
        <div class="flex gap-4">
          <img src="${p.image}" alt="${p.name}" class="w-20 h-20 rounded-xl object-cover">
          <div class="flex-1">
            <div class="font-black text-white">${p.name}</div>
            <div class="mt-2 flex items-center justify-between">
              <div class="font-black" style="color: var(--primary-gold)">${formatIQD(p.price)}</div>
              <button class="text-gray-400 hover:text-white" data-compare-remove="${idx}">حذف</button>
            </div>
          </div>
        </div>
      </div>
    `;
    }).join('');

    wrap.querySelectorAll('[data-compare-remove]').forEach(btn => {
        btn.addEventListener('click', () => removeFromCompare(Number(btn.getAttribute('data-compare-remove'))));
    });
}

// ----------------------------
// Counts + Totals + Init
// ----------------------------
function updateCounts() {
    const totals = cartTotals();

    // Header badges
    const cartCount = byId('cartCount');
    const wishCount = byId('wishlistCount');
    const compareCount = byId('compareCount');

    const mobileCartCount = byId('mobileCartCount');
    const mobileWishCount = byId('mobileWishlistCount');
    if (cartCount) cartCount.textContent = String(totals.count);
    if (wishCount) wishCount.textContent = String(wishlist.length);
    if (compareCount) compareCount.textContent = String(compare.length);

    // Drawer headings counts
    const cartItemsCount = byId('cartItemsCount');
    const wishlistItemsCount = byId('wishlistItemsCount');
    const compareItemsCount = byId('compareItemsCount');

    if (cartItemsCount) cartItemsCount.textContent = String(totals.count);
    if (wishlistItemsCount) wishlistItemsCount.textContent = String(wishlist.length);
    if (compareItemsCount) compareItemsCount.textContent = String(compare.length);

    // Totals
    const subtotalEl = byId('subtotal');
    const shippingFeeEl = byId('shippingFee');
    const totalEl = byId('total');
    const progressBox = byId('shippingProgress');
    const progressFill = byId('shippingProgressFill');
    const progressText = byId('shippingProgressText');
    if (subtotalEl) subtotalEl.textContent = formatIQD(totals.subtotal);
    if (shippingFeeEl) shippingFeeEl.textContent = totals.isFreeShipping ? 'مجاني' : formatIQD(totals.shipping);
    if (totalEl) totalEl.textContent = formatIQD(totals.total);

    // Progress visual state
    if (progressBox) {
        progressBox.classList.toggle('is-free', !!totals.isFreeShipping);
    }

    // Free shipping progress
    if (progressFill) {
        const pct = Math.max(0, Math.min(100, (totals.subtotal / FREE_SHIPPING_THRESHOLD) * 100));
        progressFill.style.width = pct.toFixed(0) + '%';
    }
    if (progressText) {
        if (totals.isFreeShipping) {
            progressText.textContent = 'مبروك! حصلت على توصيل مجاني.';
        } else {
            const remain = Math.max(0, FREE_SHIPPING_THRESHOLD - totals.subtotal);
            progressText.textContent = `تبقّى ${formatIQD(remain)} للحصول على توصيل مجاني.`;
        }
    }
}

function updateUI(rerenderProducts = true) {
    if (rerenderProducts) {
        renderProducts();
        renderExtraSections();
    }
    renderCartDrawer();
    renderWishlistDrawer();
    renderCompareDrawer();
    updateCounts();
}

// Newsletter form
document.addEventListener('submit', (e) => {
    if (e.target && e.target.id === 'newsletterForm') {
        e.preventDefault();
        toast('تم استلام اشتراكك ✅');
        e.target.reset();
    }
});

// Global actions (works across all pages via data-* attributes)
document.addEventListener('click', (e) => {
    const btn = e.target?.closest?.('[data-add],[data-qv],[data-wish],[data-compare],[data-share]');
    if (!btn) return;

    // Prevent card-hit navigation
    e.preventDefault();
    e.stopPropagation();

    const addId = btn.getAttribute('data-add');
    const variantId = btn.getAttribute('data-variant');
    const qvId = btn.getAttribute('data-qv');
    const wishId = btn.getAttribute('data-wish');
    const cmpId = btn.getAttribute('data-compare');
    const shareId = btn.getAttribute('data-share');

    if (addId) {
        const p = findProduct(addId);
        return addToCart(addId, {
            variantId: variantId || p?.variant_id,
            unitPrice: p?.price
        });
    }
    if (qvId) return openQuickView(qvId);
    if (wishId) return toggleWishlist(wishId);
    if (cmpId) return toggleCompare(cmpId);
    if (shareId) return shareProduct(shareId);
});



// Initial
document.addEventListener('DOMContentLoaded', async () => {
    // Load products from API
    try {
        const result = await fetchProducts({ per_page: 100 });
        if (result.products.length > 0) {
            products = result.products;
        } else {
            // Fallback to mock products if API returns empty
            products = mockProducts;
        }
    } catch (error) {
        console.error('Failed to load products from API, using mock data:', error);
        products = mockProducts;
    }

    // Filters
    document.querySelectorAll('.filter-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const f = btn.getAttribute('data-filter') || 'all';
            currentFilter = f;
            document.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
            btn.classList.add('active');
            renderProducts();
        });
    });

    updateUI(true);
});

// ===================== Expose Functions Globally =====================
// Make all drawer and action functions available on window for Blade templates
window.openCart = openCart;
window.closeCart = closeCart;
window.openWishlist = openWishlist;
window.closeWishlist = closeWishlist;
window.openCompare = openCompare;
window.closeCompare = closeCompare;
window.openSearch = openSearch;
window.closeSearch = closeSearch;
window.openQuickView = openQuickView;
window.closeQuickView = closeQuickView;
window.addToCart = addToCart;
window.toggleWishlist = toggleWishlist;
window.toggleCompare = toggleCompare;
window.shareProduct = shareProduct;
window.removeFromCart = removeFromCart;
window.setCartQty = setCartQty;
window.clearCart = clearCart;
window.removeFromWishlist = removeFromWishlist;
window.removeFromCompare = removeFromCompare;
window.updateUI = updateUI;
window.formatIQD = formatIQD;
window.normalizeIQD = normalizeIQD;
window.findProduct = findProduct;
window.products = products;
window.fetchProducts = fetchProducts;
window.fetchCategories = fetchCategories;
window.fetchBrands = fetchBrands;
window.toast = toast;


// ===================== Header Search (Popup) =====================
(function initHeaderSearch() {
    const modal = byId('searchModal');
    const overlay = byId('searchOverlay');
    const input = byId('searchInput');
    const results = byId('searchResults');
    const openBtn = byId('openSearchBtn');
    const closeBtn = byId('closeSearchBtn');

    if (!modal || !input || !results || !openBtn || !closeBtn || !overlay) return;

    const open = () => {
        modal.classList.remove('hidden');
        setTimeout(() => input.focus(), 50);
    };
    const close = () => {
        modal.classList.add('hidden');
        input.value = '';
        results.innerHTML = '';
    };

    openBtn.addEventListener('click', open);
    closeBtn.addEventListener('click', close);
    overlay.addEventListener('click', close);
    window.addEventListener('keydown', (e) => {
        if (e.key === 'Escape' && !modal.classList.contains('hidden')) close();
    });

    const render = (items) => {
        if (!items.length) {
            results.innerHTML = '<div class="p-6 text-gray-400">لا توجد نتائج.</div>';
            return;
        }
        results.innerHTML = items.map(p => `
      <a href="product.html?id=${p.id}" class="flex items-center gap-3 p-3 rounded-2xl hover:bg-white/5 transition">
        <img src="${p.image}" class="w-14 h-14 rounded-xl object-cover" alt="${escapeHtml(p.name)}">
        <div class="flex-1">
          <div class="text-white font-black">${escapeHtml(p.name)}</div>
          <div class="text-sm text-gray-400">${formatIQD(p.price)}</div>
        </div>
        <span class="text-gray-500">&larr;</span>
      </a>
    `).join('');
    };

    input.addEventListener('input', () => {
        const q = String(input.value || '').trim().toLowerCase();
        if (!q) { results.innerHTML = ''; return; }
        const items = products.filter(p => (p.name || '').toLowerCase().includes(q) || (p.description || '').toLowerCase().includes(q)).slice(0, 8);
        render(items);
    });
})();


// ===================== Simple Auto Carousel (5s) =====================
function autoCarousel(container, intervalMs = 5000) {
    if (!container) return;
    const el = container;
    // Ensure horizontal scroll behavior
    el.style.scrollBehavior = 'smooth';
    let timer = null;

    const tick = () => {
        const max = el.scrollWidth - el.clientWidth;
        if (max <= 0) return;
        const next = el.scrollLeft + el.clientWidth;
        el.scrollLeft = next >= max - 5 ? 0 : next;
    };

    const start = () => {
        if (timer) return;
        timer = setInterval(tick, intervalMs);
    };
    const stop = () => {
        if (!timer) return;
        clearInterval(timer);
        timer = null;
    };

    el.addEventListener('mouseenter', stop);
    el.addEventListener('mouseleave', start);
    start();
}


// ===================== Active Navbar Link =====================
(function initActiveNav() {
    const file = (location.pathname.split('/').pop() || 'index.html').toLowerCase();
    document.querySelectorAll('nav a[href]').forEach(a => {
        const href = (a.getAttribute('href') || '').toLowerCase();
        if (!href || href.startsWith('#') || href.startsWith('http')) return;
        if (href === file) {
            a.classList.add('text-white');
            a.style.color = 'var(--primary-gold)';
        }
    });
})();


function initMobileBottomNav() {
    const nav = document.querySelector('[data-mobile-nav]');
    if (!nav) return;

    const path = (location.pathname.split('/').pop() || 'index.html').toLowerCase();

    // Active mapping
    const isHome = path.endsWith('index.html') || path === '';
    const isCategories = path.endsWith('categories.html');
    const isProduct = path.endsWith('product.html');

    nav.querySelectorAll('.mobile-nav__item').forEach(el => el.classList.remove('is-active'));

    const setActive = (key) => {
        const el = nav.querySelector(`[data-nav="${key}"]`);
        if (el) el.classList.add('is-active');
    };

    if (isHome) setActive('home');
    else if (isCategories) setActive('store');
    else if (isProduct) setActive('store');
    else setActive('home');

    // Actions
    const cartBtn = nav.querySelector('[data-nav="cart"]');
    const wishBtn = nav.querySelector('[data-nav="wishlist"]');
    const searchBtn = nav.querySelector('[data-nav="search"]');

    if (cartBtn) cartBtn.addEventListener('click', () => {
        if (typeof openCart === 'function') openCart();
        else location.href = 'index.html';
    });

    if (wishBtn) wishBtn.addEventListener('click', () => {
        if (typeof openWishlist === 'function') openWishlist();
        else location.href = 'works.html';
    });

    if (searchBtn) searchBtn.addEventListener('click', () => {
        // Try focus any search input if exists; fallback toast
        const input = document.querySelector('input[type="search"], input[data-site-search], #siteSearch');
        if (input) {
            input.focus();
            input.scrollIntoView({ behavior: 'smooth', block: 'center' });
        } else if (typeof toast === 'function') {
            toast('البحث سيتم إضافته قريباً');
        }
    });
}


document.addEventListener('DOMContentLoaded', () => { initMobileBottomNav(); });
