<?php

namespace Database\Seeders;

use App\Models\City;
use App\Models\User;
use App\Models\Branch;
use App\Models\Warehouse;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class WarehouseSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('🏗️ بدء إنشاء المستودعات...');

        // الحصول على أول مدينة ومستخدم (يمكن تعديلها حسب الحاجة)
        $defaultCity = City::first();
        $defaultUser = User::first();

        if (! $defaultCity) {
            $this->command->error('❌ لا توجد مدن في قاعدة البيانات. يرجى إنشاء مدينة أولاً.');
            return;
        }

        if (! $defaultUser) {
            $this->command->error('❌ لا يوجد مستخدمين في قاعدة البيانات. يرجى إنشاء مستخدم أولاً.');
            return;
        }

        // الحصول على جميع الفروع
        $branches = Branch::all();

        if ($branches->isEmpty()) {
            $this->command->error('❌ لا توجد فروع في قاعدة البيانات. يرجى إنشاء فرع أولاً.');
            return;
        }

        $this->command->info("📍 تم العثور على {$branches->count()} فرع");

        // تعريف المستودعات (عامة يمكن ربطها بعد ذلك كما تشاء)
        $warehousesData = [
            [
                'name'      => 'المستودع الرئيسي',
                'city_id'   => $defaultCity->id,
                'user_id'   => $defaultUser->id,
                'is_active' => true,
            ],
            [
                'name'      => 'المستودع الفرعي',
                'city_id'   => $defaultCity->id,
                'user_id'   => $defaultUser->id,
                'is_active' => true,
            ],
            [
                'name'      => 'المستودع الاحتياطي',
                'city_id'   => $defaultCity->id,
                'user_id'   => $defaultUser->id,
                'is_active' => true,
            ],
        ];

        DB::beginTransaction();

        try {
            $createdWarehouses = [];

            // إنشاء المستودعات (أو استرجاعها إن كانت موجودة بنفس الاسم)
            foreach ($warehousesData as $warehouseData) {
                $warehouse = Warehouse::firstOrCreate(
                    ['name' => $warehouseData['name']],
                    $warehouseData
                );

                $createdWarehouses[] = $warehouse;

                $this->command->info("✅ تم التأكد من وجود المستودع: {$warehouse->name}");
            }

            // نعتبر أن "المستودع الرئيسي" هو الافتراضي لكل فرع
            $defaultWarehouse = collect($createdWarehouses)
                ->first(fn(Warehouse $w) => $w->name === 'المستودع الرئيسي')
                ?? $createdWarehouses[0];

            if (! $defaultWarehouse) {
                $this->command->error('❌ لم يتم العثور على أي مستودع لاستخدامه كمستودع افتراضي.');
                DB::rollBack();
                return;
            }

            $now = now();

            // ربط مستودع واحد (افتراضي) مع كل فرع
            foreach ($branches as $branch) {
                $this->command->info("🔗 تعيين المستودع الافتراضي للفرع: {$branch->name}");

                // إزالة أي ربط سابق كمستودعات افتراضية لنفس الفرع (احتياطًا)
                DB::table('branch_warehouse')
                    ->where('branch_id', $branch->id)
                    ->update(['is_primary' => false]);

                // ربط المستودع الافتراضي مع الفرع كـ is_primary = true
                DB::table('branch_warehouse')->updateOrInsert(
                    [
                        'branch_id'    => $branch->id,
                        'warehouse_id' => $defaultWarehouse->id,
                    ],
                    [
                        'is_primary'   => true,
                        'assigned_at'  => $now,
                        'unassigned_at' => null,
                        'created_at'   => $now,
                        'updated_at'   => $now,
                    ]
                );

                $this->command->info("   ➤ المستودع الافتراضي: {$defaultWarehouse->name}");
            }

            DB::commit();

            $this->command->info('');
            $this->command->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
            $this->command->info('✅ تم إنشاء المستودعات وتعيين مستودع افتراضي لكل فرع بنجاح!');
            $this->command->info('📦 عدد المستودعات المعرفة: ' . count($createdWarehouses));
            $this->command->info("🏢 عدد الفروع التي لها مستودع افتراضي: {$branches->count()}");
            $this->command->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        } catch (\Exception $e) {
            DB::rollBack();
            $this->command->error('❌ حدث خطأ أثناء إنشاء/ربط المستودعات: ' . $e->getMessage());
            throw $e;
        }
    }
}
