<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\{Role, Permission};
use Spatie\Permission\PermissionRegistrar;

class RolesWithPermissionsSeeder extends Seeder
{
    public function run(): void
    {
        // 🧹 نظّف كاش الصلاحيات
        app(PermissionRegistrar::class)->forgetCachedPermissions();

        // 1) الأدوار
        $roleNames = [
            'admin',
            'accounting_manager',
            'branch_manager',
            'sales',
            'accountant',
            'warehouse_keeper',
            'driver',
            'planning',
        ];

        $roles = collect();
        foreach ($roleNames as $name) {
            $roles[$name] = Role::firstOrCreate(['name' => $name, 'guard_name' => 'web']);
        }

        // 2) الأذونات (مُحدَّثة)
        $permissionNames = [
            // المدن والفروع والمستخدمون والأدوار
            'cities.view',
            'cities.create',
            'cities.update',
            'cities.delete',
            'branches.view',
            'branches.create',
            'branches.update',
            'branches.delete',
            'users.view',
            'users.create',
            'users.update',
            'users.delete',
            'roles.view',
            'roles.create',
            'roles.update',
            'roles.delete',

            // العملاء (مطلوب لـ sales.role)
            'clients.view',

            // الموارد البشرية
            'pay_grades.view',
            'pay_grades.create',
            'pay_grades.update',
            'pay_grades.delete',
            'employees.view',
            'employees.create',
            'employees.update',
            'employees.delete',

            // المخازن/الجرد
            'warehouses.view',
            'warehouses.create',
            'warehouses.update',
            'warehouses.delete',
            'inventory.items.view',
            'inventory.items.create',
            'inventory.items.update',
            'inventory.items.delete',
            'inventory.categories.view',
            'inventory.categories.create',
            'inventory.categories.update',
            'inventory.categories.delete',
            'inventory.stock.view',
            'inventory.stock.in',
            'inventory.stock.out',
            'inventory.stock.transfer',
            'inventory.stock.adjust',

            // المبيعات (مُحدَّثة بإرسال/طباعة)
            'sales.quotations.view',
            'sales.quotations.create',
            'sales.quotations.update',
            'sales.quotations.delete',
            'sales.quotations.approve',
            'sales.quotations.send',
            'sales.quotations.print',
            'sales.invoices.view',
            'sales.invoices.create',
            'sales.invoices.update',
            'sales.invoices.delete',
            'sales.invoices.approve',
            'sales.invoices.send',
            'sales.invoices.print',
            'sales.payments.view',
            'sales.payments.create',
            'sales.payments.refund',

            // المشتريات
            'purchases.requests.view',
            'purchases.requests.create',
            'purchases.requests.update',
            'purchases.requests.delete',
            'purchases.orders.view',
            'purchases.orders.create',
            'purchases.orders.update',
            'purchases.orders.delete',
            'purchases.orders.approve',
            'purchases.receipts.view',
            'purchases.receipts.create',
            'purchases.receipts.update',
            'purchases.receipts.delete',
            'purchases.supplier_invoices.view',
            'purchases.supplier_invoices.create',
            'purchases.supplier_invoices.update',
            'purchases.supplier_invoices.delete',
            'purchases.supplier_payments.view',
            'purchases.supplier_payments.create',

            // المشاريع/التصميم
            'projects.view',
            'projects.create',
            'projects.update',
            'projects.delete',
            'projects.assign',
            'projects.schedule',
            'projects.progress',
            'designs.view',
            'designs.create',
            'designs.update',
            'designs.delete',
            'designs.approve',

            // التوصيل
            'delivery.tasks.view',
            'delivery.tasks.assign',
            'delivery.tasks.update_status',
            'delivery.tasks.complete',

            // المحاسبة والتقارير والإعدادات (مُحدَّثة بتقارير تصدير)
            'accounting.vouchers.view',
            'accounting.vouchers.create',
            'accounting.vouchers.approve',
            'accounting.expenses.view',
            'accounting.expenses.create',
            'accounting.expenses.update',
            'accounting.expenses.delete',
            'accounting.reports.view',
            'accounting.treasury.view',
            'accounting.treasury.close_shift',
            'reports.view',
            'reports.export',
            'settings.view',
            'settings.update',
            'files.view',
            'files.upload',
        ];

        foreach ($permissionNames as $p) {
            Permission::firstOrCreate(['name' => $p, 'guard_name' => 'web']);
        }

        $allPermissions = Permission::where('guard_name', 'web')->pluck('name')->values();

        // Helper: يدعم wildcard prefix.* أو الاسم الكامل
        $grant = function (array $patterns) use ($allPermissions) {
            $matched = collect();
            foreach ($patterns as $pat) {
                $pat = trim($pat);
                if (! str_ends_with($pat, '.*')) {
                    if ($allPermissions->contains($pat)) $matched->push($pat);
                    continue;
                }
                $prefix = rtrim($pat, '.*');
                $matched = $matched->merge(
                    $allPermissions->filter(fn($p) => str_starts_with($p, $prefix . '.'))
                );
            }
            return $matched->unique()->values()->all();
        };

        // 3) ربط الأذونات بالأدوار

        // (1) admin: كل شيء
        $roles['admin']->syncPermissions($allPermissions);

        // (2) accounting_manager
        $roles['accounting_manager']->syncPermissions($grant([
            'accounting.*',
            'sales.invoices.*',
            'sales.payments.*',
            'purchases.*',
            'reports.*',
            'settings.view',
            'files.*',
            'branches.view',
            'users.view',
            'employees.view',
            'warehouses.view',
            'inventory.items.view',
            'inventory.stock.view',
        ]));

        // (3) branch_manager
        $roles['branch_manager']->syncPermissions($grant([
            'branches.view',
            'employees.*',
            'users.view',
            'sales.quotations.*',
            'sales.invoices.*',
            'sales.payments.*',
            'inventory.items.*',
            'inventory.categories.*',
            'inventory.stock.*',
            'warehouses.*',
            'purchases.requests.*',
            'purchases.orders.*',
            'purchases.receipts.*',
            'projects.view',
            'projects.create',
            'projects.update',
            'projects.assign',
            'projects.schedule',
            'projects.progress',
            'reports.view',
            'files.*',
        ]));

        // (4) sales
        $roles['sales']->syncPermissions($grant([
            'sales.quotations.view',
            'sales.quotations.create',
            'sales.quotations.update',
            'sales.quotations.send',
            'sales.quotations.print',
            'sales.invoices.view',
            'sales.invoices.create',
            'sales.invoices.update',
            'sales.invoices.send',
            'sales.invoices.print',
            'sales.payments.view',
            'sales.payments.create',
            'clients.view',
            'files.upload',
            'files.view',
            'inventory.items.view',
            'inventory.stock.view',
        ]));

        // (5) accountant
        $roles['accountant']->syncPermissions($grant([
            'accounting.*',
            'sales.invoices.view',
            'sales.invoices.approve',
            'sales.payments.*',
            'purchases.*',
            'reports.view',
            'files.*',
        ]));

        // (6) warehouse_keeper
        $roles['warehouse_keeper']->syncPermissions($grant([
            'warehouses.*',
            'inventory.items.view',
            'inventory.stock.*',
            'inventory.categories.view',
            'purchases.receipts.*',
            'files.view',
        ]));

        // (7) driver
        $roles['driver']->syncPermissions($grant([
            'delivery.tasks.view',
            'delivery.tasks.update_status',
            'delivery.tasks.complete',
            'files.view',
        ]));

        // (8) planning
        $roles['planning']->syncPermissions($grant([
            'projects.view',
            'projects.create',
            'projects.update',
            'projects.assign',
            'projects.schedule',
            'projects.progress',
            'designs.view',
            'designs.create',
            'designs.update',
            'designs.approve',
            'files.view',
            'files.upload',
        ]));

        // 🧹 نظّف الكاش بعد الربط
        app(PermissionRegistrar::class)->forgetCachedPermissions();
    }
}
