<?php

namespace Database\Seeders;

use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\Attribute;
use App\Models\AttributeValue;
use App\Enums\ProductUnit;
use Illuminate\Database\Seeder;

class ProductAndVariantSeeder extends Seeder
{
    /**
     * تعبئة جدول المنتجات والمتغيرات
     */
    public function run(): void
    {
        // جلب الخصائص والقيم
        $colorAttr = Attribute::where('name', 'color')->first();
        $sizeAttr = Attribute::where('name', 'size')->first();
        $materialAttr = Attribute::where('name', 'material')->first();

        $colors = $colorAttr?->values()->pluck('id', 'value') ?? collect();
        $sizes = $sizeAttr?->values()->pluck('id', 'value') ?? collect();
        $materials = $materialAttr?->values()->pluck('id', 'value') ?? collect();

        // =================== منتج بسيط (Simple Product) ===================
        $simpleProduct = Product::firstOrCreate(
            ['name' => 'قلم حبر أزرق'],
            [
                'description' => 'قلم حبر جاف لون أزرق، مناسب للاستخدام اليومي',
                'unit'        => ProductUnit::PIECE,
                'is_variable' => false,
                'is_active'   => true,
            ]
        );

        // متغير افتراضي واحد للمنتج البسيط
        if ($simpleProduct->variants()->count() === 0) {
            ProductVariant::create([
                'product_id' => $simpleProduct->id,
                'sku'        => 'PEN-BLUE-001',
                'cost'       => 5000, // 50 دينار
                'is_active'  => true,
                'is_default' => true,
            ]);
        }

        $this->command->info("✅ تم إنشاء منتج بسيط: {$simpleProduct->name}");

        // =================== منتج متغير: قميص (Variable Product) ===================
        $tshirtProduct = Product::firstOrCreate(
            ['name' => 'قميص قطني'],
            [
                'description' => 'قميص قطني عالي الجودة بألوان وأحجام متعددة',
                'unit'        => ProductUnit::PIECE,
                'is_variable' => true,
                'is_active'   => true,
            ]
        );

        if ($tshirtProduct->variants()->count() === 0) {
            $variantData = [
                // أحمر - صغير
                [
                    'cost'       => 150000, // 1500 دينار
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('red')],
                        ['attribute_id' => $sizeAttr->id, 'value_id' => $sizes->get('s')],
                    ],
                    'is_default' => true,
                ],
                // أحمر - متوسط
                [
                    'cost'       => 150000,
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('red')],
                        ['attribute_id' => $sizeAttr->id, 'value_id' => $sizes->get('m')],
                    ],
                    'is_default' => false,
                ],
                // أحمر - كبير
                [
                    'cost'       => 150000,
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('red')],
                        ['attribute_id' => $sizeAttr->id, 'value_id' => $sizes->get('l')],
                    ],
                    'is_default' => false,
                ],
                // أزرق - صغير
                [
                    'cost'       => 150000,
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('blue')],
                        ['attribute_id' => $sizeAttr->id, 'value_id' => $sizes->get('s')],
                    ],
                    'is_default' => false,
                ],
                // أزرق - متوسط
                [
                    'cost'       => 150000,
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('blue')],
                        ['attribute_id' => $sizeAttr->id, 'value_id' => $sizes->get('m')],
                    ],
                    'is_default' => false,
                ],
                // أزرق - كبير
                [
                    'cost'       => 150000,
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('blue')],
                        ['attribute_id' => $sizeAttr->id, 'value_id' => $sizes->get('l')],
                    ],
                    'is_default' => false,
                ],
                // أسود - متوسط
                [
                    'cost'       => 150000,
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('black')],
                        ['attribute_id' => $sizeAttr->id, 'value_id' => $sizes->get('m')],
                    ],
                    'is_default' => false,
                ],
                // أسود - كبير
                [
                    'cost'       => 150000,
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('black')],
                        ['attribute_id' => $sizeAttr->id, 'value_id' => $sizes->get('l')],
                    ],
                    'is_default' => false,
                ],
            ];

            foreach ($variantData as $data) {
                $variant = ProductVariant::create([
                    'product_id' => $tshirtProduct->id,
                    'cost'       => $data['cost'],
                    'is_active'  => true,
                    'is_default' => $data['is_default'],
                ]);

                // ربط الخصائص بالمتغير
                foreach ($data['attributes'] as $attr) {
                    $variant->attributeValues()->attach($attr['value_id'], [
                        'attribute_id' => $attr['attribute_id'],
                    ]);
                }
            }
        }

        $this->command->info("✅ تم إنشاء منتج متغير: {$tshirtProduct->name} مع " . $tshirtProduct->variants()->count() . " متغير");

        // =================== منتج متغير: حذاء (Variable Product) ===================
        $shoeProduct = Product::firstOrCreate(
            ['name' => 'حذاء رياضي'],
            [
                'description' => 'حذاء رياضي مريح بألوان وأحجام متعددة',
                'unit'        => ProductUnit::PIECE,
                'is_variable' => true,
                'is_active'   => true,
            ]
        );

        if ($shoeProduct->variants()->count() === 0) {
            $shoeVariants = [
                // أبيض
                [
                    'cost'       => 350000, // 3500 دينار
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('white')],
                    ],
                    'is_default' => true,
                ],
                // أسود
                [
                    'cost'       => 350000,
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('black')],
                    ],
                    'is_default' => false,
                ],
                // رمادي
                [
                    'cost'       => 350000,
                    'attributes' => [
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('gray')],
                    ],
                    'is_default' => false,
                ],
            ];

            foreach ($shoeVariants as $data) {
                $variant = ProductVariant::create([
                    'product_id' => $shoeProduct->id,
                    'cost'       => $data['cost'],
                    'is_active'  => true,
                    'is_default' => $data['is_default'],
                ]);

                // ربط الخصائص بالمتغير
                foreach ($data['attributes'] as $attr) {
                    $variant->attributeValues()->attach($attr['value_id'], [
                        'attribute_id' => $attr['attribute_id'],
                    ]);
                }
            }
        }

        $this->command->info("✅ تم إنشاء منتج متغير: {$shoeProduct->name} مع " . $shoeProduct->variants()->count() . " متغير");

        // =================== منتج متغير: حقيبة جلدية ===================
        $bagProduct = Product::firstOrCreate(
            ['name' => 'حقيبة جلدية'],
            [
                'description' => 'حقيبة جلدية فاخرة بألوان ومواد متعددة',
                'unit'        => ProductUnit::PIECE,
                'is_variable' => true,
                'is_active'   => true,
            ]
        );

        if ($bagProduct->variants()->count() === 0) {
            $bagVariants = [
                // جلد - أسود
                [
                    'cost'       => 750000, // 7500 دينار
                    'attributes' => [
                        ['attribute_id' => $materialAttr->id, 'value_id' => $materials->get('leather')],
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('black')],
                    ],
                    'is_default' => true,
                ],
                // جلد - أبيض
                [
                    'cost'       => 750000,
                    'attributes' => [
                        ['attribute_id' => $materialAttr->id, 'value_id' => $materials->get('leather')],
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('white')],
                    ],
                    'is_default' => false,
                ],
                // نايلون - أسود
                [
                    'cost'       => 450000, // 4500 دينار
                    'attributes' => [
                        ['attribute_id' => $materialAttr->id, 'value_id' => $materials->get('nylon')],
                        ['attribute_id' => $colorAttr->id, 'value_id' => $colors->get('black')],
                    ],
                    'is_default' => false,
                ],
            ];

            foreach ($bagVariants as $data) {
                $variant = ProductVariant::create([
                    'product_id' => $bagProduct->id,
                    'cost'       => $data['cost'],
                    'is_active'  => true,
                    'is_default' => $data['is_default'],
                ]);

                // ربط الخصائص بالمتغير
                foreach ($data['attributes'] as $attr) {
                    $variant->attributeValues()->attach($attr['value_id'], [
                        'attribute_id' => $attr['attribute_id'],
                    ]);
                }
            }
        }

        $this->command->info("✅ تم إنشاء منتج متغير: {$bagProduct->name} مع " . $bagProduct->variants()->count() . " متغير");

        $this->command->info('✅✅ تم إنشاء جميع المنتجات والمتغيرات بنجاح');
    }
}
