<?php

namespace Database\Seeders;

use App\Models\Product;
use App\Models\ProductPrice;
use Illuminate\Database\Seeder;

class OffersSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // جلب أول 10 منتجات لإضافة عروض عليها
        $products = Product::query()
            ->active()
            ->with('variants')
            ->limit(10)
            ->get();

        if ($products->isEmpty()) {
            $this->command->warn('لا توجد منتجات في قاعدة البيانات. الرجاء إضافة منتجات أولاً.');

            return;
        }

        $discounts = [10, 15, 20, 25, 30, 35, 40, 50]; // نسب خصم مختلفة

        foreach ($products as $index => $product) {
            $variant = $product->variants()->where('is_default', true)->first()
                ?? $product->variants()->first();

            if (! $variant) {
                continue;
            }

            // جلب أو إنشاء ProductPrice
            $price = ProductPrice::query()
                ->where('product_id', $product->id)
                ->where('product_variant_id', $variant->id)
                ->first();

            if (! $price) {
                // إنشاء سعر جديد إذا لم يكن موجوداً
                $price = ProductPrice::create([
                    'product_id' => $product->id,
                    'product_variant_id' => $variant->id,
                    'retail_price' => $variant->cost > 0 ? (int) ($variant->cost * 1.3) : 50000,
                    'wholesale_price' => $variant->cost > 0 ? (int) ($variant->cost * 1.2) : 40000,
                    'retail_margin' => 30,
                    'wholesale_margin' => 20,
                    'status' => true,
                ]);
            }

            // تطبيق خصم
            $discountPercent = $discounts[$index % count($discounts)];
            $salePrice = (int) round($price->retail_price * (1 - $discountPercent / 100));

            $price->update([
                'is_on_sale' => true,
                'discount_percentage' => $discountPercent,
                'sale_price' => $salePrice,
                'sale_start_date' => now()->subDays(2),
                'sale_end_date' => now()->addDays(30),
            ]);

            $this->command->info("تم إضافة عرض {$discountPercent}% على منتج: {$product->name}");
        }

        $this->command->info('تم إضافة العروض بنجاح!');
    }
}
