<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use Illuminate\Support\Collection;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\PermissionRegistrar;

class ModelHasPermissionsSeeder extends Seeder
{
    /**
     * إن كنت تفضّل الإضافة فقط استخدم givePermissionTo،
     * وإن كنت تريد جعل الصلاحيات المباشرة للمستخدم تماماً = الخريطة أدناه،
     * فعِّل $useSync بدلاً من give.
     */
    protected bool $useSync = false;

    public function run(): void
    {
        app(PermissionRegistrar::class)->forgetCachedPermissions();

        // اجلب كل الصلاحيات (web) مرّة واحدة لسرعة المطابقة
        $allPermissions = Permission::where('guard_name', 'web')->pluck('name');

        // Helper: يحوّل أنماط مثل reports.* إلى قائمة أسماء صلاحيات فعلية
        $grant = function (array $patterns) use ($allPermissions): array {
            $matched = collect();

            foreach ($patterns as $pat) {
                $pat = trim($pat);
                if ($pat === '') continue;

                // اسم كامل
                if (! str_ends_with($pat, '.*')) {
                    if ($allPermissions->contains($pat)) {
                        $matched->push($pat);
                    } else {
                        echo "⚠️ Permission not found: {$pat}\n";
                    }
                    continue;
                }

                // wildcard prefix.*
                $prefix = rtrim($pat, '.*');
                $subset = $allPermissions->filter(
                    fn($name) => str_starts_with($name, $prefix . '.')
                );

                if ($subset->isEmpty()) {
                    echo "⚠️ No permissions match wildcard: {$pat}\n";
                } else {
                    $matched = $matched->merge($subset);
                }
            }

            return $matched->unique()->values()->all();
        };

        // ===========================
        // خريطة: email => أنماط صلاحيات
        // يمكنك استخدام أسماء كاملة أو wildcards
        // ===========================
        $map = [
            'admin@nova.test' => [
                'settings.view',
                'reports.export',
                // أمثلة:
                // 'reports.*',
                // 'sales.invoices.*',
            ],
            'manager@nova.test' => [
                'sales.invoices.approve',
            ],
        ];

        foreach ($map as $email => $patterns) {
            $user = User::where('email', $email)->first();

            if (! $user) {
                echo "⚠️ User not found: {$email} — skipped.\n";
                continue;
            }

            $perms = $grant($patterns);

            if (empty($perms)) {
                echo "⚠️ No resolved permissions for {$email} — skipped.\n";
                continue;
            }

            if ($this->useSync) {
                $user->syncPermissions($perms);
                echo "✅ Synced direct permissions for {$email}: " . implode(', ', $perms) . "\n";
            } else {
                $user->givePermissionTo($perms);
                echo "✅ Granted direct permissions to {$email}: " . implode(', ', $perms) . "\n";
            }
        }

        app(PermissionRegistrar::class)->forgetCachedPermissions();
    }
}
