<?php

namespace Database\Seeders;

use App\Enums\MovementDirection;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\StockMovement;
use App\Models\StockBalance;
use App\Models\Warehouse;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class InitialStockSeeder extends Seeder
{
    /**
     * إدخال كميات افتتاحية لكل متغير منتج في مستودع افتراضي
     * مع تسجيل حركات واردة IN في جدول stock_movements
     * وتحديث جدول stock_balances بالمخزون الحالي.
     */
    public function run(): void
    {
        $this->command->info('📦 بدء إدخال المخزون الافتتاحي وتسجيل الحركات...');

        // حماية: إذا كانت هناك حركات مخزون مسبقة، يمكن تجاهل هذا السييدر
        if (StockMovement::count() > 0) {
            $this->command->warn('⚠️ توجد حركات مخزون مسبقاً، سيتم تجاهل InitialStockSeeder.');
            return;
        }

        // اختيار مستودع افتراضي: "المستودع الرئيسي" أو أول مستودع
        $warehouse = Warehouse::where('name', 'المستودع الرئيسي')->first()
            ?? Warehouse::first();

        if (! $warehouse) {
            $this->command->error('❌ لا توجد مستودعات، يرجى تشغيل WarehouseSeeder أولاً.');
            return;
        }

        // اختيار مستخدم (اختياري) لربط الحركة به
        $defaultUser = User::first();
        $userId      = $defaultUser?->id;

        $this->command->info("🏬 سيتم إدخال الكميات في المستودع: {$warehouse->name}");

        DB::beginTransaction();

        try {
            // 1) منتج بسيط: قلم حبر أزرق
            $this->seedSimpleProductStock(
                productName: 'قلم حبر أزرق',
                warehouseId: $warehouse->id,
                userId: $userId,
                quantity: 500 // مثال: 500 قلم
            );

            // 2) منتج متغير: قميص قطني
            $this->seedVariableProductStock(
                productName: 'قميص قطني',
                warehouseId: $warehouse->id,
                userId: $userId,
                defaultQty: 100, // للمتغير الافتراضي
                otherQty: 50     // لباقي المتغيرات
            );

            // 3) منتج متغير: حذاء رياضي
            $this->seedVariableProductStock(
                productName: 'حذاء رياضي',
                warehouseId: $warehouse->id,
                userId: $userId,
                defaultQty: 80,
                otherQty: 40
            );

            // 4) منتج متغير: حقيبة جلدية
            $this->seedVariableProductStock(
                productName: 'حقيبة جلدية',
                warehouseId: $warehouse->id,
                userId: $userId,
                defaultQty: 30,
                otherQty: 20
            );

            DB::commit();

            $this->command->info('✅✅ تم إدخال المخزون الافتتاحي وتسجيل الحركات وتحديث الأرصدة بنجاح.');
        } catch (\Throwable $e) {
            DB::rollBack();
            $this->command->error('❌ خطأ أثناء إدخال المخزون الافتتاحي: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * إدخال مخزون لمنتج بسيط (متغير واحد)
     */
    protected function seedSimpleProductStock(
        string $productName,
        int $warehouseId,
        ?int $userId,
        int $quantity
    ): void {
        /** @var Product|null $product */
        $product = Product::where('name', $productName)->first();

        if (! $product) {
            $this->command->warn("⚠️ لم يتم العثور على المنتج البسيط: {$productName}");
            return;
        }

        /** @var ProductVariant|null $variant */
        $variant = $product->variants()->first();

        if (! $variant) {
            $this->command->warn("⚠️ المنتج {$productName} لا يحتوي على متغيرات.");
            return;
        }

        $this->createInMovement(
            warehouseId: $warehouseId,
            variant: $variant,
            quantity: $quantity,
            userId: $userId,
            reference: "رصيد افتتاحي - {$productName}"
        );

        $this->command->info("✅ تم إدخال {$quantity} قطعة من {$productName} إلى المستودع.");
    }

    /**
     * إدخال مخزون لمنتج متغير (عدة متغيرات)
     */
    protected function seedVariableProductStock(
        string $productName,
        int $warehouseId,
        ?int $userId,
        int $defaultQty,
        int $otherQty
    ): void {
        /** @var Product|null $product */
        $product = Product::where('name', $productName)->first();

        if (! $product) {
            $this->command->warn("⚠️ لم يتم العثور على المنتج المتغير: {$productName}");
            return;
        }

        $variants = $product->variants;

        if ($variants->isEmpty()) {
            $this->command->warn("⚠️ المنتج {$productName} لا يحتوي على متغيرات.");
            return;
        }

        foreach ($variants as $variant) {
            $qty = $variant->is_default ? $defaultQty : $otherQty;

            $this->createInMovement(
                warehouseId: $warehouseId,
                variant: $variant,
                quantity: $qty,
                userId: $userId,
                reference: "رصيد افتتاحي - {$productName}"
            );

            $this->command->info(
                "   ➤ تم إدخال {$qty} قطعة من متغير (ID={$variant->id}) للمنتج {$productName}"
            );
        }

        $this->command->info("✅ تم إدخال مخزون لكل متغيرات: {$productName}");
    }

    /**
     * إنشاء حركة واردة IN لمتغير معيّن
     * وتحديث/إنشاء رصيد في StockBalance لنفس المتغير والمستودع.
     */
    protected function createInMovement(
        int $warehouseId,
        ProductVariant $variant,
        int $quantity,
        ?int $userId,
        ?string $reference = null
    ): StockMovement {
        $quantity  = max(0, (int) $quantity);
        $unitCost  = (int) ($variant->cost ?? 0);
        $reference = $reference ?: 'رصيد افتتاحي من InitialStockSeeder';

        return DB::transaction(function () use ($warehouseId, $variant, $quantity, $userId, $reference, $unitCost) {
            // 1) إنشاء حركة المخزون
            /** @var StockMovement $movement */
            $movement = StockMovement::create([
                'warehouse_id'       => $warehouseId,
                'product_variant_id' => $variant->id,
                'direction'          => MovementDirection::IN,
                'source_type'        => null, // لا يوجد مصدر محدد (رصيد افتتاحي)
                'source_id'          => null,
                'quantity'           => $quantity,
                'unit_cost'          => $unitCost,
                'total_cost'         => $unitCost * $quantity,
                'moved_at'           => now(),
                'reference'          => $reference,
                'note'               => 'إدخال مخزون افتتاحي عبر Seeder',
                'user_id'            => $userId,
            ]);

            // 2) إنشاء / تحديث رصيد المخزون StockBalance
            /** @var StockBalance $balance */
            $balance = StockBalance::query()
                ->lockForUpdate()
                ->firstOrNew([
                    'warehouse_id'       => $warehouseId,
                    'product_variant_id' => $variant->id,
                ]);

            // تأمين قيم افتراضية للباقي
            $balance->on_hand            = (int) $balance->on_hand + $quantity;
            $balance->reserved_qty       = $balance->reserved_qty       ?? 0;
            $balance->min_stock_level    = $balance->min_stock_level    ?? 0;
            $balance->max_stock_level    = $balance->max_stock_level    ?? 0;
            $balance->reorder_point      = $balance->reorder_point      ?? 0;
            $balance->reorder_quantity   = $balance->reorder_quantity   ?? 0;
            $balance->last_movement_at   = $movement->moved_at;

            if ($unitCost > 0) {
                $balance->last_unit_cost = $unitCost;
            }

            $balance->save();

            return $movement;
        });
    }
}
