<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('invoices', function (Blueprint $table) {
            $table->id();

            $table->foreignId('branch_id')
                ->constrained('branches')
                ->cascadeOnDelete();

            $table->foreignId('client_id')
                ->constrained('clients')
                ->cascadeOnDelete();

            // order_id سيتم إضافة foreign key له لاحقاً بعد إنشاء جدول orders
            $table->unsignedBigInteger('order_id')->nullable();

            $table->foreignId('currency_id')
                ->nullable()
                ->constrained('currencies')
                ->nullOnDelete();

            $table->unsignedInteger('exchange_rate')
                ->default(1);

            // 6 خانات - يسمح بالتكرار بين الفروع
            $table->string('invoice_number');
            $table->enum('invoice_type', ['ready'])->default('ready');
            $table->string('sale_type')->default('retail');
            $table->string('status')->default('draft');
            $table->string('payment_status')->default('unpaid');

            $table->date('invoice_date');
            $table->date('due_date')->nullable();

            // مبالغ (Integer + unsigned)
            $table->unsignedBigInteger('subtotal')->default(0);
            $table->unsignedBigInteger('discount_total')->default(0);
            $table->unsignedBigInteger('shipping_total')->default(0);
            $table->unsignedBigInteger('grand_total')->default(0);
            $table->unsignedBigInteger('paid_amount')->default(0);
            $table->unsignedBigInteger('due_amount')->default(0);

            $table->text('notes')->nullable();
            $table->text('internal_notes')->nullable();

            $table->foreignId('user_id')
                ->nullable()
                ->constrained('users')
                ->nullOnDelete();

            $table->foreignId('approved_by')
                ->nullable()
                ->constrained('users')
                ->nullOnDelete();

            $table->timestamp('approved_at')->nullable();

            $table->timestamps();
            $table->softDeletes();

            // ✅ فصل الترقيم حسب الفرع
            $table->unique(['branch_id', 'invoice_number'], 'invoices_branch_invoice_number_unique');

            // فهارس مساعدة
            $table->index(['branch_id', 'invoice_date'], 'invoices_branch_date_idx');
            $table->index(['client_id', 'invoice_date'], 'invoices_client_date_idx');
            $table->index('order_id', 'invoices_order_id_idx');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('invoices');
    }
};
