<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

/**
 * ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
 * 1) جدول فواتير المشتريات (Purchase Invoices)
 * ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
 */
return new class extends Migration
{
    public function up(): void
    {
        Schema::create('purchase_invoices', function (Blueprint $table) {
            $table->id();

            // معلومات أساسية
            $table->string('invoice_number'); // 'رقم فاتورة المورد كما ورد منه (مثال: INV-2025-001)'
            $table->foreignId('supplier_id')->constrained()->restrictOnDelete(); // 'المورّد صاحب الفاتورة - لا يُحذف إذا كانت له فواتير'
            $table->foreignId('branch_id')->nullable()->constrained()->cascadeOnDelete(); // 'الفرع المالِك للفاتورة'
            $table->foreignId('warehouse_id')->constrained()->restrictOnDelete(); // 'المستودع المستلم للبضاعة - لا يُحذف إذا كانت له فواتير'
            $table->foreignId('user_id')->nullable()->constrained(); // 'المستخدم الذي أنشأ الفاتورة'

            // تواريخ
            $table->date('invoice_date'); // 'تاريخ إصدار/استلام الفاتورة من المورد'
            $table->date('due_date')->nullable(); // 'تاريخ استحقاق الدفع (اختياري للفواتير الآجلة)'

            // المعلومات المالية (بالأعداد الصحيحة - أصغر وحدة عملة)
            $table->unsignedBigInteger('subtotal')->default(0); // 'المجموع الفرعي قبل الخصم والتوصيل'
            $table->unsignedBigInteger('discount')->default(0); // 'قيمة الخصم الإجمالي على الفاتورة'
            $table->unsignedBigInteger('freight')->default(0); // 'أجور/تكاليف التوصيل والشحن الوارد (Inbound)'
            $table->unsignedBigInteger('total')->default(0); // 'الإجمالي النهائي = subtotal - discount + freight'
            $table->unsignedBigInteger('paid')->default(0); // 'المبلغ المدفوع من الفاتورة حتى الآن'
            $table->unsignedBigInteger('due')->default(0); // 'المبلغ المتبقي (المستحق) على الفاتورة'

            // حالات الفاتورة
            $table->enum('status', ['draft', 'pending', 'partial', 'received', 'cancelled'])
                ->default('draft'); // 'حالة معالجة/استلام الفاتورة: مسودة/معلقة/استلام جزئي/مستلمة كاملة/ملغاة'

            $table->enum('payment_status', ['unpaid', 'partial', 'paid', 'overpaid'])
                ->default('unpaid'); // 'حالة سداد الفاتورة: غير مدفوعة/مدفوعة جزئيا/مدفوعة كاملة/دفع زائد'

            // ملاحظات
            $table->text('notes')->nullable(); // 'ملاحظات عامة على الفاتورة'

            $table->timestamps();
            $table->softDeletes(); // 'الحذف الآمن'

            // قيود التفريد والفهارس
            $table->unique(['branch_id', 'invoice_number'], 'uniq_branch_invoice_number'); // 'تفريد رقم الفاتورة داخل كل فرع'
            $table->index(['supplier_id', 'status', 'invoice_date'], 'idx_pi_supplier_status_date');
            $table->index(['supplier_id', 'payment_status', 'due_date'], 'idx_pi_supplier_pay_duedate'); // 'فهرس لسرعة البحث عن فواتير مورد معين حسب حالة السداد وتاريخ الاستحقاق'
            $table->index(['branch_id', 'warehouse_id'], 'idx_pi_branch_wh');
            $table->index(['payment_status', 'due_date'], 'idx_pi_payment_due');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('purchase_invoices');
    }
};
