<?php

namespace Database\Factories;

use App\Enums\PromotionAppliesTo;
use App\Enums\PromotionType;
use App\Models\Promotion;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends Factory<Promotion>
 */
class PromotionFactory extends Factory
{
    protected $model = Promotion::class;

    public function definition(): array
    {
        return [
            'name' => 'عرض '.fake()->word(),
            'description' => fake()->sentence(),
            'type' => PromotionType::Percentage,
            'discount_value' => fake()->randomElement([5, 10, 15, 20, 25]),
            'buy_quantity' => null,
            'get_quantity' => null,
            'min_purchase_amount' => null,
            'max_discount_amount' => null,
            'applies_to' => PromotionAppliesTo::All,
            'applicable_ids' => null,
            'priority' => fake()->numberBetween(0, 10),
            'is_stackable' => false,
            'usage_limit' => null,
            'usage_count' => 0,
            'starts_at' => now()->subDay(),
            'ends_at' => now()->addMonth(),
            'is_active' => true,
        ];
    }

    /**
     * عرض خصم ثابت
     */
    public function fixed(int $amount = 10000): static
    {
        return $this->state(fn (array $attributes) => [
            'type' => PromotionType::Fixed,
            'discount_value' => $amount,
        ]);
    }

    /**
     * عرض اشتري X واحصل على Y
     */
    public function buyXGetY(int $buy = 4, int $get = 1): static
    {
        return $this->state(fn (array $attributes) => [
            'type' => PromotionType::BuyXGetY,
            'discount_value' => null,
            'buy_quantity' => $buy,
            'get_quantity' => $get,
        ]);
    }

    /**
     * عرض شحن مجاني
     */
    public function freeShipping(int $minAmount = 100000): static
    {
        return $this->state(fn (array $attributes) => [
            'type' => PromotionType::FreeShipping,
            'discount_value' => null,
            'min_purchase_amount' => $minAmount,
        ]);
    }

    /**
     * عرض منتهي
     */
    public function expired(): static
    {
        return $this->state(fn (array $attributes) => [
            'starts_at' => now()->subMonth(),
            'ends_at' => now()->subDay(),
        ]);
    }

    /**
     * عرض غير نشط
     */
    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => false,
        ]);
    }

    /**
     * عرض لمنتجات محددة
     */
    public function forProducts(array $productIds): static
    {
        return $this->state(fn (array $attributes) => [
            'applies_to' => PromotionAppliesTo::Product,
            'applicable_ids' => $productIds,
        ]);
    }

    /**
     * عرض لتصنيفات محددة
     */
    public function forCategories(array $categoryIds): static
    {
        return $this->state(fn (array $attributes) => [
            'applies_to' => PromotionAppliesTo::Category,
            'applicable_ids' => $categoryIds,
        ]);
    }
}
