<?php

namespace Database\Factories;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\ProductVariant;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends Factory<OrderItem>
 */
class OrderItemFactory extends Factory
{
    protected $model = OrderItem::class;

    public function definition(): array
    {
        $variant = ProductVariant::query()->inRandomOrder()->first()
            ?? ProductVariant::factory()->create();

        $quantity = fake()->numberBetween(1, 5);
        $unitPrice = fake()->numberBetween(10000, 100000);
        $unitCost = (int) ($unitPrice * 0.7);
        $discountAmount = 0;
        $lineTotal = ($quantity * $unitPrice) - $discountAmount;

        return [
            'order_id' => Order::factory(),
            'product_id' => $variant->product_id,
            'product_variant_id' => $variant->id,
            'product_name' => $variant->product?->name ?? 'منتج تجريبي',
            'variant_name' => $variant->variant_label ?? null,
            'sku' => $variant->sku,
            'quantity' => $quantity,
            'unit_price' => $unitPrice,
            'unit_cost' => $unitCost,
            'discount_amount' => $discountAmount,
            'line_total' => $lineTotal,
            'is_free_item' => false,
            'promotion_id' => null,
        ];
    }

    /**
     * عنصر مجاني
     */
    public function free(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_free_item' => true,
            'unit_price' => 0,
            'line_total' => 0,
        ]);
    }

    /**
     * مع خصم
     */
    public function withDiscount(int $amount): static
    {
        return $this->state(function (array $attributes) use ($amount) {
            $lineTotal = ($attributes['quantity'] * $attributes['unit_price']) - $amount;

            return [
                'discount_amount' => $amount,
                'line_total' => max(0, $lineTotal),
            ];
        });
    }
}
