<?php

namespace Database\Factories;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Enums\ShippingMethod;
use App\Models\Branch;
use App\Models\Client;
use App\Models\Currency;
use App\Models\Order;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends Factory<Order>
 */
class OrderFactory extends Factory
{
    protected $model = Order::class;

    public function definition(): array
    {
        $subtotal = fake()->numberBetween(50000, 500000);
        $discountTotal = fake()->numberBetween(0, (int) ($subtotal * 0.2));
        $shippingFee = fake()->numberBetween(5000, 15000);
        $grandTotal = $subtotal - $discountTotal + $shippingFee;

        return [
            'order_number' => 'ORD-'.fake()->unique()->numerify('##-######-#####'),
            'branch_id' => Branch::query()->inRandomOrder()->first()?->id ?? Branch::factory(),
            'client_id' => Client::factory(),
            'currency_id' => Currency::query()->inRandomOrder()->first()?->id ?? 1,
            'exchange_rate' => 1,
            'shipping_address_id' => null,
            'shipping_method' => ShippingMethod::SelfDelivery,
            'shipping_company_id' => null,
            'subtotal' => $subtotal,
            'discount_total' => $discountTotal,
            'shipping_fee' => $shippingFee,
            'cod_fee' => 0,
            'grand_total' => $grandTotal,
            'status' => OrderStatus::Pending,
            'payment_status' => PaymentStatus::UNPAID,
            'payment_method' => 'cod',
            'promotion_id' => null,
            'coupon_id' => null,
            'client_discount_amount' => 0,
            'customer_notes' => fake()->optional()->sentence(),
            'internal_notes' => null,
        ];
    }

    /**
     * طلب مؤكد
     */
    public function confirmed(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => OrderStatus::Confirmed,
            'confirmed_at' => now(),
        ]);
    }

    /**
     * طلب مكتمل
     */
    public function completed(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => OrderStatus::Completed,
            'payment_status' => PaymentStatus::PAID,
            'confirmed_at' => now()->subDays(3),
            'completed_at' => now(),
        ]);
    }

    /**
     * طلب ملغي
     */
    public function canceled(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => OrderStatus::Canceled,
            'canceled_at' => now(),
        ]);
    }

    /**
     * طلب مدفوع
     */
    public function paid(): static
    {
        return $this->state(fn (array $attributes) => [
            'payment_status' => PaymentStatus::PAID,
        ]);
    }

    /**
     * طلب بشحن خارجي
     */
    public function withShippingCompany(): static
    {
        return $this->state(fn (array $attributes) => [
            'shipping_method' => ShippingMethod::Company,
        ]);
    }
}
