<?php

namespace Database\Factories;

use App\Models\Coupon;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Support\Str;

/**
 * @extends Factory<Coupon>
 */
class CouponFactory extends Factory
{
    protected $model = Coupon::class;

    public function definition(): array
    {
        return [
            'code' => strtoupper(Str::random(8)),
            'name' => 'كوبون '.fake()->word(),
            'description' => fake()->optional()->sentence(),
            'type' => 'percentage',
            'discount_value' => fake()->randomElement([5, 10, 15, 20]),
            'min_purchase_amount' => null,
            'max_discount_amount' => null,
            'usage_limit_total' => null,
            'usage_limit_per_client' => 1,
            'used_count' => 0,
            'starts_at' => now()->subDay(),
            'ends_at' => now()->addMonth(),
            'is_active' => true,
        ];
    }

    /**
     * كوبون خصم ثابت
     */
    public function fixed(int $amount = 10000): static
    {
        return $this->state(fn (array $attributes) => [
            'type' => 'fixed',
            'discount_value' => $amount,
        ]);
    }

    /**
     * كوبون بحد أدنى للشراء
     */
    public function minPurchase(int $amount): static
    {
        return $this->state(fn (array $attributes) => [
            'min_purchase_amount' => $amount,
        ]);
    }

    /**
     * كوبون بحد أقصى للخصم
     */
    public function maxDiscount(int $amount): static
    {
        return $this->state(fn (array $attributes) => [
            'max_discount_amount' => $amount,
        ]);
    }

    /**
     * كوبون محدود الاستخدام
     */
    public function limitedUsage(int $total = 100, int $perClient = 1): static
    {
        return $this->state(fn (array $attributes) => [
            'usage_limit_total' => $total,
            'usage_limit_per_client' => $perClient,
        ]);
    }

    /**
     * كوبون منتهي
     */
    public function expired(): static
    {
        return $this->state(fn (array $attributes) => [
            'starts_at' => now()->subMonth(),
            'ends_at' => now()->subDay(),
        ]);
    }

    /**
     * كوبون غير نشط
     */
    public function inactive(): static
    {
        return $this->state(fn (array $attributes) => [
            'is_active' => false,
        ]);
    }

    /**
     * كوبون مستنفد
     */
    public function exhausted(): static
    {
        return $this->state(fn (array $attributes) => [
            'usage_limit_total' => 10,
            'used_count' => 10,
        ]);
    }
}
