<?php

namespace App\Traits;

use App\Models\User;
use Illuminate\Support\Facades\Log;
use Filament\Notifications\Notification;

trait SendsUserActivityNotifications
{
    /**
     * تسجيل النشاط وإرسال الإشعار
     */
    protected function logAndNotifyUserActivity(
        User $targetUser,
        string $event = 'updated',
        array $properties = []
    ): void {
        $actor = user_info();

        if (!$actor instanceof User) {
            return;
        }

        // ✅ 1. تسجيل النشاط يدوياً
        $activity = activity()
            ->performedOn($targetUser)
            ->causedBy($actor)
            ->withProperties($properties)
            ->log($this->getActivityDescription($event));

        // ✅ 2. إرسال الإشعار - بطريقة مباشرة
        $this->sendNotificationToAdmins($targetUser, $event, $actor);
    }

    /**
     * إرسال إشعار بسيط ومباشر
     */
    protected function sendNotificationToAdmins(User $targetUser, string $event, User $actor): void
    {
        // ✅ جلب جميع الأدمن ما عدا الفاعل
        $admins = User::role('admin')
            ->where('id', '!=', $actor->id)
            ->get();

        // تأكد من وجود أدمن
        if ($admins->isEmpty()) {
            Log::info('No admins found to send notification');
            return;
        }

        // ✅ تحميل العلاقة
        $targetUser->load('branch');

        // تحضير النص
        $eventText = match ($event) {
            'created' => 'إضافة مستخدم جديد',
            'updated' => 'تحديث بيانات المستخدم',
            'deleted' => 'حذف المستخدم',
            default => 'نشاط جديد'
        };

        $icon = match ($event) {
            'created' => 'heroicon-o-user-plus',
            'updated' => 'heroicon-o-pencil-square',
            'deleted' => 'heroicon-o-trash',
            default => 'heroicon-o-bell'
        };

        // ✅ بناء رسالة الإشعار
        $title = $eventText;
        $body = "قام {$actor->name} ب{$eventText}: {$targetUser->name}";

        if ($targetUser->branch) {
            $body .= " | الفرع: {$targetUser->branch->name}";
        }

        // ✅ إرسال الإشعار لكل أدمن
        foreach ($admins as $admin) {
            Notification::make()
                ->title($title)
                ->body($body)
                ->icon($icon)
                ->iconColor('success')
                ->send()
                ->sendToDatabase($admin);
        }

        Log::info("Notification sent to {$admins->count()} admins");
    }

    /**
     * وصف النشاط
     */
    protected function getActivityDescription(string $event): string
    {
        return match ($event) {
            'created' => 'تم إنشاء المستخدم',
            'updated' => 'تم تحديث المستخدم',
            'deleted' => 'تم حذف المستخدم',
            default => "تم {$event} المستخدم"
        };
    }
}
