<?php

namespace App\Traits;

use Carbon\Carbon;
use Filament\Widgets\Concerns\InteractsWithPageFilters;

trait HasDateFilters
{
    use InteractsWithPageFilters;

    protected function getDateRange(): array
    {
        $filter = $this->filters['dateRange'] ?? 'today';

        return match ($filter) {
            'today' => [Carbon::today()->startOfDay(), Carbon::today()->endOfDay()],
            'yesterday' => [Carbon::yesterday()->startOfDay(), Carbon::yesterday()->endOfDay()],
            'this_week' => [Carbon::now()->startOfWeek(), Carbon::now()->endOfWeek()],
            'last_week' => [Carbon::now()->subWeek()->startOfWeek(), Carbon::now()->subWeek()->endOfWeek()],
            'last_7_days' => [Carbon::now()->subDays(6)->startOfDay(), Carbon::now()->endOfDay()],
            'this_month' => [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()],
            'last_month' => [Carbon::now()->subMonth()->startOfMonth(), Carbon::now()->subMonth()->endOfMonth()],
            'last_30_days' => [Carbon::now()->subDays(29)->startOfDay(), Carbon::now()->endOfDay()],
            'last_90_days' => [Carbon::now()->subDays(89)->startOfDay(), Carbon::now()->endOfDay()],
            'this_year' => [Carbon::now()->startOfYear(), Carbon::now()->endOfYear()],
            'last_year' => [Carbon::now()->subYear()->startOfYear(), Carbon::now()->subYear()->endOfYear()],
            'custom' => [
                ! empty($this->filters['startDate'])
                    ? Carbon::parse($this->filters['startDate'])->startOfDay()
                    : Carbon::today()->startOfDay(),
                ! empty($this->filters['endDate'])
                    ? Carbon::parse($this->filters['endDate'])->endOfDay()
                    : Carbon::today()->endOfDay(),
            ],
            'all' => [null, null],
            default => [Carbon::today()->startOfDay(), Carbon::today()->endOfDay()],
        };
    }

    protected function applyDateFilter($query, string $dateColumn = 'created_at')
    {
        [$startDate, $endDate] = $this->getDateRange();

        if ($startDate && $endDate) {
            return $query->whereBetween($dateColumn, [$startDate, $endDate]);
        }

        return $query;
    }

    protected function getSelectedBranch(): ?int
    {
        $branchId = $this->filters['branch_id'] ?? null;

        if (empty($branchId)) {
            $branchId = function_exists('user_info') ? user_info('branch_id') : null;
        }

        return $branchId ? (int) $branchId : null;
    }

    protected function applyBranchFilter($query)
    {
        $branchId = $this->getSelectedBranch();

        if ($branchId) {
            return $query->where('branch_id', $branchId);
        }

        return $query;
    }

    protected function dateRangeLabel(): string
    {
        [$from, $to] = $this->getDateRange();

        if (! $from || ! $to) {
            return 'كل الفترات';
        }

        $f = $from->toDateString();
        $t = $to->toDateString();

        return $f === $t ? "التاريخ: {$f}" : "من {$f} إلى {$t}";
    }
}
