<?php

namespace App\Support;

use Closure;
use Filament\Forms\Get;
use App\Helper\PriceHelper;
use Filament\Forms\Components\TextInput;
use Illuminate\Contracts\Support\Htmlable;
use Filament\Schemas\Components\Utilities\Set;

class MoneyInput
{
    /**
     * حقل مبالغ مع فواصل آلاف بصريًا وتخزين كعدد صحيح (دينار عراقي).
     *
     * @param  string  $name
     * @param  string|Closure|Htmlable|null  $label
     */
    public static function make(string $name, string|Closure|Htmlable|null $label = null): TextInput
    {
        return TextInput::make($name)
            ->label($label)
            ->suffix('د.ع')
            ->extraInputAttributes([
                'inputmode' => 'numeric',
                'dir'       => 'ltr',
                'style'     => 'text-align: right;',
            ])
            ->live(debounce: 300)
            // عند تعبئة النموذج (عرض من DB) → صياغة بفواصل
            ->formatStateUsing(fn($state) => PriceHelper::format($state))
            // عند التحديث أثناء الكتابة → نظّف ثم صيّغ
            ->afterStateUpdated(function ($state, Set $set) use ($name) {
                $set($name, PriceHelper::format($state));
            })
            // عند التجفيف (قبل الحفظ) → حوّل لعدد صحيح
            ->dehydrateStateUsing(fn($state) => PriceHelper::dehydrate($state))
            // اجعل القيمة تُرسل دائمًا حتى لو فارغة (ليتحوّل 0)
            ->dehydrated(true)
            // وللتحقق: اسمح فقط بالأرقام والفواصل/النقاط أثناء الإدخال
            ->rule('regex:/^[\p{N}\d,.\s]+$/u')
            ->validationMessages([
                'regex' => 'أدخل أرقامًا فقط.',
            ]);
    }
}
