<?php

namespace App\Services\Purchases;

use App\Enums\MovementDirection;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceItem;
use App\Models\PurchaseReturn;
use App\Models\PurchaseReturnItem;
use App\Models\StockMovement;
use App\Models\SupplierLedger;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PurchaseReturnService
{
    // ══════════════════════════════════════════════════════════════
    //  إنشاء مرتجع جديد
    // ══════════════════════════════════════════════════════════════

    public function create(array $data, array $items): PurchaseReturn
    {
        return DB::transaction(function () use ($data, $items) {
            if (empty($items)) {
                throw new \InvalidArgumentException('يجب إضافة عنصر واحد على الأقل للمرتجع');
            }

            $data['return_number'] = $this->generateReturnNumber();
            $data['user_id']       = user_info('id');
            $data['total']         = $this->calculateTotal($items);

            /** @var PurchaseReturn $return */
            $return = PurchaseReturn::create($data);

            $this->createReturnItems($return, $items);
            $this->processStockOut($return);
            $this->createLedgerEntry($return);
            $this->updateInvoiceAfterReturn($return);

            Log::info('تم إنشاء مرتجع شراء', [
                'return_id'  => $return->id,
                'invoice_id' => $return->purchase_invoice_id,
                'total'      => $return->total,
            ]);

            return $return;
        });
    }

    // ══════════════════════════════════════════════════════════════
    //  تعديل مرتجع
    // ══════════════════════════════════════════════════════════════

    public function update(PurchaseReturn $return, array $data, array $items): PurchaseReturn
    {
        return DB::transaction(function () use ($return, $data, $items) {
            if ($return->status === 'completed') {
                throw new \InvalidArgumentException('لا يمكن تعديل مرتجع مكتمل');
            }

            // عكس تأثير المرتجع السابق (مخزون + أستاذ)
            $this->reverseStockMovements($return);
            $this->reverseLedgerEntry($return);

            // حذف عناصر المرتجع القديمة
            $return->items()->delete();

            // حساب الإجمالي الجديد
            $data['total'] = $this->calculateTotal($items);

            $return->update($data);
            $return->refresh();

            // إعادة إنشاء عناصر المرتجع والحركات والقيود
            $this->createReturnItems($return, $items);
            $this->processStockOut($return);
            $this->createLedgerEntry($return);
            $this->updateInvoiceAfterReturn($return);

            Log::info('تم تعديل مرتجع شراء', [
                'return_id' => $return->id,
                'total'     => $return->total,
            ]);

            return $return;
        });
    }

    // ══════════════════════════════════════════════════════════════
    //  حذف مرتجع
    // ══════════════════════════════════════════════════════════════

    public function delete(PurchaseReturn $return): bool
    {
        return DB::transaction(function () use ($return) {
            if ($return->status === 'completed') {
                throw new \InvalidArgumentException('لا يمكن حذف مرتجع مكتمل');
            }

            $invoiceId = $return->purchase_invoice_id;

            // عكس تأثير المرتجع على المخزون
            $this->reverseStockMovements($return);

            // حذف قيود الأستاذ والعناصر
            $return->ledgerEntries()->delete();
            $return->items()->delete();

            // حذف المرتجع نفسه (هنا forceDelete، غيّره لـ delete() لو تريد SoftDeletes فقط)
            $return->forceDelete();

            if ($invoiceId) {
                $this->recalculateInvoice($invoiceId);
            }

            Log::info('تم حذف مرتجع شراء', ['return_id' => $return->id]);

            return true;
        });
    }

    // ══════════════════════════════════════════════════════════════
    //  إرجاع فاتورة كاملة
    // ══════════════════════════════════════════════════════════════

    public function returnFullInvoice(PurchaseInvoice $invoice, array $data = []): PurchaseReturn
    {
        if ($invoice->status === 'returned') {
            throw new \InvalidArgumentException('تم إرجاع هذه الفاتورة بالكامل مسبقاً');
        }

        $items = $invoice->items->map(function (PurchaseInvoiceItem $item) {
            $alreadyReturned = $item->returnItems()->sum('quantity_returned') ?? 0;
            $returnableQty   = $item->quantity_received - $alreadyReturned;

            return [
                'purchase_invoice_item_id' => $item->id,
                'product_variant_id'       => $item->product_variant_id,
                'quantity_returned'        => $returnableQty,
                'unit_cost'                => $item->unit_cost,
                'batch_number'             => $item->batch_number,
            ];
        })->filter(fn($item) => $item['quantity_returned'] > 0)->toArray();

        if (empty($items)) {
            throw new \InvalidArgumentException('لا توجد عناصر قابلة للإرجاع');
        }

        $returnData = array_merge([
            'purchase_invoice_id' => $invoice->id,
            'supplier_id'         => $invoice->supplier_id,
            'warehouse_id'        => $invoice->warehouse_id,
            'return_date'         => now(),
            'status'              => 'pending',
            'refund_method'       => 'credit_note',
        ], $data);

        return $this->create($returnData, $items);
    }

    // ══════════════════════════════════════════════════════════════
    //  إنشاء عناصر المرتجع
    // ══════════════════════════════════════════════════════════════

    protected function createReturnItems(PurchaseReturn $return, array $items): void
    {
        foreach ($items as $item) {
            $qty  = $this->toInt($item['quantity_returned'] ?? 0);
            $cost = $this->toInt($item['unit_cost'] ?? 0);

            if ($qty <= 0 || $cost < 0) {
                continue;
            }

            if (! empty($item['purchase_invoice_item_id'])) {
                $this->validateReturnQuantity(
                    (int) $item['purchase_invoice_item_id'],
                    $qty
                );
            }

            $lineTotal = $qty * $cost;

            PurchaseReturnItem::create([
                'purchase_return_id'       => $return->id,
                'purchase_invoice_item_id' => $item['purchase_invoice_item_id'] ?? null,
                'product_variant_id'       => (int) $item['product_variant_id'],
                'quantity_returned'        => $qty,
                'unit_cost'                => $cost,
                'line_total'               => $lineTotal,
                'batch_number'             => $item['batch_number'] ?? null,
                'notes'                    => $item['notes'] ?? null,
            ]);
        }
    }

    // ══════════════════════════════════════════════════════════════
    //  التحقق من الكمية القابلة للإرجاع
    // ══════════════════════════════════════════════════════════════

    protected function validateReturnQuantity(int $invoiceItemId, int $quantity): void
    {
        /** @var PurchaseInvoiceItem $item */
        $item = PurchaseInvoiceItem::findOrFail($invoiceItemId);

        $alreadyReturned = (int) $item->returnItems()->sum('quantity_returned');
        $maxReturnable   = (int) $item->quantity_received - $alreadyReturned;

        if ($quantity > $maxReturnable) {
            throw new \InvalidArgumentException(
                "الكمية المطلوب إرجاعها ({$quantity}) تتجاوز الكمية القابلة للإرجاع ({$maxReturnable})"
            );
        }
    }

    // ══════════════════════════════════════════════════════════════
    //  إخراج العناصر من المخزن (OUT)
    // ══════════════════════════════════════════════════════════════

    protected function processStockOut(PurchaseReturn $return): void
    {
        // نستخدم العلاقة polymorphic: $return->stockMovements()
        foreach ($return->items as $item) {
            $return->stockMovements()->create([
                'warehouse_id'       => $return->warehouse_id,
                'product_variant_id' => $item->product_variant_id,
                'direction'          => MovementDirection::OUT,
                'quantity'           => $item->quantity_returned,
                'unit_cost'          => $item->unit_cost,
                'total_cost'         => $item->line_total,
                'moved_at'           => $return->return_date,
                'reference'          => $return->return_number,
                'note'               => "مرتجع شراء #{$return->return_number}",
                'user_id'            => $return->user_id,
            ]);
        }
    }

    // ══════════════════════════════════════════════════════════════
    //  عكس حركات المخزون (IN لإلغاء OUT)
    // ══════════════════════════════════════════════════════════════

    protected function reverseStockMovements(PurchaseReturn $return): void
    {
        foreach ($return->items as $item) {
            $return->stockMovements()->create([
                'warehouse_id'       => $return->warehouse_id,
                'product_variant_id' => $item->product_variant_id,
                'direction'          => MovementDirection::IN,
                'quantity'           => $item->quantity_returned,
                'unit_cost'          => $item->unit_cost,
                'total_cost'         => $item->line_total,
                'moved_at'           => now(),
                'reference'          => $return->return_number,
                'note'               => "إلغاء مرتجع #{$return->return_number}",
                'user_id'            => user_info('id'),
            ]);
        }
    }

    // ══════════════════════════════════════════════════════════════
    //  تسجيل في دفتر الأستاذ
    // ══════════════════════════════════════════════════════════════

    protected function createLedgerEntry(PurchaseReturn $return): void
    {
        SupplierLedger::create([
            'supplier_id'      => $return->supplier_id,
            'branch_id'        => user_info('branch_id'),
            'ledgerable_type'  => PurchaseReturn::class,
            'ledgerable_id'    => $return->id,
            'transaction_date' => $return->return_date,
            'type'             => 'return',
            'amount'           => (int) $return->total, // موجب دائماً
            'reference'        => $return->return_number,
            'description'      => 'مرتجع شراء',
        ]);
    }

    // ══════════════════════════════════════════════════════════════
    //  إلغاء قيد دفتر الأستاذ
    // ══════════════════════════════════════════════════════════════

    protected function reverseLedgerEntry(PurchaseReturn $return): void
    {
        $return->ledgerEntries()->delete();
    }

    // ══════════════════════════════════════════════════════════════
    //  تحديث الفاتورة بعد المرتجع
    // ══════════════════════════════════════════════════════════════

    protected function updateInvoiceAfterReturn(PurchaseReturn $return): void
    {
        if (! $return->purchase_invoice_id) {
            return;
        }

        $invoice = PurchaseInvoice::find($return->purchase_invoice_id);

        if (! $invoice) {
            return;
        }

        $totalReturns = (int) PurchaseReturn::where('purchase_invoice_id', $invoice->id)->sum('total');

        $originalTotal = (int) ($invoice->subtotal - $invoice->discount + ($invoice->freight ?? 0));
        $newTotal      = max(0, $originalTotal - $totalReturns);
        $newDue        = max(0, $newTotal - $invoice->paid);

        $invoice->update([
            'total'          => $newTotal,
            'due'            => $newDue,
            'payment_status' => $this->determinePaymentStatus($newTotal, (int) $invoice->paid),
        ]);

        $this->checkIfFullyReturned($invoice);
    }

    // ══════════════════════════════════════════════════════════════
    //  التحقق من الإرجاع الكامل
    // ══════════════════════════════════════════════════════════════

    protected function checkIfFullyReturned(PurchaseInvoice $invoice): void
    {
        $allItemsReturned = true;

        foreach ($invoice->items as $item) {
            $totalReturned = (int) $item->returnItems()->sum('quantity_returned');

            if ($totalReturned < (int) $item->quantity_received) {
                $allItemsReturned = false;
                break;
            }
        }

        if ($allItemsReturned) {
            $invoice->update(['status' => 'returned']);
        }
    }

    // ══════════════════════════════════════════════════════════════
    //  إعادة حساب الفاتورة (بعد حذف مرتجع مثلاً)
    // ══════════════════════════════════════════════════════════════

    protected function recalculateInvoice(int $invoiceId): void
    {
        $invoice = PurchaseInvoice::find($invoiceId);

        if (! $invoice) {
            return;
        }

        $totalReturns = (int) PurchaseReturn::where('purchase_invoice_id', $invoiceId)->sum('total');

        $originalTotal = (int) ($invoice->subtotal - $invoice->discount + ($invoice->freight ?? 0));
        $newTotal      = max(0, $originalTotal - $totalReturns);
        $newDue        = max(0, $newTotal - $invoice->paid);

        $invoice->update([
            'total'          => $newTotal,
            'due'            => $newDue,
            'payment_status' => $this->determinePaymentStatus($newTotal, (int) $invoice->paid),
            'status'         => $totalReturns >= $originalTotal ? 'returned' : $invoice->status,
        ]);
    }

    // ══════════════════════════════════════════════════════════════
    //  تحديد حالة الدفع
    // ══════════════════════════════════════════════════════════════

    protected function determinePaymentStatus(int $total, int $paid): string
    {
        if ($total <= 0) {
            return 'paid';
        }

        if ($paid <= 0) {
            return 'unpaid';
        }

        if ($paid >= $total) {
            return 'paid';
        }

        return 'partial';
    }

    // ══════════════════════════════════════════════════════════════
    //  حساب الإجمالي
    // ══════════════════════════════════════════════════════════════

    protected function calculateTotal(array $items): int
    {
        return collect($items)->sum(function ($item) {
            $qty  = $this->toInt($item['quantity_returned'] ?? 0);
            $cost = $this->toInt($item['unit_cost'] ?? 0);

            if ($qty <= 0 || $cost < 0) {
                return 0;
            }

            return $qty * $cost;
        });
    }

    // ══════════════════════════════════════════════════════════════
    //  توليد رقم المرتجع
    // ══════════════════════════════════════════════════════════════

    protected function generateReturnNumber(): string
    {
        $prefix = 'RET-';
        $year   = date('Y');

        $lastNumber = (int) PurchaseReturn::withTrashed()
            ->where('return_number', 'like', "{$prefix}{$year}-%")
            ->max(DB::raw("CAST(SUBSTRING_INDEX(return_number, '-', -1) AS UNSIGNED)"));

        return $prefix . $year . '-' . str_pad($lastNumber + 1, 5, '0', STR_PAD_LEFT);
    }

    // ══════════════════════════════════════════════════════════════
    //  ملخص رصيد المورد
    // ══════════════════════════════════════════════════════════════

    public function getSupplierBalance(int $supplierId): array
    {
        $totalInvoices = (int) PurchaseInvoice::where('supplier_id', $supplierId)->sum('total');

        $totalReturns = (int) PurchaseReturn::where('supplier_id', $supplierId)->sum('total');

        // المدفوعات من دفتر الأستاذ (type = payment) بمبالغ موجبة
        $totalPaid = (int) SupplierLedger::where('supplier_id', $supplierId)
            ->where('type', 'payment')
            ->sum(DB::raw('ABS(amount)'));

        $netInvoices = $totalInvoices - $totalReturns;
        $balance     = $netInvoices - $totalPaid;

        return [
            'total_invoices' => $totalInvoices,
            'total_returns'  => $totalReturns,
            'net_invoices'   => $netInvoices,
            'total_paid'     => $totalPaid,
            'balance'        => $balance,
            'we_owe'         => max(0, $balance),   // ما ندين به للمورد
            'they_owe'       => max(0, -$balance),  // ما يدين به المورد لنا (رصيد دائن)
        ];
    }

    // ══════════════════════════════════════════════════════════════
    //  Helper: تحويل القيم إلى int (مع دعم الأرقام العربية والفواصل)
    // ══════════════════════════════════════════════════════════════

    protected function toInt($value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        if (is_int($value)) {
            return $value;
        }

        if (is_numeric($value)) {
            return (int) $value;
        }

        // إزالة كل شيء عدا الأرقام العربية والإنجليزية
        $cleaned = preg_replace('/[^\d٠-٩]/u', '', (string) $value);

        // تحويل الأرقام العربية إلى إنجليزية
        $cleaned = preg_replace_callback('/[٠-٩]/u', function ($m) {
            return mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]);
        }, $cleaned);

        return (int) ($cleaned === '' ? 0 : $cleaned);
    }
}
