<?php

namespace App\Services\Purchases;

use App\Models\{
    PurchaseInvoice,
    PurchaseReturn,
    SupplierLedger
};
use Illuminate\Support\Facades\DB;

class PurchaseReportService
{
    /**
     * تقرير المشتريات خلال فترة
     * يعيد إجماليات + قائمة الفواتير (مع المورد والعناصر والمتغير)
     */
    public function purchaseReport($startDate, $endDate, ?int $supplierId = null, ?int $branchId = null): array
    {
        $q = PurchaseInvoice::query()
            ->whereBetween('invoice_date', [$startDate, $endDate])
            ->when($supplierId, fn($qq) => $qq->where('supplier_id', $supplierId))
            ->when($branchId, fn($qq) => $qq->where('branch_id', $branchId))
            ->with([
                'supplier',
                'items.variant',   // العلاقة في PurchaseInvoiceItem هي variant
            ]);

        $invoices = $q->get();

        return [
            'total_invoices' => $invoices->count(),
            'total_amount'   => (int) $invoices->sum('total'),
            'total_paid'     => (int) $invoices->sum('paid'),
            'total_due'      => (int) $invoices->sum('due'),
            'total_discount' => (int) $invoices->sum('discount'),
            'total_freight'  => (int) $invoices->sum('freight'),
            'invoices'       => $invoices,
        ];
    }

    /**
     * تقرير المرتجعات خلال فترة
     */
    public function returnsReport($startDate, $endDate, ?int $supplierId = null, ?int $branchId = null): array
    {
        $q = PurchaseReturn::query()
            ->whereBetween('return_date', [$startDate, $endDate])
            ->when($supplierId, fn($qq) => $qq->where('supplier_id', $supplierId))
            ->when($branchId, fn($qq) => $qq->where('warehouse.branch_id', $branchId), function ($qq) use ($branchId) {
                // لو حاب تربط المرتجع بالفرع عبر المستودع:
                if ($branchId) {
                    $qq->whereHas('warehouse', function ($wq) use ($branchId) {
                        $wq->where('branch_id', $branchId);
                    });
                }
            })
            ->with([
                'supplier',
                'items.variant',
            ]);

        $returns = $q->get();

        return [
            'total_returns' => $returns->count(),
            'total_amount'  => (int) $returns->sum('total'),
            'returns'       => $returns,
        ];
    }

    /**
     * تقرير دفتر أستاذ المورد (مع رصيد افتتاحي وجاري)
     * - نعتمد على type لتحديد إشارة الحركة:
     *   purchase => +amount (زيادة التزامنا للمورد)
     *   return   => -amount (تقليل التزامنا)
     *   payment  => -amount (سداد المورد)
     */
    public function supplierLedgerReport(int $supplierId, $startDate = null, $endDate = null): array
    {
        // قيود الفترة المطلوبة
        $q = SupplierLedger::query()
            ->where('supplier_id', $supplierId)
            ->when($startDate && $endDate, fn($qq) => $qq->whereBetween('transaction_date', [$startDate, $endDate]))
            ->with('ledgerable')
            ->orderBy('transaction_date')
            ->orderBy('id');

        $entries = $q->get();

        // الرصيد الافتتاحي قبل startDate (إن وُجد)
        $opening = 0;
        if ($startDate) {
            $beforeEntries = SupplierLedger::where('supplier_id', $supplierId)
                ->where('transaction_date', '<', $startDate)
                ->get();

            $opening = $beforeEntries->reduce(function ($carry, SupplierLedger $e) {
                return $carry + $this->signedLedgerAmount($e);
            }, 0);
        }

        // رصيد جاري
        $running     = $opening;
        $withBalance = $entries->map(function (SupplierLedger $e) use (&$running) {
            $running += $this->signedLedgerAmount($e);
            $e->running_balance = $running;
            return $e;
        });

        // إجماليات داخل الفترة فقط
        $totalPurchases = (int) $entries->where('type', 'purchase')->sum('amount');
        $totalReturns   = (int) $entries->where('type', 'return')->sum('amount');
        $totalPayments  = (int) $entries->where('type', 'payment')->sum('amount');

        // في التقارير الكلاسيكية:
        // "debits" = ما يزيد الدين علينا (purchases)
        // "credits" = ما يقلل الدين (returns + payments)
        $totalDebits  = $totalPurchases;
        $totalCredits = $totalReturns + $totalPayments;

        return [
            'opening_balance' => (int) $opening,
            'closing_balance' => (int) $running,
            'total_debits'    => (int) $totalDebits,
            'total_credits'   => (int) $totalCredits,
            'total_purchases' => (int) $totalPurchases,
            'total_returns'   => (int) $totalReturns,
            'total_payments'  => (int) $totalPayments,
            'entries'         => $withBalance,
        ];
    }

    /**
     * تقرير الفواتير المتأخرة (due < اليوم وحالتها unpaid/partial)
     */
    public function overdueInvoicesReport(?int $supplierId = null, ?int $branchId = null): array
    {
        $today = now()->toDateString();

        $q = PurchaseInvoice::query()
            ->whereNotNull('due_date')
            ->where('due_date', '<', $today)
            ->whereIn('payment_status', ['unpaid', 'partial'])
            ->when($supplierId, fn($qq) => $qq->where('supplier_id', $supplierId))
            ->when($branchId, fn($qq) => $qq->where('branch_id', $branchId))
            ->with(['supplier', 'warehouse']);

        $invoices = $q->get();

        return [
            'total_overdue'    => $invoices->count(),
            'total_amount_due' => (int) $invoices->sum('due'),
            'invoices'         => $invoices,
        ];
    }

    /**
     * تقرير أعلى الموردين شراءً (إجمالي مبالغ الفواتير خلال فترة)
     */
    public function topSuppliersReport($startDate, $endDate, int $limit = 10, ?int $branchId = null): array
    {
        $suppliers = PurchaseInvoice::query()
            ->select(
                'supplier_id',
                DB::raw('SUM(total) as total_purchases'),
                DB::raw('COUNT(*) as invoice_count')
            )
            ->whereBetween('invoice_date', [$startDate, $endDate])
            ->when($branchId, fn($qq) => $qq->where('branch_id', $branchId))
            ->groupBy('supplier_id')
            ->orderByDesc('total_purchases')
            ->with('supplier')
            ->limit($limit)
            ->get();

        return ['suppliers' => $suppliers];
    }

    /**
     * تحويل حركة أستاذ المورد إلى قيمة موقعة بحسب النوع
     * purchase => +amount
     * return   => -amount
     * payment  => -amount
     */
    protected function signedLedgerAmount(SupplierLedger $entry): int
    {
        $amount = (int) $entry->amount;

        return match ($entry->type) {
            'purchase' => +$amount,
            'return'   => -$amount,
            'payment'  => -$amount,
            default    => +$amount, // كقيمة افتراضية
        };
    }
}
