<?php

namespace App\Services\Purchases\PurchaseInvoice;

use App\Enums\MovementDirection;
use App\Models\Barcode;
use App\Models\Expense;
use App\Models\ExpenseCategory;
use App\Models\PaymentAllocation;
use App\Models\ProductPrice;
use App\Models\ProductVariant;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceItem;
use App\Models\PurchaseInvoiceItemVariant;
use App\Models\PurchaseInvoiceItemVariantAttributeValue;
use App\Models\SupplierLedger;
use App\Models\SupplierPayment;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class PurchaseInvoiceService
{
    public function createInvoice(array $data): PurchaseInvoice
    {
        return DB::transaction(function () use ($data) {
            $itemsData = $data['items'] ?? [];
            unset($data['items']);

            [$itemsData, $subtotal] = $this->normalizeItemsFromOldForm($itemsData);

            $this->computeHeaderTotals($data, $subtotal);
            $this->validateFinancialAmounts($data);

            /** @var PurchaseInvoice $invoice */
            $invoice = PurchaseInvoice::create($data);

            // ✅ حفظ البنود + المخزون + التفريعات
            $this->storeItemsAndStockOldForm($invoice, $itemsData, false);

            // ✅ دفتر المورد + الدفعات
            $this->syncSupplierLedgerAndPayments($invoice);

            // ✅ مصروف أجور النقل
            $this->syncFreightExpense($invoice);

            Log::info('تم إنشاء فاتورة شراء', [
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'supplier_id' => $invoice->supplier_id,
                'total' => $invoice->total,
                'paid' => $invoice->paid,
                'items_count' => count($itemsData),
            ]);

            return $invoice->fresh(['items']);
        });
    }

    public function updateInvoice(PurchaseInvoice $invoice, array $data): PurchaseInvoice
    {
        return DB::transaction(function () use ($invoice, $data) {
            $itemsData = $data['items'] ?? [];
            unset($data['items']);

            [$itemsData, $subtotal] = $this->normalizeItemsFromOldForm($itemsData);

            $this->computeHeaderTotals($data, $subtotal);
            $this->validateFinancialAmounts($data);

            $invoice->fill($data);
            $invoice->save();

            // ✅ حذف تأثيرات قديمة قبل إعادة التخزين
            $invoice->stockMovements()->delete();
            $this->deleteOldPayments($invoice);

            // ✅ إعادة حفظ البنود + الحركات + التسعير + الباركود
            $this->storeItemsAndStockOldForm($invoice, $itemsData, true);

            // ✅ دفتر المورد + الدفعات
            $this->syncSupplierLedgerAndPayments($invoice);

            // ✅ مصروف أجور النقل (حذف القديم + إنشاء الجديد)
            $this->syncFreightExpense($invoice);

            Log::info('تم تحديث فاتورة شراء', [
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'supplier_id' => $invoice->supplier_id,
                'total' => $invoice->total,
                'paid' => $invoice->paid,
                'items_count' => count($itemsData),
            ]);

            return $invoice->fresh(['items']);
        });
    }

    public function deleteInvoice(PurchaseInvoice $invoice): void
    {
        DB::transaction(function () use ($invoice) {
            // ✅ حذف مصروف أجور النقل قبل حذف الفاتورة
            $this->deleteFreightExpense($invoice);

            // ✅ تنظيف البنود وتفريعاتها
            $itemIds = PurchaseInvoiceItem::query()
                ->where('purchase_invoice_id', $invoice->id)
                ->pluck('id')
                ->all();

            if (! empty($itemIds)) {
                $this->deleteItemVariantsByItemIds($itemIds);
                PurchaseInvoiceItem::whereIn('id', $itemIds)->delete();
            }

            $invoice->stockMovements()->delete();

            SupplierLedger::where('ledgerable_type', PurchaseInvoice::class)
                ->where('ledgerable_id', $invoice->id)
                ->delete();

            $this->deleteOldPayments($invoice);

            $invoice->delete();
        });
    }

    /* ============================
     * ✅ تطبيع (الفورم السابق)
     * base_attributes + color_quantities
     * ============================ */

    protected function normalizeItemsFromOldForm(array $itemsData): array
    {
        $subtotal = 0;
        $normalized = [];

        foreach ($itemsData as $item) {
            $hasProduct = ! empty($item['product_id']) || ! empty($item['product_variant_id']);
            if (! $hasProduct) {
                continue;
            }

            $colorRows = $this->normalizeColorQuantities($item['color_quantities'] ?? []);
            $qtyFromColors = $this->sumColorRowsQty($colorRows);

            $mainVariantId = (int) ($item['product_variant_id'] ?? 0);
            if ($mainVariantId <= 0 && ! empty($colorRows)) {
                $mainVariantId = (int) ($colorRows[0]['product_variant_id'] ?? 0);
            }
            if ($mainVariantId <= 0) {
                continue;
            }

            $unitCost = $this->toInt($item['unit_cost'] ?? 0);
            if ($unitCost <= 0) {
                continue;
            }

            $qty = $qtyFromColors > 0
                ? $qtyFromColors
                : $this->toInt($item['quantity_ordered'] ?? 0);

            if ($qty <= 0) {
                continue;
            }

            $lineTotal = $qty * $unitCost;

            $row = [
                'id' => ! empty($item['id']) ? (int) $item['id'] : null,
                'product_id' => (int) ($item['product_id'] ?? 0),
                'product_variant_id' => $mainVariantId,

                'quantity_ordered' => $qty,
                'quantity_received' => $this->toInt($item['quantity_received'] ?? $qty),

                'unit_cost' => $unitCost,
                'line_total' => (int) $lineTotal,

                'notes' => $item['notes'] ?? null,
                'expiry_date' => $item['expiry_date'] ?? null,

                'base_attributes' => is_array($item['base_attributes'] ?? null) ? $item['base_attributes'] : [],
                'color_quantities' => $colorRows,

                'barcode' => $item['barcode'] ?? null,

                'retail_margin' => isset($item['retail_margin']) ? (float) $item['retail_margin'] : 0.0,
                'wholesale_margin' => isset($item['wholesale_margin']) ? (float) $item['wholesale_margin'] : 0.0,
                'agent_margin' => isset($item['agent_margin']) ? (float) $item['agent_margin'] : 0.0,
                'retail_price' => isset($item['retail_price']) ? $this->toInt($item['retail_price']) : 0,
                'wholesale_price' => isset($item['wholesale_price']) ? $this->toInt($item['wholesale_price']) : 0,
                'agent_price' => isset($item['agent_price']) ? $this->toInt($item['agent_price']) : 0,
            ];

            $row = $this->computePricesForRow($row);

            $normalized[] = $row;
            $subtotal += (int) $lineTotal;
        }

        return [$normalized, $subtotal];
    }

    /**
     * color_quantities expected:
     * [
     *  ['product_variant_id'=>.., 'color_value_id'=>.., 'qty'=>..],
     * ]
     */
    protected function normalizeColorQuantities($rows): array
    {
        if (! is_array($rows) || empty($rows)) {
            return [];
        }

        $clean = [];
        foreach ($rows as $r) {
            if (! is_array($r)) {
                continue;
            }

            $variantId = (int) ($r['product_variant_id'] ?? 0);
            $colorVal = (int) ($r['color_value_id'] ?? 0);
            $qty = $this->toInt($r['qty'] ?? 0);

            if ($variantId <= 0 || $qty <= 0) {
                continue;
            }

            $clean[] = [
                'product_variant_id' => $variantId,
                'color_value_id' => $colorVal,
                'qty' => $qty,
            ];
        }

        // دمج نفس variant إذا تكرر
        $grouped = [];
        foreach ($clean as $r) {
            $k = (int) $r['product_variant_id'];
            if (! isset($grouped[$k])) {
                $grouped[$k] = $r;
            } else {
                $grouped[$k]['qty'] += (int) $r['qty'];
            }
        }

        return array_values($grouped);
    }

    protected function sumColorRowsQty(array $rows): int
    {
        $sum = 0;
        foreach ($rows as $r) {
            $sum += (int) ($r['qty'] ?? 0);
        }

        return max(0, $sum);
    }

    /* ============================
     * ✅ التخزين + المخزون (Old Form)
     * ============================ */

    protected function storeItemsAndStockOldForm(PurchaseInvoice $invoice, array $itemsData, bool $isUpdate): void
    {
        $existing = $isUpdate ? $invoice->items()->get()->keyBy('id') : collect();
        $existingIds = $existing->keys();
        $seen = [];

        foreach ($itemsData as $item) {
            $itemId = $item['id'] ?? null;

            $barcode = $item['barcode'] ?? null;
            $baseAttrs = $item['base_attributes'] ?? [];
            $colorRows = $item['color_quantities'] ?? [];

            unset($item['id'], $item['barcode'], $item['base_attributes'], $item['color_quantities']);

            /** @var PurchaseInvoiceItem $invoiceItem */
            if ($isUpdate && $itemId && $existing->has($itemId)) {
                $invoiceItem = $existing->get($itemId);
                $invoiceItem->fill($item);
                $invoiceItem->save();
            } else {
                $item['purchase_invoice_id'] = $invoice->id;
                $invoiceItem = PurchaseInvoiceItem::create($item);
            }

            $seen[] = $invoiceItem->id;

            // ✅ تفريعات السطر: لكل لون Variant row
            $createdVariantRows = $this->syncItemVariantsFromOldForm($invoiceItem, $baseAttrs, $colorRows);

            // ✅ مخزون: حركة لكل لون
            if (! empty($createdVariantRows)) {
                foreach ($createdVariantRows as $row) {
                    $variantId = (int) $row['product_variant_id'];
                    $qty = (int) $row['quantity'];

                    if ($variantId <= 0 || $qty <= 0) {
                        continue;
                    }

                    $invoice->stockMovements()->create([
                        'warehouse_id' => $invoice->warehouse_id,
                        'product_variant_id' => $variantId,
                        'direction' => MovementDirection::IN,
                        'quantity' => $qty,
                        'unit_cost' => (int) $invoiceItem->unit_cost,
                        'total_cost' => (int) ($qty * (int) $invoiceItem->unit_cost),
                        'moved_at' => $invoice->invoice_date,
                        'reference' => $invoice->invoice_number,
                        'note' => 'فاتورة شراء رقم '.$invoice->invoice_number,
                        'user_id' => user_info('id'),
                    ]);

                    ProductVariant::whereKey($variantId)->update(['cost' => (int) $invoiceItem->unit_cost]);
                    $this->syncPriceForVariantId($variantId, $item);
                }
            } else {
                // fallback لو لم تُدخل ألوان
                $invoice->stockMovements()->create([
                    'warehouse_id' => $invoice->warehouse_id,
                    'product_variant_id' => $invoiceItem->product_variant_id,
                    'direction' => MovementDirection::IN,
                    'quantity' => $invoiceItem->quantity_received,
                    'unit_cost' => $invoiceItem->unit_cost,
                    'total_cost' => $invoiceItem->line_total,
                    'moved_at' => $invoice->invoice_date,
                    'reference' => $invoice->invoice_number,
                    'note' => 'فاتورة شراء رقم '.$invoice->invoice_number,
                    'user_id' => user_info('id'),
                ]);

                ProductVariant::whereKey($invoiceItem->product_variant_id)->update(['cost' => $invoiceItem->unit_cost]);
                $this->syncPriceForVariantId((int) $invoiceItem->product_variant_id, $item);
            }

            $this->syncBarcodeForItem($invoice, $invoiceItem, ['barcode' => $barcode]);
        }

        if ($isUpdate) {
            $idsToDelete = $existingIds->diff($seen);

            if ($idsToDelete->isNotEmpty()) {
                // ✅ تنظيف تفريعات البنود المحذوفة (variants + attribute values)
                $this->deleteItemVariantsByItemIds($idsToDelete->values()->all());

                PurchaseInvoiceItem::whereIn('id', $idsToDelete)->delete();
            }
        }
    }

    /**
     * ✅ ينشئ PurchaseInvoiceItemVariant لكل لون + يربط:
     * - base_attributes (قياس/مادة) مرة واحدة لكل تفريعة
     * - color_value_id لكل تفريعة
     *
     * ويرجع rows مبسطة لاستخدامها بالمخزون:
     * [ ['product_variant_id'=>..,'quantity'=>..], ... ]
     */
    protected function syncItemVariantsFromOldForm(PurchaseInvoiceItem $invoiceItem, array $baseAttributes, array $colorRows): array
    {
        // ✅ تنظيف attribute values قبل حذف rows
        $oldVariantIds = PurchaseInvoiceItemVariant::query()
            ->where('purchase_invoice_item_id', $invoiceItem->id)
            ->pluck('id')
            ->all();

        if (! empty($oldVariantIds)) {
            PurchaseInvoiceItemVariantAttributeValue::query()
                ->whereIn('purchase_invoice_item_variant_id', $oldVariantIds)
                ->delete();
        }

        PurchaseInvoiceItemVariant::where('purchase_invoice_item_id', $invoiceItem->id)->delete();

        if (empty($colorRows)) {
            return [];
        }

        $now = now();
        $result = [];

        $basePairs = [];
        foreach ($baseAttributes as $b) {
            if (! is_array($b)) {
                continue;
            }
            $attrId = (int) ($b['attribute_id'] ?? 0);
            $valId = (int) ($b['attribute_value_id'] ?? 0);
            if ($attrId > 0 && $valId > 0) {
                $basePairs[] = ['attribute_id' => $attrId, 'attribute_value_id' => $valId];
            }
        }

        foreach ($colorRows as $c) {
            $variantId = (int) ($c['product_variant_id'] ?? 0);
            $colorVal = (int) ($c['color_value_id'] ?? 0);
            $qty = (int) ($c['qty'] ?? 0);

            if ($variantId <= 0 || $qty <= 0) {
                continue;
            }

            $attrs = $basePairs;

            if ($colorVal > 0) {
                $colorAttrId = $this->guessColorAttributeIdFromVariant($variantId);
                if ($colorAttrId) {
                    $attrs[] = ['attribute_id' => (int) $colorAttrId, 'attribute_value_id' => $colorVal];
                }
            }

            $hash = substr(sha1(json_encode([$variantId, $attrs], JSON_UNESCAPED_UNICODE)), 0, 64);

            $variantRow = PurchaseInvoiceItemVariant::create([
                'purchase_invoice_item_id' => $invoiceItem->id,
                'product_variant_id' => $variantId,
                'quantity' => $qty,
                'options_hash' => $hash,
                'barcode' => null,
                'expiry_date' => $invoiceItem->expiry_date,
                'notes' => null,
            ]);

            $insert = [];
            foreach ($attrs as $a) {
                $insert[] = [
                    'purchase_invoice_item_variant_id' => $variantRow->id,
                    'attribute_id' => (int) $a['attribute_id'],
                    'attribute_value_id' => (int) $a['attribute_value_id'],
                    'created_at' => $now,
                    'updated_at' => $now,
                ];
            }
            if (! empty($insert)) {
                PurchaseInvoiceItemVariantAttributeValue::insert($insert);
            }

            $result[] = [
                'product_variant_id' => $variantId,
                'quantity' => $qty,
            ];
        }

        return $result;
    }

    /**
     * ✅ تنظيف تفريعات البنود المحذوفة.
     */
    protected function deleteItemVariantsByItemIds(array $itemIds): void
    {
        if (empty($itemIds)) {
            return;
        }

        $variantIds = PurchaseInvoiceItemVariant::query()
            ->whereIn('purchase_invoice_item_id', $itemIds)
            ->pluck('id')
            ->all();

        if (! empty($variantIds)) {
            PurchaseInvoiceItemVariantAttributeValue::query()
                ->whereIn('purchase_invoice_item_variant_id', $variantIds)
                ->delete();

            PurchaseInvoiceItemVariant::query()
                ->whereIn('id', $variantIds)
                ->delete();
        }
    }

    /**
     * يحاول استنتاج attribute_id الخاص باللون من variant نفسه (مرة واحدة بالذاكرة)
     */
    protected function guessColorAttributeIdFromVariant(int $variantId): ?int
    {
        static $cached = null;

        if ($cached !== null) {
            return $cached;
        }

        $variant = ProductVariant::with(['attributeValues' => function ($q) {
            $q->select(['attribute_values.id', 'attribute_values.attribute_id']);
        }])->find($variantId);

        if (! $variant) {
            return null;
        }

        $attrId = $variant->attributeValues->first()?->attribute_id;

        $cached = $attrId ? (int) $attrId : null;

        return $cached;
    }

    /* ============================
     * Prices + Barcode
     * ============================ */

    protected function syncPriceForVariantId(int $productVariantId, array $item): void
    {
        $variant = ProductVariant::withTrashed()->find($productVariantId);
        if (! $variant) {
            return;
        }

        $productId = $variant->product_id;

        $hasAnyPriceOrMargin =
            (($item['retail_price'] ?? 0) > 0 ||
                ($item['wholesale_price'] ?? 0) > 0 ||
                ($item['agent_price'] ?? 0) > 0 ||
                ($item['retail_margin'] ?? 0) > 0.0 ||
                ($item['wholesale_margin'] ?? 0) > 0.0 ||
                ($item['agent_margin'] ?? 0) > 0.0);

        if (! $hasAnyPriceOrMargin) {
            return;
        }

        ProductPrice::updateOrCreate(
            [
                'product_id' => $productId,
                'product_variant_id' => $productVariantId,
            ],
            [
                'retail_margin' => (float) ($item['retail_margin'] ?? 0),
                'wholesale_margin' => (float) ($item['wholesale_margin'] ?? 0),
                'agent_margin' => (float) ($item['agent_margin'] ?? 0),
                'retail_price' => (int) ($item['retail_price'] ?? 0),
                'wholesale_price' => (int) ($item['wholesale_price'] ?? 0),
                'agent_price' => (int) ($item['agent_price'] ?? 0),
                'status' => true,
            ]
        );
    }

    protected function syncBarcodeForItem(PurchaseInvoice $invoice, PurchaseInvoiceItem $invoiceItem, array $item): void
    {
        $rawCode = $item['barcode'] ?? null;
        if (! $rawCode) {
            return;
        }

        $code = preg_replace('/\D/', '', (string) $rawCode);
        if ($code === '' || strlen($code) !== 13) {
            return;
        }

        if (! Barcode::validateEAN13($code)) {
            throw ValidationException::withMessages([
                'items' => ['الباركود "'.$code.'" ليس EAN-13 صحيحاً.'],
            ]);
        }

        $existing = Barcode::where('code', $code)->first();

        if ($existing) {
            if ((int) $existing->product_variant_id === (int) $invoiceItem->product_variant_id) {
                return;
            }

            throw ValidationException::withMessages([
                'items' => ['الباركود "'.$code.'" مستخدم مسبقاً لمنتج/متغير آخر.'],
            ]);
        }

        Barcode::create([
            'product_variant_id' => $invoiceItem->product_variant_id,
            'code' => $code,
            'is_primary' => ! Barcode::where('product_variant_id', $invoiceItem->product_variant_id)->exists(),
            'source' => Barcode::SOURCE_VENDOR,
            'is_active' => true,
            'user_id' => user_info('id'),
        ]);
    }

    /* ============================
     * Ledger + Payments
     * ============================ */

    protected function syncSupplierLedgerAndPayments(PurchaseInvoice $invoice): void
    {
        SupplierLedger::where('ledgerable_type', PurchaseInvoice::class)
            ->where('ledgerable_id', $invoice->id)
            ->delete();

        SupplierLedger::create([
            'supplier_id' => $invoice->supplier_id,
            'branch_id' => user_info('branch_id'),
            'ledgerable_type' => PurchaseInvoice::class,
            'ledgerable_id' => $invoice->id,
            'transaction_date' => $invoice->invoice_date,
            'type' => 'purchase',
            'amount' => $invoice->total,
            'reference' => $invoice->invoice_number,
            'description' => 'فاتورة شراء رقم '.$invoice->invoice_number,
        ]);

        if ($invoice->paid > 0) {
            $payment = SupplierPayment::create([
                'payment_number' => $this->generatePaymentNumber(),
                'supplier_id' => $invoice->supplier_id,
                'branch_id' => user_info('branch_id'),
                'user_id' => user_info('id'),
                'payment_date' => $invoice->invoice_date,
                'amount' => $invoice->paid,
                'payment_method' => 'cash',
                'reference_number' => $invoice->invoice_number,
                'notes' => 'دفعة مرتبطة بفاتورة شراء #'.$invoice->invoice_number,
            ]);

            PaymentAllocation::create([
                'supplier_payment_id' => $payment->id,
                'purchase_invoice_id' => $invoice->id,
                'allocated_amount' => $invoice->paid,
            ]);

            SupplierLedger::create([
                'supplier_id' => $invoice->supplier_id,
                'branch_id' => user_info('branch_id'),
                'ledgerable_type' => SupplierPayment::class,
                'ledgerable_id' => $payment->id,
                'transaction_date' => $payment->payment_date,
                'type' => 'payment',
                'amount' => $invoice->paid,
                'reference' => $payment->payment_number,
                'description' => 'سداد فاتورة شراء #'.$invoice->invoice_number,
            ]);
        }
    }

    protected function deleteOldPayments(PurchaseInvoice $invoice): void
    {
        $allocations = PaymentAllocation::where('purchase_invoice_id', $invoice->id)->get();

        foreach ($allocations as $alloc) {
            $paymentId = $alloc->supplier_payment_id;
            $alloc->delete();

            if ($paymentId) {
                SupplierLedger::where('ledgerable_type', SupplierPayment::class)
                    ->where('ledgerable_id', $paymentId)
                    ->delete();

                SupplierPayment::whereKey($paymentId)->delete();
            }
        }
    }

    protected function generatePaymentNumber(): string
    {
        $prefix = 'SP-'.now()->format('Ymd').'-';

        $seq = SupplierPayment::withTrashed()
            ->whereDate('payment_date', now()->toDateString())
            ->count() + 1;

        return $prefix.str_pad((string) $seq, 4, '0', STR_PAD_LEFT);
    }

    /* ============================
     * Totals + validation helpers
     * ============================ */

    protected function computeHeaderTotals(array &$data, int $subtotal): void
    {
        $data['subtotal'] = (int) $subtotal;

        $discount = (int) ($data['discount'] ?? 0);
        $freight = (int) ($data['freight'] ?? 0); // فقط تسجيل
        $paid = (int) ($data['paid'] ?? 0);

        // ✅ freight لا يدخل بالحساب
        $total = max(0, $subtotal - $discount);

        $data['discount'] = $discount;
        $data['freight'] = $freight;
        $data['total'] = $total;
        $data['paid'] = $paid;
        $data['due'] = max(0, $total - $paid);

        if ($total <= 0) {
            $data['payment_status'] = 'unpaid';
        } elseif ($paid >= $total) {
            $data['payment_status'] = 'paid';
        } elseif ($paid > 0) {
            $data['payment_status'] = 'partial';
        } else {
            $data['payment_status'] = 'unpaid';
        }
    }

    protected function validateFinancialAmounts(array $data): void
    {
        $total = (int) ($data['total'] ?? 0);
        $paid = (int) ($data['paid'] ?? 0);
        $subtotal = (int) ($data['subtotal'] ?? 0);
        $discount = (int) ($data['discount'] ?? 0);

        $errors = [];

        if ($paid < 0) {
            $errors['paid'][] = 'المبلغ المدفوع لا يمكن أن يكون سالباً.';
        }
        if ($paid > $total && $total > 0) {
            $errors['paid'][] = 'المبلغ المدفوع لا يمكن أن يتجاوز إجمالي الفاتورة.';
        }
        if ($discount > $subtotal && $subtotal > 0) {
            $errors['discount'][] = 'الخصم لا يمكن أن يتجاوز المجموع.';
        }
        if ($subtotal <= 0) {
            $errors['items'][] = 'يجب إضافة عنصر واحد على الأقل.';
        }

        if (! empty($errors)) {
            throw ValidationException::withMessages($errors);
        }
    }

    protected function computePricesForRow(array $row): array
    {
        $cost = (int) ($row['unit_cost'] ?? 0);
        if ($cost <= 0) {
            return $row;
        }

        $retailPrice = (int) ($row['retail_price'] ?? 0);
        $retailMargin = (float) ($row['retail_margin'] ?? 0);

        if ($retailMargin > 0 && $retailPrice <= 0) {
            $retailPrice = (int) round($cost * (1 + $retailMargin / 100));
        } elseif ($retailPrice > 0 && $retailMargin <= 0) {
            $retailMargin = $cost > 0 ? (($retailPrice - $cost) / $cost) * 100 : 0;
        }

        $wholesalePrice = (int) ($row['wholesale_price'] ?? 0);
        $wholesaleMargin = (float) ($row['wholesale_margin'] ?? 0);

        if ($wholesaleMargin > 0 && $wholesalePrice <= 0) {
            $wholesalePrice = (int) round($cost * (1 + $wholesaleMargin / 100));
        } elseif ($wholesalePrice > 0 && $wholesaleMargin <= 0) {
            $wholesaleMargin = $cost > 0 ? (($wholesalePrice - $cost) / $cost) * 100 : 0;
        }

        $agentPrice = (int) ($row['agent_price'] ?? 0);
        $agentMargin = (float) ($row['agent_margin'] ?? 0);

        if ($agentMargin > 0 && $agentPrice <= 0) {
            $agentPrice = (int) round($cost * (1 + $agentMargin / 100));
        } elseif ($agentPrice > 0 && $agentMargin <= 0) {
            $agentMargin = $cost > 0 ? (($agentPrice - $cost) / $cost) * 100 : 0;
        }

        $row['retail_price'] = $retailPrice;
        $row['retail_margin'] = $retailMargin;
        $row['wholesale_price'] = $wholesalePrice;
        $row['wholesale_margin'] = $wholesaleMargin;
        $row['agent_price'] = $agentPrice;
        $row['agent_margin'] = $agentMargin;

        return $row;
    }

    /* ============================
     * Freight Expense
     * ============================ */

    protected function ensureFreightCategoryId(int $branchId): int
    {
        $cat = ExpenseCategory::query()->firstOrCreate(
            ['branch_id' => $branchId, 'name' => 'أجور النقل'],
            ['is_active' => true]
        );

        return (int) $cat->id;
    }

    protected function syncFreightExpense(PurchaseInvoice $invoice): void
    {
        $freight = (int) ($invoice->freight ?? 0);

        $branchId = (int) ($invoice->branch_id ?: user_info('branch_id'));
        $catId = $this->ensureFreightCategoryId($branchId);

        $desc = 'أجور نقل لفاتورة شراء رقم '.$invoice->invoice_number;

        // ✅ احذف أي مصروف سابق لنفس الفاتورة حتى ما يتكرر
        Expense::query()
            ->where('branch_id', $branchId)
            ->where('expense_category_id', $catId)
            ->where('description', $desc)
            ->delete();

        if ($freight <= 0) {
            return; // ماكو أجور نقل
        }

        Expense::create([
            'branch_id' => $branchId,
            'expense_category_id' => $catId,
            'user_id' => (int) user_info('id'),
            'amount' => $freight,
            'description' => $desc,
            'spent_at' => $invoice->invoice_date,
        ]);
    }

    protected function deleteFreightExpense(PurchaseInvoice $invoice): void
    {
        $branchId = (int) ($invoice->branch_id ?: user_info('branch_id'));

        $catId = ExpenseCategory::query()
            ->where('branch_id', $branchId)
            ->where('name', 'أجور النقل')
            ->value('id');

        if (! $catId) {
            return;
        }

        $desc = 'أجور نقل لفاتورة شراء رقم '.$invoice->invoice_number;

        Expense::query()
            ->where('branch_id', $branchId)
            ->where('expense_category_id', (int) $catId)
            ->where('description', $desc)
            ->delete();
    }

    /* ============================
     * Helpers
     * ============================ */

    protected function toInt($value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }
        if (is_int($value)) {
            return $value;
        }
        if (is_numeric($value)) {
            return (int) $value;
        }

        $cleaned = preg_replace('/[^\d٠-٩]/u', '', (string) $value);
        $cleaned = preg_replace_callback('/[٠-٩]/u', fn ($m) => mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]), $cleaned);

        return (int) ($cleaned === '' ? 0 : $cleaned);
    }
}
