<?php

namespace App\Services\Purchases\PurchaseInvoice;

use App\Models\Expense;
use App\Models\ExpenseCategory;
use App\Models\PaymentAllocation;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceItem;
use App\Models\PurchaseInvoiceItemVariant;
use App\Models\PurchaseInvoiceItemVariantAttributeValue;
use App\Models\SupplierLedger;
use App\Models\SupplierPayment;
use Illuminate\Support\Facades\DB;

class DeletePurchaseInvoiceService
{
    public function handle(PurchaseInvoice $invoice): void
    {
        DB::transaction(function () use ($invoice) {

            // ✅ حذف مصروف أجور النقل المرتبط بهذه الفاتورة
            $this->deleteFreightExpense($invoice);

            // ✅ تنظيف البنود + تفريعاتها (variants + attribute values) قبل حذف البنود
            $itemIds = PurchaseInvoiceItem::query()
                ->where('purchase_invoice_id', $invoice->id)
                ->pluck('id')
                ->all();

            if (! empty($itemIds)) {
                $this->deleteItemVariantsByItemIds($itemIds);

                // حذف البنود
                PurchaseInvoiceItem::whereIn('id', $itemIds)->delete();
            } else {
                // fallback (بس غالباً ما يوصل هنا)
                PurchaseInvoiceItem::where('purchase_invoice_id', $invoice->id)->delete();
            }

            // حذف حركات المخزون المرتبطة
            $invoice->stockMovements()->delete();

            // حذف قيود دفتر المورد المرتبطة بالفاتورة
            SupplierLedger::where('ledgerable_type', PurchaseInvoice::class)
                ->where('ledgerable_id', $invoice->id)
                ->delete();

            // حذف الدفعات والتخصيصات المرتبطة
            $this->deleteOldPayments($invoice);

            // حذف الفاتورة
            $invoice->delete();
        });
    }

    /**
     * ✅ حذف تفريعات البنود المحذوفة.
     */
    protected function deleteItemVariantsByItemIds(array $itemIds): void
    {
        if (empty($itemIds)) {
            return;
        }

        $variantIds = PurchaseInvoiceItemVariant::query()
            ->whereIn('purchase_invoice_item_id', $itemIds)
            ->pluck('id')
            ->all();

        if (! empty($variantIds)) {
            PurchaseInvoiceItemVariantAttributeValue::query()
                ->whereIn('purchase_invoice_item_variant_id', $variantIds)
                ->delete();

            PurchaseInvoiceItemVariant::query()
                ->whereIn('id', $variantIds)
                ->delete();
        }
    }

    /**
     * ✅ حذف الدفعات والتخصيصات المرتبطة بالفاتورة (كما قبل)
     */
    protected function deleteOldPayments(PurchaseInvoice $invoice): void
    {
        $allocations = PaymentAllocation::where('purchase_invoice_id', $invoice->id)->get();

        foreach ($allocations as $alloc) {
            $paymentId = $alloc->supplier_payment_id;
            $alloc->delete();

            if ($paymentId) {
                SupplierLedger::where('ledgerable_type', SupplierPayment::class)
                    ->where('ledgerable_id', $paymentId)
                    ->delete();

                SupplierPayment::whereKey($paymentId)->delete();
            }
        }
    }

    /**
     * ✅ حذف مصروف أجور النقل لهذه الفاتورة
     */
    protected function deleteFreightExpense(PurchaseInvoice $invoice): void
    {
        $branchId = (int) ($invoice->branch_id ?: user_info('branch_id'));

        $catId = ExpenseCategory::query()
            ->where('branch_id', $branchId)
            ->where('name', 'أجور النقل')
            ->value('id');

        if (! $catId) {
            return;
        }

        $desc = 'أجور نقل لفاتورة شراء رقم '.$invoice->invoice_number;

        Expense::query()
            ->where('branch_id', $branchId)
            ->where('expense_category_id', (int) $catId)
            ->where('description', $desc)
            ->delete();
    }
}
