<?php

namespace App\Services\Products;

use App\Models\Barcode;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\ProductVariant;
use App\Models\StockBalance;
use App\Models\StockMovement;
use Illuminate\Support\Facades\DB;

class ProductService
{
    /* ===================== أدوات رقمية (أعداد صحيحة) ===================== */

    protected function toInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        if (is_int($value)) {
            return $value;
        }

        $s = (string) $value;

        // أرقام عربية/فارسية -> إنجليزية
        $s = strtr($s, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        // حذف أي شيء غير رقم (يشيل الفواصل)
        $s = preg_replace('/[^\d]/u', '', $s);

        return (int) ($s ?: 0);
    }

    protected function toFloat(mixed $value): float
    {
        if ($value === null || $value === '') {
            return 0.0;
        }
        if (is_float($value) || is_int($value)) {
            return (float) $value;
        }

        $s = (string) $value;
        $s = strtr($s, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        // السماح بالنقطة فقط
        $s = preg_replace('/[^\d.]/u', '', $s);

        return (float) ($s ?: 0);
    }

    /* ===================== Public API ===================== */

    /**
     * إنشاء أو تحديث منتج مع كامل بياناته (البيانات الأساسية + النسخ + التسعير + المخزون).
     */
    public function createOrUpdate(?Product $product, array $data): Product
    {
        return DB::transaction(function () use ($product, $data) {
            $isNew = $product === null;

            // 1) حفظ / تحديث المنتج الأساسي
            $product = $this->saveProduct($product, $data);

            // 2) معالجة النسخ
            if ($product->is_variable) {
                $this->processVariableProduct($product, $data, $isNew);
            } else {
                $this->processSimpleProduct($product, $data, $isNew);
            }

            return $product->fresh();
        });
    }

    /* ===================== Core ===================== */

    /**
     * حفظ / تحديث بيانات المنتج الأساسية.
     */
    protected function saveProduct(?Product $product, array $data): Product
    {
        $productData = [
            'name' => $data['name'] ?? '',
            'description' => $data['description'] ?? null,
            'category_id' => $data['category_id'] ?? null,
            'brand_id' => $data['brand_id'] ?? null,
            'unit' => $data['unit'] ?? null,
            'is_variable' => (bool) ($data['is_variable'] ?? false),
            'is_active' => (bool) ($data['is_active'] ?? true),
        ];

        if ($product) {
            $product->update($productData);
        } else {
            $product = Product::create($productData);
        }

        return $product;
    }

    /**
     * معالجة منتج بسيط (نسخة واحدة فقط - default).
     */
    protected function processSimpleProduct(Product $product, array $data, bool $isNew): void
    {
        // الحصول على النسخة الافتراضية أو إنشاؤها
        $variant = $product->variants()->where('is_default', true)->first();

        if (! $variant) {
            $variant = new ProductVariant([
                'product_id' => $product->id,
                'is_default' => true,
                'is_active' => true,
            ]);
        }

        // تحديث بيانات النسخة (بدون cost - تم نقله إلى stock_balances)
        $variant->fill([
            'sku' => $data['simple_sku'] ?? null,
            'is_active' => (bool) ($data['simple_is_active'] ?? true),
        ]);

        $variant->save();

        // معالجة الباركود
        if (! empty($data['simple_manual_barcode'])) {
            $this->processBarcodeForVariant($variant, (string) $data['simple_manual_barcode']);
        }

        // ✅ التسعير (int + float)
        $this->processPricing($product, $variant, [
            'retail_margin' => $this->toFloat($data['simple_retail_margin'] ?? 0),
            'retail_price' => $this->toInt($data['simple_retail_price'] ?? 0),
            'wholesale_margin' => $this->toFloat($data['simple_wholesale_margin'] ?? 0),
            'wholesale_price' => $this->toInt($data['simple_wholesale_price'] ?? 0),
            'agent_margin' => $this->toFloat($data['simple_agent_margin'] ?? 0),
            'agent_price' => $this->toInt($data['simple_agent_price'] ?? 0),
            'status' => (bool) ($data['simple_pricing_status'] ?? true),
        ]);

        // ✅ المخزون
        $warehouseId = (int) ($data['simple_warehouse_id'] ?? config('inventory.default_warehouse_id', 1));
        $initialStock = $this->toInt($data['simple_initial_stock'] ?? 0);
        $unitCost = $this->toInt($data['simple_initial_unit_cost'] ?? 0);

        $this->processStockBalance(
            $variant,
            $warehouseId,
            $initialStock,
            $unitCost,
            (int) ($data['simple_min_stock_level'] ?? 0),
            (int) ($data['simple_max_stock_level'] ?? 0),
            (int) ($data['simple_reorder_point'] ?? 0),
            (int) ($data['simple_reorder_quantity'] ?? 0),
            $isNew // ✅ مهم
        );
    }

    /**
     * معالجة منتج متغير (نسخ متعددة).
     * ملاحظة مهمة:
     * - الفورم عندك مرتبط بـ variableVariants (يعني لا يرسل default/base variant).
     * - لذلك هنا لا نحذف النسخة الافتراضية أبداً.
     */
    protected function processVariableProduct(Product $product, array $data, bool $isNew): void
    {
        $variants = $data['variants'] ?? null;
        if (empty($variants) || ! is_array($variants)) {
            // حتى لو لم يرسل variants، لا نحذف شيء
            return;
        }

        $incomingVariantIds = [];

        foreach ($variants as $variantData) {
            $variant = null;
            $variantIsNew = true;

            if (! empty($variantData['id'])) {
                $variant = ProductVariant::query()->find($variantData['id']);

                if ($variant && (int) $variant->product_id === (int) $product->id) {
                    $variantIsNew = false;
                } else {
                    $variant = null;
                }
            }

            if (! $variant) {
                $variant = new ProductVariant(['product_id' => $product->id]);
                $variantIsNew = true;
            }

            // بيانات النسخة الأساسية
            $variant->fill([
                'sku' => $variantData['sku'] ?? null,
                'is_active' => (bool) ($variantData['is_active'] ?? true),
                // لا نجعل نسخة من الواجهة "default" إلا إذا فعلاً تستخدمها (عادةً لا)
                'is_default' => (bool) ($variantData['is_default'] ?? false),
            ]);

            $variant->save();

            $incomingVariantIds[] = $variant->id;

            // الباركود
            if (! empty($variantData['manual_barcode'])) {
                $this->processBarcodeForVariant($variant, (string) $variantData['manual_barcode']);
            }

            // الخصائص
            $this->processVariantAttributes($variant, $variantData);

            // ✅ التسعير (int + float)
            $this->processPricing($product, $variant, [
                'retail_margin' => $this->toFloat($variantData['retail_margin'] ?? 0),
                'retail_price' => $this->toInt($variantData['retail_price'] ?? 0),
                'wholesale_margin' => $this->toFloat($variantData['wholesale_margin'] ?? 0),
                'wholesale_price' => $this->toInt($variantData['wholesale_price'] ?? 0),
                'agent_margin' => $this->toFloat($variantData['agent_margin'] ?? 0),
                'agent_price' => $this->toInt($variantData['agent_price'] ?? 0),
                'status' => (bool) ($variantData['pricing_status'] ?? true),
            ]);

            // ✅ المخزون (للنسخ الجديدة: حركة افتتاحية، وللقديم: تحديث الحدود + last_unit_cost)
            $warehouseId = (int) ($variantData['warehouse_id'] ?? config('inventory.default_warehouse_id', 1));
            $initialStock = $this->toInt($variantData['initial_stock'] ?? 0);
            $unitCost = $this->toInt($variantData['initial_unit_cost'] ?? 0);

            $this->processStockBalance(
                $variant,
                $warehouseId,
                $initialStock,
                $unitCost,
                (int) ($variantData['min_stock_level'] ?? 0),
                (int) ($variantData['max_stock_level'] ?? 0),
                (int) ($variantData['reorder_point'] ?? 0),
                (int) ($variantData['reorder_quantity'] ?? 0),
                $variantIsNew // ✅ مهم
            );
        }

        // ✅ حذف النسخ التي لم تعد موجودة في النموذج (ولكن لا نحذف default)
        // لأن الفورم لا يرسل default variant، فلابد نستثني is_default=true
        $product->variants()
            ->where('is_default', false)
            ->when(! empty($incomingVariantIds), fn ($q) => $q->whereNotIn('id', $incomingVariantIds))
            ->delete();
    }

    /**
     * معالجة المخزون:
     * - إنشاء/تحديث StockBalance
     * - إنشاء حركة افتتاحية فقط عند إنشاء الرصيد لأول مرة (للنسخة الجديدة)
     * - تحديث last_unit_cost عند التعديل إذا تم إدخال unitCost
     */
    protected function processStockBalance(
        ProductVariant $variant,
        int $warehouseId,
        int $initialStock,
        int $unitCost,
        int $minStockLevel,
        int $maxStockLevel = 0,
        int $reorderPoint = 0,
        int $reorderQuantity = 0,
        bool $isNewEntity = false
    ): void {
        $stockBalance = StockBalance::firstOrCreate(
            [
                'warehouse_id' => $warehouseId,
                'product_variant_id' => $variant->id,
            ],
            [
                'on_hand' => 0,
                'reserved_qty' => 0,
                'min_stock_level' => $minStockLevel,
                'max_stock_level' => $maxStockLevel,
                'reorder_point' => $reorderPoint,
                'reorder_quantity' => $reorderQuantity,
                'last_unit_cost' => null,
                'last_movement_at' => null,
            ]
        );

        // تحديث الحدود دائماً
        $stockBalance->update([
            'min_stock_level' => $minStockLevel,
            'max_stock_level' => $maxStockLevel,
            'reorder_point' => $reorderPoint,
            'reorder_quantity' => $reorderQuantity,
        ]);

        // ✅ تحديث تكلفة آخر وحدة إذا تم إدخالها (بدون حركة)
        if ($unitCost > 0) {
            $stockBalance->update([
                'last_unit_cost' => $unitCost,
            ]);
        }

        /**
         * حركة افتتاحية:
         * - فقط عندما يكون هذا السطر تم إنشاؤه الآن + الكمية > 0
         * - عادةً هذا يحدث للنسخ الجديدة فقط
         */
        if ($initialStock > 0 && $stockBalance->wasRecentlyCreated && $isNewEntity) {
            StockMovement::create([
                'warehouse_id' => $warehouseId,
                'product_variant_id' => $variant->id,
                'direction' => 'IN',
                'source' => 'DIRECT',
                'quantity' => $initialStock,
                'unit_cost' => $unitCost,
                'moved_at' => now(),
                'note' => 'رصيد افتتاحي عند إنشاء المنتج',
                'user_id' => user_info('id'),
            ]);
        }
    }

    /**
     * معالجة خصائص النسخة (Attributes).
     */
    protected function processVariantAttributes(ProductVariant $variant, array $variantData): void
    {
        $attributeValueIds = [];

        // (أ) عند الإنشاء: variant_attributes
        if (! empty($variantData['variant_attributes']) && is_array($variantData['variant_attributes'])) {
            foreach ($variantData['variant_attributes'] as $attr) {
                if (! empty($attr['attribute_value_id'])) {
                    $attributeValueIds[(int) $attr['attribute_value_id']] = [
                        'attribute_id' => $attr['attribute_id'] ?? null,
                    ];
                }
            }
        }

        // (ب) عند التعديل: variant_attributes_rel (relationship attributeValues)
        if (! empty($variantData['variant_attributes_rel']) && is_array($variantData['variant_attributes_rel'])) {
            foreach ($variantData['variant_attributes_rel'] as $attr) {
                if (! empty($attr['id'])) {
                    $attributeValueIds[(int) $attr['id']] = [
                        'attribute_id' => $attr['attribute_id'] ?? null,
                    ];
                }
            }
        }

        if (! empty($attributeValueIds)) {
            $variant->attributeValues()->sync($attributeValueIds);
        }
    }

    /**
     * معالجة الباركود للنسخة.
     */
    protected function processBarcodeForVariant(ProductVariant $variant, string $barcodeCode): void
    {
        $barcodeCode = trim($barcodeCode);
        if ($barcodeCode === '') {
            return;
        }

        $primaryBarcode = $variant->barcodes()->primary()->first();

        if ($primaryBarcode) {
            if ($primaryBarcode->code !== $barcodeCode) {
                $primaryBarcode->update(['code' => $barcodeCode]);
            }
        } else {
            Barcode::create([
                'product_variant_id' => $variant->id,
                'code' => $barcodeCode,
                'is_primary' => true,
                'is_active' => true,
                'source' => Barcode::SOURCE_INTERNAL,
                'user_id' => user_info('id'),
            ]);
        }
    }

    /**
     * معالجة التسعير (Upsert ProductPrice).
     */
    protected function processPricing(Product $product, ProductVariant $variant, array $pricingData): void
    {
        // البحث عن سعر موجود للنسخة
        $price = ProductPrice::query()
            ->where('product_id', $product->id)
            ->where('product_variant_id', $variant->id)
            ->first();

        // إذا لم يوجد، نبحث عن سعر عام للمنتج ونستنسخه
        if (! $price) {
            $price = new ProductPrice([
                'product_id' => $product->id,
                'product_variant_id' => $variant->id,
            ]);
        }

        // تحضير البيانات
        $retailMargin = $this->toFloat($pricingData['retail_margin'] ?? 0);
        $retailPrice = $this->toInt($pricingData['retail_price'] ?? 0);
        $wholesaleMargin = $this->toFloat($pricingData['wholesale_margin'] ?? 0);
        $wholesalePrice = $this->toInt($pricingData['wholesale_price'] ?? 0);
        $agentMargin = $this->toFloat($pricingData['agent_margin'] ?? 0);
        $agentPrice = $this->toInt($pricingData['agent_price'] ?? 0);

        // حفظ البيانات
        $price->fill([
            'retail_margin' => $retailMargin,
            'retail_price' => $retailPrice,
            'wholesale_margin' => $wholesaleMargin,
            'wholesale_price' => $wholesalePrice,
            'agent_margin' => $agentMargin,
            'agent_price' => $agentPrice,
            'status' => (bool) ($pricingData['status'] ?? true),
        ]);

        $price->save();
    }

    /**
     * حذف منتج مع كل بياناته المرتبطة.
     */
    public function deleteProduct(Product $product, bool $force = false): bool
    {
        return DB::transaction(function () use ($product, $force) {
            ProductPrice::where('product_id', $product->id)->delete();

            foreach ($product->variants as $variant) {
                $variant->barcodes()->delete();
                $variant->attributeValues()->detach();

                ProductPrice::where('product_variant_id', $variant->id)->delete();
                StockBalance::where('product_variant_id', $variant->id)->delete();
                StockMovement::where('product_variant_id', $variant->id)->delete();

                $force ? $variant->forceDelete() : $variant->delete();
            }

            return $force ? $product->forceDelete() : $product->delete();
        });
    }
}
