<?php

namespace App\Services\Payments;

use App\Models\PaymentAllocation;
use App\Models\PurchaseInvoice;
use App\Models\SupplierLedger;
use App\Models\SupplierPayment;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SupplierPaymentService
{
    // ══════════════════════════════════════════════════════════════
    //  إنشاء دفعة جديدة
    // ══════════════════════════════════════════════════════════════

    public function create(array $data): SupplierPayment
    {
        return DB::transaction(function () use ($data) {
            // تنظيف المبلغ - مهم جداً
            if (isset($data['amount'])) {
                $data['amount'] = $this->cleanAmount($data['amount']);
            }

            if (($data['amount'] ?? 0) <= 0) {
                throw new \InvalidArgumentException('المبلغ يجب أن يكون أكبر من صفر');
            }

            $data['payment_number'] = $this->generatePaymentNumber();
            $data['user_id'] = user_info('id');
            $data['branch_id'] = $data['branch_id'] ?? user_info('branch_id');

            $payment = SupplierPayment::create($data);

            $this->allocatePayment($payment);
            $this->createLedgerEntry($payment);

            return $payment;
        });
    }

    protected function cleanAmount(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        if (is_int($value)) {
            return $value;
        }

        $cleaned = str_replace([',', ' ', '،'], '', (string) $value);

        $cleaned = strtr($cleaned, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        return (int) $cleaned;
    }

    // ══════════════════════════════════════════════════════════════
    //  تعديل دفعة
    // ══════════════════════════════════════════════════════════════

    public function update(SupplierPayment $payment, array $data): SupplierPayment
    {
        return DB::transaction(function () use ($payment, $data) {
            $oldAmount     = $payment->amount;
            $oldSupplierId = $payment->supplier_id;

            // تنظيف المبلغ لو تم إرساله كنص
            if (isset($data['amount'])) {
                $data['amount'] = $this->cleanAmount($data['amount']);
            }

            if (isset($data['amount']) && $data['amount'] <= 0) {
                throw new \InvalidArgumentException('المبلغ يجب أن يكون أكبر من صفر');
            }

            // إلغاء التوزيعات السابقة
            $this->reverseAllocations($payment);

            // حذف قيد دفتر الأستاذ القديم
            $payment->ledgerEntries()->delete();

            // تحديث الدفعة
            $payment->update($data);
            $payment->refresh();

            // إعادة توزيع المبلغ
            $this->allocatePayment($payment);

            // إنشاء قيد جديد
            $this->createLedgerEntry($payment);

            // إذا تغير المورد، نحدث فواتير المورد القديم
            if ($oldSupplierId !== $payment->supplier_id) {
                $this->recalculateSupplierInvoices($oldSupplierId);
            }

            Log::info('تم تعديل دفعة مورد', [
                'payment_id' => $payment->id,
                'old_amount' => $oldAmount,
                'new_amount' => $payment->amount,
            ]);

            return $payment;
        });
    }

    // ══════════════════════════════════════════════════════════════
    //  حذف دفعة
    // ══════════════════════════════════════════════════════════════

    public function delete(SupplierPayment $payment): bool
    {
        return DB::transaction(function () use ($payment) {
            $supplierId = $payment->supplier_id;

            // إلغاء التوزيعات
            $this->reverseAllocations($payment);

            // حذف قيود دفتر الأستاذ
            $payment->ledgerEntries()->delete();

            // حذف التخصيصات
            $payment->allocations()->delete();

            // حذف الدفعة
            $payment->forceDelete();

            Log::info('تم حذف دفعة مورد', [
                'payment_id'  => $payment->id,
                'supplier_id' => $supplierId,
            ]);

            return true;
        });
    }

    // ══════════════════════════════════════════════════════════════
    //  توزيع المبلغ على الفواتير (FIFO - الأقدم أولاً)
    // ══════════════════════════════════════════════════════════════

    protected function allocatePayment(SupplierPayment $payment): void
    {
        $remainingAmount = $payment->amount;

        $invoices = PurchaseInvoice::where('supplier_id', $payment->supplier_id)
            ->whereIn('payment_status', ['unpaid', 'partial'])
            ->orderBy('invoice_date')
            ->orderBy('id')
            ->get();

        foreach ($invoices as $invoice) {
            if ($remainingAmount <= 0) {
                break;
            }

            $invoiceDue = $invoice->total - $invoice->paid;

            if ($invoiceDue <= 0) {
                continue;
            }

            $allocationAmount = min($remainingAmount, $invoiceDue);

            PaymentAllocation::create([
                'supplier_payment_id' => $payment->id,
                'purchase_invoice_id' => $invoice->id,
                'allocated_amount'    => $allocationAmount,
            ]);

            $this->updateInvoicePayment($invoice, $allocationAmount);

            $remainingAmount -= $allocationAmount;
        }
    }

    // ══════════════════════════════════════════════════════════════
    //  إلغاء التوزيعات
    // ══════════════════════════════════════════════════════════════

    protected function reverseAllocations(SupplierPayment $payment): void
    {
        $allocations = $payment->allocations()->with('invoice')->get();

        foreach ($allocations as $allocation) {
            $invoice = $allocation->invoice;

            if ($invoice) {
                $newPaid = max(0, $invoice->paid - $allocation->allocated_amount);
                $invoice->update([
                    'paid'           => $newPaid,
                    'due'            => $invoice->total - $newPaid,
                    'payment_status' => $this->determinePaymentStatus($invoice->total, $newPaid),
                ]);
            }
        }

        $payment->allocations()->delete();
    }

    // ══════════════════════════════════════════════════════════════
    //  تحديث الفاتورة بعد الدفع
    // ══════════════════════════════════════════════════════════════

    protected function updateInvoicePayment(PurchaseInvoice $invoice, int $amount): void
    {
        $newPaid = $invoice->paid + $amount;
        $newDue  = max(0, $invoice->total - $newPaid);

        $invoice->update([
            'paid'           => $newPaid,
            'due'            => $newDue,
            'payment_status' => $this->determinePaymentStatus($invoice->total, $newPaid),
        ]);
    }

    protected function determinePaymentStatus(int $total, int $paid): string
    {
        if ($paid <= 0) {
            return 'unpaid';
        }

        if ($paid >= $total) {
            return 'paid';
        }

        return 'partial';
    }

    // ══════════════════════════════════════════════════════════════
    //  إعادة حساب فواتير المورد
    // ══════════════════════════════════════════════════════════════

    protected function recalculateSupplierInvoices(int $supplierId): void
    {
        $invoices = PurchaseInvoice::where('supplier_id', $supplierId)->get();

        foreach ($invoices as $invoice) {
            $totalPaid = $invoice->paymentAllocations()->sum('allocated_amount');

            $invoice->update([
                'paid'           => $totalPaid,
                'due'            => max(0, $invoice->total - $totalPaid),
                'payment_status' => $this->determinePaymentStatus($invoice->total, $totalPaid),
            ]);
        }
    }

    // ══════════════════════════════════════════════════════════════
    //  تسجيل في دفتر الأستاذ (مبلغ موجب)
    // ══════════════════════════════════════════════════════════════

    protected function createLedgerEntry(SupplierPayment $payment): void
    {
        SupplierLedger::create([
            'supplier_id'      => $payment->supplier_id,
            'branch_id'        => $payment->branch_id,
            'ledgerable_type'  => SupplierPayment::class,
            'ledgerable_id'    => $payment->id,
            'transaction_date' => $payment->payment_date,
            'type'             => 'payment',
            'amount'           => $payment->amount, // مــوجــب دائماً – نوع الحركة يحدد هل هو دائن أو مدين
            'reference'        => $payment->payment_number,
            'description'      => 'دفعة للمورد',
        ]);
    }

    // ══════════════════════════════════════════════════════════════
    //  توليد رقم الدفعة
    // ══════════════════════════════════════════════════════════════

    protected function generatePaymentNumber(): string
    {
        $prefix = 'PAY-';
        $year   = date('Y');

        $lastPayment = SupplierPayment::withTrashed()
            ->where('payment_number', 'like', "{$prefix}{$year}%")
            ->orderByDesc('id')
            ->first();

        if ($lastPayment) {
            $lastNumber = (int) substr($lastPayment->payment_number, -5);
            $newNumber  = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }

        return $prefix . $year . '-' . str_pad($newNumber, 5, '0', STR_PAD_LEFT);
    }

    // ══════════════════════════════════════════════════════════════
    //  ملخص المورد
    // ══════════════════════════════════════════════════════════════

    public function getSupplierSummary(int $supplierId): array
    {
        $invoiceData = PurchaseInvoice::where('supplier_id', $supplierId)
            ->selectRaw('COALESCE(SUM(total), 0) as total, COALESCE(SUM(paid), 0) as paid')
            ->first();

        return [
            'total_invoices' => (int) $invoiceData->total,
            'total_paid'     => (int) $invoiceData->paid,
            'total_due'      => max(0, $invoiceData->total - $invoiceData->paid),
        ];
    }
}
