<?php

namespace App\Services\Orders;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Models\Coupon;
use App\Models\Currency;
use App\Models\Order;
use App\Models\Promotion;
use App\Models\StockBalance;
use App\Models\StockMovement;
use App\Models\Vault;
use Illuminate\Support\Facades\DB;

class OrderService
{
    /**
     * إنشاء طلب جديد
     */
    public function createOrder(array $data, array $items): Order
    {
        return DB::transaction(function () use ($data, $items) {
            // إنشاء الطلب
            $order = Order::create([
                'branch_id' => $data['branch_id'],
                'client_id' => $data['client_id'],
                'currency_id' => $data['currency_id'] ?? Currency::getDefaultId() ?? 1,
                'exchange_rate' => $data['exchange_rate'] ?? 1,
                'shipping_address_id' => $data['shipping_address_id'] ?? null,
                'shipping_method' => $data['shipping_method'] ?? null,
                'shipping_company_id' => $data['shipping_company_id'] ?? null,
                'shipping_fee' => $data['shipping_fee'] ?? 0,
                'cod_fee' => $data['cod_fee'] ?? 0,
                'promotion_id' => $data['promotion_id'] ?? null,
                'coupon_id' => $data['coupon_id'] ?? null,
                'client_discount_amount' => $data['client_discount_amount'] ?? 0,
                'customer_notes' => $data['customer_notes'] ?? null,
                'internal_notes' => $data['internal_notes'] ?? null,
                'status' => OrderStatus::Pending,
                'payment_status' => PaymentStatus::UNPAID,
                'subtotal' => 0,
                'discount_total' => 0,
                'grand_total' => 0,
            ]);

            // إضافة العناصر
            $subtotal = 0;
            $discountTotal = 0;

            foreach ($items as $item) {
                $orderItem = $order->items()->create([
                    'product_id' => $item['product_id'],
                    'product_variant_id' => $item['product_variant_id'] ?? null,
                    'product_name' => $item['product_name'],
                    'variant_name' => $item['variant_name'] ?? null,
                    'sku' => $item['sku'] ?? null,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'unit_cost' => $item['unit_cost'] ?? 0,
                    'discount_amount' => $item['discount_amount'] ?? 0,
                    'is_free_item' => $item['is_free_item'] ?? false,
                    'promotion_id' => $item['promotion_id'] ?? null,
                ]);

                $subtotal += ($item['quantity'] * $item['unit_price']);
                $discountTotal += ($item['discount_amount'] ?? 0);
            }

            // إضافة خصم العميل
            $discountTotal += ($data['client_discount_amount'] ?? 0);

            // حساب الإجمالي
            $grandTotal = $subtotal - $discountTotal + ($data['shipping_fee'] ?? 0) + ($data['cod_fee'] ?? 0);

            // تحديث المجاميع
            $order->update([
                'subtotal' => $subtotal,
                'discount_total' => $discountTotal,
                'grand_total' => $grandTotal,
            ]);

            return $order->fresh(['items', 'client', 'branch']);
        });
    }

    /**
     * تأكيد الطلب
     */
    public function confirmOrder(Order $order): bool
    {
        if ($order->status !== OrderStatus::Pending) {
            return false;
        }

        return DB::transaction(function () use ($order) {
            $order->update([
                'status' => OrderStatus::Confirmed,
                'confirmed_at' => now(),
            ]);

            // حجز المخزون
            $this->reserveStock($order);

            return true;
        });
    }

    /**
     * بدء معالجة الطلب
     */
    public function processOrder(Order $order): bool
    {
        if ($order->status !== OrderStatus::Confirmed) {
            return false;
        }

        $order->update([
            'status' => OrderStatus::Processing,
        ]);

        return true;
    }

    /**
     * تجهيز الطلب للشحن
     */
    public function markReadyToShip(Order $order): bool
    {
        if ($order->status !== OrderStatus::Processing) {
            return false;
        }

        $order->update([
            'status' => OrderStatus::ReadyToShip,
        ]);

        return true;
    }

    /**
     * شحن الطلب
     */
    public function shipOrder(Order $order, ?string $trackingNumber = null): bool
    {
        if ($order->status !== OrderStatus::ReadyToShip) {
            return false;
        }

        return DB::transaction(function () use ($order, $trackingNumber) {
            // خصم المخزون وتسجيل الحركة
            $this->deductStock($order);

            $order->update([
                'status' => OrderStatus::Shipped,
            ]);

            // إنشاء شحنة إذا لم تكن موجودة
            if (! $order->shipment && $order->shipping_company_id) {
                $order->shipment()->create([
                    'shipping_company_id' => $order->shipping_company_id,
                    'tracking_number' => $trackingNumber,
                    'status' => 'shipped',
                    'shipped_at' => now(),
                ]);
            }

            return true;
        });
    }

    /**
     * تسليم الطلب
     */
    public function deliverOrder(Order $order): bool
    {
        if ($order->status !== OrderStatus::Shipped) {
            return false;
        }

        return DB::transaction(function () use ($order) {
            $order->update([
                'status' => OrderStatus::Delivered,
            ]);

            // تحديث الشحنة
            if ($order->shipment) {
                $order->shipment->update([
                    'status' => 'delivered',
                    'delivered_at' => now(),
                ]);
            }

            return true;
        });
    }

    /**
     * معالجة الدفع - تسجيل المبلغ في الخزنة
     */
    protected function processPaymentToVault(Order $order): void
    {
        // التحقق من أن الطلب لم يُدفع بالكامل مسبقاً
        if ($order->payment_status === PaymentStatus::PAID) {
            return;
        }

        // الحصول على المبلغ المتبقي (grand_total - paid_amount)
        $amountToCollect = $order->grand_total - ($order->paid_amount ?? 0);

        if ($amountToCollect <= 0) {
            return;
        }

        // الحصول على خزنة الفرع
        $vault = Vault::forBranch($order->branch_id);

        // تحديد نوع الدفع للملاحظة
        $paymentType = match ($order->payment_method) {
            'cod' => 'COD',
            'cash' => 'نقدي',
            'card' => 'بطاقة',
            'online' => 'أونلاين',
            default => 'طلب',
        };

        // إيداع المبلغ في الخزنة
        $vault->deposit(
            $order,
            $amountToCollect,
            "تحصيل طلب {$paymentType} رقم {$order->order_number}"
        );

        // تحديث المبلغ المدفوع وحالة الدفع
        $order->update([
            'paid_amount' => $order->grand_total,
            'payment_status' => PaymentStatus::PAID,
        ]);
    }

    /**
     * إكمال الطلب
     */
    public function completeOrder(Order $order): bool
    {
        if ($order->status !== OrderStatus::Delivered) {
            return false;
        }

        return DB::transaction(function () use ($order) {
            // تسجيل المبلغ في الخزنة
            $this->processPaymentToVault($order);

            $order->update([
                'status' => OrderStatus::Completed,
                'completed_at' => now(),
            ]);

            // تسجيل استخدام الكوبون
            if ($order->coupon_id && $order->discount_total > 0) {
                $coupon = Coupon::find($order->coupon_id);
                if ($coupon) {
                    $coupon->redeem(
                        $order->client,
                        $order,
                        $order->discount_total
                    );
                }
            }

            // زيادة عداد استخدام العرض
            if ($order->promotion_id) {
                Promotion::find($order->promotion_id)?->incrementUsage();
            }

            return true;
        });
    }

    /**
     * إلغاء الطلب
     */
    public function cancelOrder(Order $order, ?string $reason = null): bool
    {
        if (! $order->is_cancelable) {
            return false;
        }

        return DB::transaction(function () use ($order, $reason) {
            // معالجة المخزون حسب الحالة
            if (in_array($order->status, [OrderStatus::Confirmed, OrderStatus::Processing, OrderStatus::ReadyToShip])) {
                // إذا كان محجوز فقط، نحرر الحجز
                $this->releaseReservedStock($order);
            } elseif ($order->status === OrderStatus::Shipped) {
                // إذا كان تم خصمه (تم الشحن)، نعيده للمخزون
                $this->returnStock($order);
            }

            $order->update([
                'status' => OrderStatus::Canceled,
                'canceled_at' => now(),
                'internal_notes' => $reason
                    ? ($order->internal_notes ? $order->internal_notes."\n" : '')."سبب الإلغاء: {$reason}"
                    : $order->internal_notes,
            ]);

            return true;
        });
    }

    /**
     * استرجاع الطلب
     */
    public function refundOrder(Order $order, ?string $reason = null): bool
    {
        if (! in_array($order->status, [OrderStatus::Delivered, OrderStatus::Completed])) {
            return false;
        }

        return DB::transaction(function () use ($order, $reason) {
            // إعادة المخزون
            $this->returnStock($order);

            $order->update([
                'status' => OrderStatus::Refunded,
                'internal_notes' => $reason
                    ? ($order->internal_notes ? $order->internal_notes."\n" : '')."سبب الاسترجاع: {$reason}"
                    : $order->internal_notes,
            ]);

            return true;
        });
    }

    /**
     * تحديث حالة الدفع
     */
    public function updatePaymentStatus(Order $order, PaymentStatus $status): void
    {
        $order->update(['payment_status' => $status]);
    }

    /**
     * حجز المخزون
     */
    protected function reserveStock(Order $order): void
    {
        foreach ($order->items as $item) {
            if (! $item->product_variant_id) {
                continue;
            }

            $stockBalance = StockBalance::query()
                ->where('product_variant_id', $item->product_variant_id)
                ->first();

            if ($stockBalance) {
                $stockBalance->increment('reserved_qty', $item->quantity);
            }
        }
    }

    /**
     * تحرير المخزون المحجوز
     */
    protected function releaseReservedStock(Order $order): void
    {
        foreach ($order->items as $item) {
            if (! $item->product_variant_id) {
                continue;
            }

            $stockBalance = StockBalance::query()
                ->where('product_variant_id', $item->product_variant_id)
                ->first();

            if ($stockBalance) {
                $stockBalance->decrement('reserved_qty', min($stockBalance->reserved_qty, $item->quantity));
            }
        }
    }

    /**
     * خصم المخزون
     */
    protected function deductStock(Order $order): void
    {
        foreach ($order->items as $item) {
            if (! $item->product_variant_id) {
                continue;
            }

            $stockBalance = StockBalance::query()
                ->where('product_variant_id', $item->product_variant_id)
                ->first();

            if ($stockBalance) {
                // خصم من المخزون
                $stockBalance->decrement('on_hand', $item->quantity);
                // تحرير المحجوز
                $stockBalance->decrement('reserved_qty', min($stockBalance->reserved_qty, $item->quantity));

                // تسجيل حركة المخزون
                StockMovement::create([
                    'warehouse_id' => $stockBalance->warehouse_id,
                    'product_variant_id' => $item->product_variant_id,
                    'movement_type' => 'sale',
                    'quantity' => -$item->quantity,
                    'reference_type' => Order::class,
                    'reference_id' => $order->id,
                    'notes' => "بيع - طلب رقم {$order->order_number}",
                ]);
            }
        }
    }

    /**
     * إعادة المخزون (استرجاع)
     */
    protected function returnStock(Order $order): void
    {
        foreach ($order->items as $item) {
            if (! $item->product_variant_id) {
                continue;
            }

            $stockBalance = StockBalance::query()
                ->where('product_variant_id', $item->product_variant_id)
                ->first();

            if ($stockBalance) {
                $stockBalance->increment('on_hand', $item->quantity);

                // تسجيل حركة المخزون
                StockMovement::create([
                    'warehouse_id' => $stockBalance->warehouse_id,
                    'product_variant_id' => $item->product_variant_id,
                    'movement_type' => 'return',
                    'quantity' => $item->quantity,
                    'reference_type' => Order::class,
                    'reference_id' => $order->id,
                    'notes' => "استرجاع - طلب رقم {$order->order_number}",
                ]);
            }
        }
    }

    /**
     * تطبيق كوبون على الطلب
     */
    public function applyCoupon(Order $order, string $couponCode): array
    {
        $coupon = Coupon::findByCode($couponCode);

        if (! $coupon) {
            return ['success' => false, 'message' => 'كود الكوبون غير صحيح'];
        }

        if (! $coupon->is_valid) {
            return ['success' => false, 'message' => 'الكوبون غير ساري المفعول'];
        }

        if (! $coupon->canBeUsedBy($order->client)) {
            return ['success' => false, 'message' => 'لقد استنفدت حد استخدام هذا الكوبون'];
        }

        $discount = $coupon->calculateDiscount($order->subtotal);

        if ($discount === 0) {
            return ['success' => false, 'message' => 'الكوبون لا ينطبق على هذا الطلب'];
        }

        $order->update([
            'coupon_id' => $coupon->id,
            'discount_total' => $order->discount_total + $discount,
            'grand_total' => $order->grand_total - $discount,
        ]);

        return [
            'success' => true,
            'message' => 'تم تطبيق الكوبون بنجاح',
            'discount' => $discount,
        ];
    }

    /**
     * إزالة الكوبون من الطلب
     */
    public function removeCoupon(Order $order): bool
    {
        if (! $order->coupon_id) {
            return false;
        }

        $coupon = Coupon::find($order->coupon_id);
        $discount = $coupon ? $coupon->calculateDiscount($order->subtotal) : 0;

        $order->update([
            'coupon_id' => null,
            'discount_total' => max(0, $order->discount_total - $discount),
            'grand_total' => $order->grand_total + $discount,
        ]);

        return true;
    }

    /**
     * الحصول على الحالات المتاحة للانتقال
     */
    public function getAvailableTransitions(Order $order): array
    {
        return match ($order->status) {
            OrderStatus::Pending => [
                ['status' => OrderStatus::Confirmed, 'label' => 'تأكيد الطلب', 'color' => 'success'],
                ['status' => OrderStatus::Canceled, 'label' => 'إلغاء الطلب', 'color' => 'danger'],
            ],
            OrderStatus::Confirmed => [
                ['status' => OrderStatus::Processing, 'label' => 'بدء المعالجة', 'color' => 'warning'],
                ['status' => OrderStatus::Canceled, 'label' => 'إلغاء الطلب', 'color' => 'danger'],
            ],
            OrderStatus::Processing => [
                ['status' => OrderStatus::ReadyToShip, 'label' => 'جاهز للشحن', 'color' => 'primary'],
                ['status' => OrderStatus::Canceled, 'label' => 'إلغاء الطلب', 'color' => 'danger'],
            ],
            OrderStatus::ReadyToShip => [
                ['status' => OrderStatus::Shipped, 'label' => 'تم الشحن', 'color' => 'info'],
                ['status' => OrderStatus::Canceled, 'label' => 'إلغاء الطلب', 'color' => 'danger'],
            ],
            OrderStatus::Shipped => [
                ['status' => OrderStatus::Delivered, 'label' => 'تم التوصيل', 'color' => 'success'],
                ['status' => OrderStatus::Canceled, 'label' => 'إلغاء الطلب', 'color' => 'danger'],
            ],
            OrderStatus::Delivered => [
                ['status' => OrderStatus::Completed, 'label' => 'إكمال الطلب', 'color' => 'success'],
                ['status' => OrderStatus::Refunded, 'label' => 'استرجاع', 'color' => 'danger'],
            ],
            default => [],
        };
    }

    /**
     * تنفيذ انتقال الحالة
     */
    public function transition(Order $order, OrderStatus $newStatus, ?string $reason = null): bool
    {
        return match ($newStatus) {
            OrderStatus::Confirmed => $this->confirmOrder($order),
            OrderStatus::Processing => $this->processOrder($order),
            OrderStatus::ReadyToShip => $this->markReadyToShip($order),
            OrderStatus::Shipped => $this->shipOrder($order),
            OrderStatus::Delivered => $this->deliverOrder($order),
            OrderStatus::Completed => $this->completeOrder($order),
            OrderStatus::Canceled => $this->cancelOrder($order, $reason),
            OrderStatus::Refunded => $this->refundOrder($order, $reason),
            default => false,
        };
    }
}
