<?php

namespace App\Services\Orders;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Models\Invoice;
use App\Models\InvoiceCounter;
use App\Models\InvoiceItem;
use App\Models\Order;
use Illuminate\Support\Facades\DB;

class OrderInvoiceService
{
    /**
     * إنشاء فاتورة من طلب
     */
    public function createInvoiceFromOrder(Order $order): Invoice
    {
        // التحقق من أن الطلب ليس معلقاً أو ملغياً
        if (in_array($order->status, [OrderStatus::Pending, OrderStatus::Canceled])) {
            throw new \InvalidArgumentException('لا يمكن إنشاء فاتورة لطلب معلق أو ملغي');
        }

        // التحقق من عدم وجود فاتورة سابقة
        $existingInvoice = Invoice::where('order_id', $order->id)->first();
        if ($existingInvoice) {
            return $existingInvoice;
        }

        return DB::transaction(function () use ($order) {
            // إنشاء الفاتورة
            $invoice = Invoice::create([
                'branch_id' => $order->branch_id,
                'client_id' => $order->client_id,
                'order_id' => $order->id,
                'currency_id' => $order->currency_id,
                'exchange_rate' => $order->exchange_rate,
                'invoice_number' => $this->generateInvoiceNumber($order->branch_id),
                'invoice_date' => now(),
                'due_date' => now()->addDays(30),
                'subtotal' => $order->subtotal,
                'discount_total' => $order->discount_total,
                'shipping_fee' => $order->shipping_fee ?? 0,
                'grand_total' => $order->grand_total,
                'paid_amount' => $order->paid_amount,
                'payment_status' => $order->payment_status,
                'notes' => $order->customer_notes,
                'internal_notes' => $order->internal_notes,
            ]);

            // إضافة عناصر الفاتورة
            foreach ($order->items as $orderItem) {
                InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $orderItem->product_id,
                    'product_variant_id' => $orderItem->product_variant_id,
                    'product_name' => $orderItem->product_name,
                    'variant_name' => $orderItem->variant_name,
                    'sku' => $orderItem->sku,
                    'quantity' => $orderItem->quantity,
                    'unit_price' => $orderItem->unit_price,
                    'unit_cost' => $orderItem->unit_cost,
                    'discount_amount' => $orderItem->discount_amount,
                    'line_total' => $orderItem->line_total,
                ]);
            }

            return $invoice->fresh(['items', 'client', 'branch']);
        });
    }

    /**
     * توليد رقم الفاتورة
     */
    protected function generateInvoiceNumber(int $branchId): string
    {
        $counter = InvoiceCounter::firstOrCreate(
            ['branch_id' => $branchId],
            ['current_sequence' => 0]
        );

        $counter->increment('current_sequence');

        $prefix = 'INV';
        $branchPart = str_pad((string) $branchId, 2, '0', STR_PAD_LEFT);
        $datePart = now()->format('ymd');
        $seqPart = str_pad((string) $counter->current_sequence, 5, '0', STR_PAD_LEFT);

        return "{$prefix}-{$branchPart}-{$datePart}-{$seqPart}";
    }

    /**
     * تحديث حالة الدفع في الفاتورة
     */
    public function updatePaymentStatus(Invoice $invoice, int $paidAmount): void
    {
        $invoice->update([
            'paid_amount' => $paidAmount,
            'payment_status' => $this->calculatePaymentStatus($invoice->grand_total, $paidAmount),
        ]);

        // تحديث حالة الدفع في الطلب أيضاً
        if ($invoice->order_id) {
            $order = Order::find($invoice->order_id);
            if ($order) {
                $order->update([
                    'payment_status' => $invoice->payment_status,
                ]);
            }
        }
    }

    /**
     * حساب حالة الدفع
     */
    protected function calculatePaymentStatus(int $total, int $paid): PaymentStatus
    {
        if ($paid <= 0) {
            return PaymentStatus::UNPAID;
        }

        if ($paid < $total) {
            return PaymentStatus::PARTIAL;
        }

        if ($paid === $total) {
            return PaymentStatus::PAID;
        }

        return PaymentStatus::OVERPAID;
    }

    /**
     * إلغاء الفاتورة
     */
    public function cancelInvoice(Invoice $invoice): bool
    {
        // التحقق من إمكانية الإلغاء
        if ($invoice->paid_amount > 0) {
            throw new \InvalidArgumentException('لا يمكن إلغاء فاتورة مدفوعة');
        }

        $invoice->delete();

        return true;
    }

    /**
     * الحصول على بيانات الطباعة
     */
    public function getInvoicePrintData(Invoice $invoice): array
    {
        $invoice->load([
            'items.product',
            'items.variant',
            'client',
            'branch.settings',
            'currency',
        ]);

        $branchSettings = $invoice->branch?->settings;

        return [
            'invoice' => $invoice,
            'items' => $invoice->items,
            'client' => $invoice->client,
            'branch' => $invoice->branch,
            'settings' => $branchSettings,
            'design' => $branchSettings?->invoiceDesignMerged() ?? [],
            'css_vars' => $branchSettings?->invoice_css_vars ?? [],
        ];
    }

    /**
     * الحصول على بيانات طباعة الطلب
     */
    public function getOrderPrintData(Order $order): array
    {
        $order->load([
            'items.product',
            'items.variant',
            'client',
            'branch.settings',
            'currency',
            'shippingAddress',
            'shippingCompany',
        ]);

        $branchSettings = $order->branch?->settings;

        return [
            'order' => $order,
            'items' => $order->items,
            'client' => $order->client,
            'branch' => $order->branch,
            'settings' => $branchSettings,
            'design' => $branchSettings?->invoiceDesignMerged() ?? [],
            'css_vars' => $branchSettings?->invoice_css_vars ?? [],
            'shipping' => [
                'address' => $order->shippingAddress,
                'company' => $order->shippingCompany,
                'method' => $order->shipping_method,
            ],
        ];
    }

    /**
     * إنشاء فواتير متعددة للطلبات المؤكدة
     */
    public function createInvoicesForConfirmedOrders(int $branchId): array
    {
        $orders = Order::query()
            ->where('branch_id', $branchId)
            ->whereIn('status', [
                OrderStatus::Confirmed,
                OrderStatus::Processing,
                OrderStatus::ReadyToShip,
                OrderStatus::Shipped,
                OrderStatus::Delivered,
                OrderStatus::Completed,
            ])
            ->whereDoesntHave('invoice')
            ->get();

        $invoices = [];

        foreach ($orders as $order) {
            try {
                $invoices[] = $this->createInvoiceFromOrder($order);
            } catch (\Exception $e) {
                // تسجيل الخطأ والمتابعة
                logger()->error("فشل إنشاء فاتورة للطلب {$order->order_number}: ".$e->getMessage());
            }
        }

        return $invoices;
    }
}
