<?php

namespace App\Services\Invoices;

use App\Enums\ClientReceiptType;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Services\Accounting\ClientAccountingService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class UpdateInvoiceService
{
    use InvoiceSupport;

    /* =========================================================
     * أدوات مساعدة لقراءة بيانات الفورم (نفس منطق الإنشاء)
     * =======================================================*/

    /**
     * تطبيع نوع العنصر داخل صف واحد.
     *
     * - $defaultType: product / custom / accessory
     */
    protected function normalizeItemType(array $row, string $defaultType): array
    {
        // لو لا يوجد item_type نضع القيمة الافتراضية
        if (! isset($row['item_type']) || empty($row['item_type'])) {
            $row['item_type'] = $defaultType;
        }

        // توافق عكسي مع الحقل القديم type إن وجد
        if (isset($row['type']) && ! isset($row['item_type'])) {
            switch ($row['type']) {
                case 'ready':
                    $row['item_type'] = 'product';
                    break;
                case 'custom':
                    $row['item_type'] = 'custom';
                    break;
                case 'accessory':
                    $row['item_type'] = 'accessory';
                    break;
            }
        }

        return $row;
    }

    /**
     * دمج عناصر:
     * - ready_items      (منتجات جاهزة من المعرض)
     * - items            (عناصر مخصصة)
     * - accessory_items  (إكسسوارات)
     *
     * في مصفوفة واحدة items تخص جدول invoice_items فقط.
     */
    protected function mergeFormItems(array $data): array
    {
        $readyItems = isset($data['ready_items']) && is_array($data['ready_items'])
            ? $data['ready_items']
            : [];

        $customItems = isset($data['items']) && is_array($data['items'])
            ? $data['items']
            : [];

        $accessoryItems = isset($data['accessory_items']) && is_array($data['accessory_items'])
            ? $data['accessory_items']
            : [];

        // المنتجات الجاهزة => product
        $readyItems = array_map(function ($row) {
            if (! is_array($row)) {
                return [];
            }

            return $this->normalizeItemType($row, 'product');
        }, $readyItems);

        // المخصصة => custom
        $customItems = array_map(function ($row) {
            if (! is_array($row)) {
                return [];
            }

            return $this->normalizeItemType($row, 'custom');
        }, $customItems);

        // الإكسسوارات => accessory
        $accessoryItems = array_map(function ($row) {
            if (! is_array($row)) {
                return [];
            }

            return $this->normalizeItemType($row, 'accessory');
        }, $accessoryItems);

        $data['items'] = array_values(array_merge($readyItems, $customItems, $accessoryItems));

        return $data;
    }

    /**
     * تهيئة الأرقام المالية من بيانات الفورم
     * نفس منطق CreateInvoiceService::normalizeFinancialFromForm
     */
    protected function normalizeFinancialFromForm(array $data): array
    {
        /* ================= الفاتورة الأساسية ================= */

        $invoiceSubtotal = $this->toInt($data['invoice_subtotal'] ?? 0);
        $invoiceDiscount = $this->toInt($data['invoice_discount'] ?? 0);
        $invoiceShipping = $this->toInt($data['invoice_shipping'] ?? 0);
        $invoicePaid = $this->toInt($data['invoice_paid'] ?? 0);

        $invoiceNet = max($invoiceSubtotal - $invoiceDiscount + $invoiceShipping, 0);
        $invoicePaid = min($invoicePaid, $invoiceNet);
        $invoiceDue = max($invoiceNet - $invoicePaid, 0);

        /* ================= الإكسسوارات ================= */

        $accessorySubtotal = $this->toInt($data['accessory_subtotal'] ?? 0);
        $accessoryDiscount = $this->toInt($data['accessory_discount'] ?? 0);
        $accessoryPaid = $this->toInt($data['accessory_paid'] ?? 0);

        $accessoryNet = max($accessorySubtotal - $accessoryDiscount, 0);
        $accessoryPaid = min($accessoryPaid, $accessoryNet);
        $accessoryDue = max($accessoryNet - $accessoryPaid, 0);

        /* ================= المجاميع النهائية ================= */

        $combinedSubtotal = $this->toInt(
            $data['subtotal'] ?? ($invoiceSubtotal + $accessorySubtotal)
        );

        $combinedDiscount = $this->toInt(
            $data['total_discount'] ?? ($invoiceDiscount + $accessoryDiscount)
        );

        $combinedShipping = $this->toInt(
            $data['total_shipping'] ?? $invoiceShipping
        );

        $combinedGrandFromForm = $this->toInt($data['grand_total'] ?? 0);
        $combinedGrandCalc = max($combinedSubtotal - $combinedDiscount + $combinedShipping, 0);
        $combinedGrand = $combinedGrandFromForm > 0
            ? $combinedGrandFromForm
            : $combinedGrandCalc;

        $combinedPaidFromForm = $this->toInt($data['total_paid'] ?? 0);
        $combinedPaidCalc = $invoicePaid + $accessoryPaid;
        $combinedPaid = $combinedPaidFromForm > 0
            ? $combinedPaidFromForm
            : $combinedPaidCalc;

        $combinedPaid = min($combinedPaid, $combinedGrand);

        $combinedDueFromForm = $this->toInt($data['total_due'] ?? 0);
        $combinedDueCalc = max($combinedGrand - $combinedPaid, 0);
        $combinedDue = $combinedDueFromForm > 0
            ? $combinedDueFromForm
            : $combinedDueCalc;

        $combinedDue = max($combinedDue, 0);

        // تفاصيل الفاتورة الأساسية
        $data['__invoice_subtotal'] = $invoiceSubtotal;
        $data['__invoice_discount'] = $invoiceDiscount;
        $data['__invoice_shipping'] = $invoiceShipping;
        $data['__invoice_paid'] = $invoicePaid;
        $data['__invoice_net'] = $invoiceNet;
        $data['__invoice_due'] = $invoiceDue;

        // تفاصيل الإكسسوارات
        $data['__accessory_subtotal'] = $accessorySubtotal;
        $data['__accessory_discount'] = $accessoryDiscount;
        $data['__accessory_net'] = $accessoryNet;
        $data['__accessory_paid'] = $accessoryPaid;
        $data['__accessory_due'] = $accessoryDue;

        // المجاميع النهائية
        $data['__subtotal_combined'] = $combinedSubtotal;
        $data['__discount_combined'] = $combinedDiscount;
        $data['__shipping_combined'] = $combinedShipping;
        $data['__grand_total_combined'] = $combinedGrand;
        $data['__paid_combined'] = $combinedPaid;
        $data['__due_combined'] = $combinedDue;

        // نوع التسوية: نقداً / أقساط
        $paymentStatusUi = $data['payment_status'] ?? 'paid';
        $data['settlement_type'] = $paymentStatusUi === 'installment'
            ? 'installment'
            : 'cash';

        return $data;
    }

    /* =========================================================
     * تحديث الفاتورة من الفورم
     * =======================================================*/

    public function updateFromForm(Invoice $invoice, array $data): Invoice
    {
        Log::info('UpdateInvoiceService::updateFromForm', ['invoice_id' => $invoice->id]);

        // دمج عناصر الفاتورة (جاهز + مخصص + إكسسوارات)
        $data = $this->mergeFormItems($data);

        // تهيئة الأرقام المالية
        $data = $this->normalizeFinancialFromForm($data);

        // تحقق أساسي
        $this->validateBasicInvoiceData($data);

        // تحقق عناصر
        $this->validateItems($data);

        $oldClient = (int) ($invoice->client_id ?? 0);

        return DB::transaction(function () use ($invoice, $data, $oldClient) {

            // ===== قيم الفاتورة الأساسية =====
            $invoiceSubtotal = $this->toInt($data['__invoice_subtotal'] ?? 0);
            $invoiceDiscount = $this->toInt($data['__invoice_discount'] ?? 0);
            $invoiceShipping = $this->toInt($data['__invoice_shipping'] ?? 0);
            $invoiceNet = $this->toInt($data['__invoice_net'] ?? 0);
            $invoicePaidBase = $this->toInt($data['__invoice_paid'] ?? 0);
            $invoiceDueBase = $this->toInt($data['__invoice_due'] ?? 0);

            // ===== قيم الإكسسوارات =====
            $accessorySubtotal = $this->toInt($data['__accessory_subtotal'] ?? 0);
            $accessoryDiscount = $this->toInt($data['__accessory_discount'] ?? 0);
            $accessoryNet = $this->toInt($data['__accessory_net'] ?? 0);
            $accessoryPaidBase = $this->toInt($data['__accessory_paid'] ?? 0);
            $accessoryDueBase = $this->toInt($data['__accessory_due'] ?? 0);

            // ===== المجاميع النهائية =====
            $grandFull = $this->toInt(
                $data['__grand_total_combined']
                    ?? ($invoiceNet + $accessoryNet)
            );

            $newPaidFull = $this->toInt(
                $data['__paid_combined']
                    ?? ($invoicePaidBase + $accessoryPaidBase)
            );

            $newPaidFull = min($newPaidFull, $grandFull);
            $dueFull = max($grandFull - $newPaidFull, 0);

            // تحديث رأس الفاتورة
            $invoice->fill([
                'branch_id' => $data['branch_id'] ?? $invoice->branch_id,
                'client_id' => $data['client_id'] ?? $invoice->client_id,
                'design_contract_id' => $data['design_contract_id'] ?? $invoice->design_contract_id,
                'invoice_number' => $data['invoice_number'] ?? $invoice->invoice_number,
                'invoice_type' => $data['invoice_type'] ?? $invoice->invoice_type,
                'sale_type' => $data['sale_type'] ?? $invoice->sale_type,
                'status' => $data['status'] ?? $invoice->status,
                'invoice_date' => $data['invoice_date'] ?? $invoice->invoice_date?->toDateString(),
                'notes' => $data['notes'] ?? $invoice->notes,

                // الفاتورة الأساسية
                'invoice_subtotal' => $invoiceSubtotal,
                'invoice_discount' => $invoiceDiscount,
                'invoice_shipping' => $invoiceShipping,
                'invoice_net_total' => $invoiceNet,
                'invoice_paid' => $invoicePaidBase,
                'invoice_due' => $invoiceDueBase,

                // الإكسسوارات
                'accessory_subtotal' => $accessorySubtotal,
                'accessory_discount' => $accessoryDiscount,
                'accessory_net_total' => $accessoryNet,
                'accessory_paid' => $accessoryPaidBase,
                'accessory_due' => $accessoryDueBase,
            ]);

            // المجاميع النهائية في رأس الفاتورة
            $invoice->subtotal = $this->toInt($data['__subtotal_combined'] ?? ($invoiceSubtotal + $accessorySubtotal));
            $invoice->discount_total = $this->toInt($data['__discount_combined'] ?? ($invoiceDiscount + $accessoryDiscount));
            $invoice->shipping_total = $this->toInt($data['__shipping_combined'] ?? $invoiceShipping);
            $invoice->grand_total = $this->toInt($data['__grand_total_combined'] ?? ($invoiceNet + $accessoryNet));
            $invoice->paid_amount = $newPaidFull;
            $invoice->due_amount = $dueFull;

            // حالة الدفع
            if ($newPaidFull <= 0) {
                $invoice->payment_status = 'unpaid';
            } elseif ($dueFull <= 0) {
                $invoice->payment_status = 'paid';
            } else {
                $invoice->payment_status = 'partial';
            }

            $invoice->save();

            // إعادة بناء عناصر الفاتورة (كل الأنواع)
            $invoice->items()->delete();
            $this->createInvoiceItems($invoice, $data['items'] ?? []);
            $invoice->refresh();

            /** @var ClientAccountingService $acc */
            $acc = app(ClientAccountingService::class);

            // المدفوع الحالي من خلال المحاسبة (كل الوصولات على الفاتورة)
            $currentPaid = (int) $acc->invoicePaidSum($invoice->fresh());

            $delta = $newPaidFull - $currentPaid;

            // منع تقليل المدفوع عن الواقع
            if ($delta < 0) {
                Validator::make([], [])->after(function ($validator) {
                    $validator->errors()->add(
                        'paid_amount',
                        'لا يمكن أن يكون مجموع المدفوع أقل من المبالغ التي تم تحصيلها فعلياً على الفاتورة.'
                    );
                })->validate();
            }

            // تحصيل إضافي إذا هناك زيادة
            if ($delta > 0 && $invoice->client_id) {
                $acc->collectPaymentForInvoice(
                    invoice: $invoice,
                    amount: $delta,
                    receiptType: ClientReceiptType::COLLECTION,
                    paymentType: $data['payment_type'] ?? 'cash',
                    paymentDetails: $data['payment_details'] ?? 'دفعة إضافية بعد تعديل الفاتورة'
                );
            }

            // إعادة احتساب المدفوع والمتبقي من خلال المحاسبة
            $invoice = $acc->recomputeInvoicePaidAndRemaining($invoice->fresh());

            // تحديث حالة الدفع بعد إعادة الحساب
            $grand = (int) $invoice->grand_total;
            $paid = (int) $invoice->paid_amount;
            $due = max($grand - $paid, 0);

            if ($paid <= 0) {
                $invoice->payment_status = 'unpaid';
            } elseif ($due <= 0 && $grand > 0) {
                $invoice->payment_status = 'paid';
            } else {
                $invoice->payment_status = 'partial';
            }
            $invoice->due_amount = $due;
            $invoice->save();

            // المبلغ المتبقي الكلي (من المجاميع النهائية)
            $remainingCombined = $this->toInt(
                $data['__due_combined'] ?? ($grandFull - $newPaidFull)
            );

            // منطق الأقساط (نفس النمط السابق تقريباً)
            $settlementType = $data['settlement_type'] ?? $data['payment_status'] ?? 'cash';
            $installmentsCnt = (int) ($data['installments_count'] ?? 0);
            $firstDueDate = $data['first_due_date'] ?? null;
            $scheduleRowsRaw = $data['installments'] ?? null;

            $scheduleRows = [];
            if (is_array($scheduleRowsRaw)) {
                $seq = 1;
                foreach ($scheduleRowsRaw as $row) {
                    if (! is_array($row)) {
                        continue;
                    }
                    $amt = $this->toInt($row['amount'] ?? 0);
                    if ($amt <= 0) {
                        continue;
                    }

                    $scheduleRows[] = [
                        'sequence' => $seq++,
                        'amount' => $amt,
                        'due_date' => $row['due_date']
                            ?? $firstDueDate
                            ?? now('Asia/Baghdad')->addMonth()->toDateString(),
                        'status' => $row['status'] ?? 'pending',
                    ];
                }
            }

            $hasActivePlan = DB::table('client_installment_plans')
                ->where('invoice_type', Invoice::class)
                ->where('invoice_id', $invoice->id)
                ->where('status', 'active')
                ->exists();

            if (
                $invoice->client_id &&
                $settlementType === 'installment' &&
                $remainingCombined > 0 &&
                (
                    $installmentsCnt > 0 ||
                    (is_array($scheduleRows) && count($scheduleRows) > 0)
                )
            ) {
                if ($hasActivePlan) {
                    $acc->rebuildInstallmentPlanForInvoice(
                        invoice: $invoice,
                        remainingAmount: $remainingCombined,
                        installmentsCount: ($installmentsCnt > 0
                            ? $installmentsCnt
                            : (is_array($scheduleRows) ? count($scheduleRows) : null)),
                        startDate: $firstDueDate,
                        notes: $data['notes'] ?? null,
                        scheduleRows: $scheduleRows,
                        preservePaid: true,
                    );
                } else {
                    $acc->createInstallmentPlanForInvoice(
                        invoice: $invoice,
                        remainingAmount: $remainingCombined,
                        installmentsCount: ($installmentsCnt > 0
                            ? $installmentsCnt
                            : (is_array($scheduleRows) ? count($scheduleRows) : null)),
                        startDate: $firstDueDate,
                        downPayment: $newPaidFull,
                        notes: $data['notes'] ?? null,
                        scheduleRows: $scheduleRows,
                    );
                }
            } else {
                if ($hasActivePlan) {
                    DB::table('client_installment_plans')
                        ->where('invoice_type', Invoice::class)
                        ->where('invoice_id', $invoice->id)
                        ->where('status', 'active')
                        ->update([
                            'status' => 'canceled',
                            'updated_at' => now(),
                        ]);
                }
            }

            // إعادة بناء ملخص حساب العميل
            $newClient = (int) ($invoice->client_id ?? 0);
            if ($newClient) {
                $acc->recomputeClientAccount($newClient, (int) $invoice->branch_id);
            }
            if ($oldClient && $oldClient !== $newClient) {
                $acc->recomputeClientAccount($oldClient, (int) $invoice->branch_id);
            }

            Log::info('UpdateInvoiceService::updateFromForm done', ['invoice_id' => $invoice->id]);

            return $invoice->fresh(['items']);
        });
    }

    /**
     * إنشاء عناصر الفاتورة من مصفوفة items (كل الأنواع: منتج / مخصص / إكسسوار)
     * نفس منطق createInvoiceItems في خدمة الإنشاء المحدثة.
     */
    protected function createInvoiceItems(Invoice $invoice, array $items): void
    {
        $rows = [];

        foreach ($items as $row) {
            if (! is_array($row)) {
                continue;
            }

            $quantity = (int) ($row['quantity'] ?? 0);
            if ($quantity <= 0) {
                continue;
            }

            $unitPrice = $this->toInt($row['unit_price'] ?? 0);

            // القياس:
            // - للمخصص: قيمة حقيقية
            // - للمنتج/الإكسسوار: غالباً 1
            $measurement = isset($row['measurement'])
                ? (float) $row['measurement']
                : 1.00;

            $lineTotal = $this->toInt($row['line_total'] ?? 0);

            // نوع العنصر من الفورم
            $itemType = $row['item_type'] ?? null;

            // توافق عكسي مع الحقل القديم type
            if (! $itemType && isset($row['type'])) {
                switch ($row['type']) {
                    case 'ready':
                        $itemType = 'product';
                        break;
                    case 'custom':
                        $itemType = 'custom';
                        break;
                    case 'accessory':
                        $itemType = 'accessory';
                        break;
                }
            }

            // لو ما زال فارغ نحاول الاستنتاج
            if (! $itemType) {
                if (! empty($row['product_variant_id']) || ! empty($row['product_id'])) {
                    $itemType = 'product';
                } else {
                    $itemType = 'custom';
                }
            }

            $isCustom = $itemType === 'custom';
            $isAccessory = $itemType === 'accessory';
            $isProductLike = ($itemType === 'product') || $isAccessory;

            // لو line_total غير موجود نحسبه حسب النوع
            if ($lineTotal <= 0) {
                if ($isCustom && $measurement > 0) {
                    $lineTotal = (int) round($unitPrice * $quantity * $measurement);
                } else {
                    // منتج جاهز / إكسسوار
                    $lineTotal = $unitPrice * $quantity;
                }
            }

            $rows[] = [
                'invoice_id' => $invoice->id,

                // نوع السطر
                'item_type' => $itemType,

                // المنتجات الجاهزة / الإكسسوارات
                'product_id' => $isProductLike ? ($row['product_id'] ?? null) : null,
                'product_variant_id' => $isProductLike ? ($row['product_variant_id'] ?? null) : null,

                'measurement' => $measurement,
                'unit_price' => $unitPrice,
                'quantity' => $quantity,
                'line_total' => $lineTotal,
                'notes' => $row['notes'] ?? null,
            ];
        }

        if (! empty($rows)) {
            InvoiceItem::insert($rows);
        }
    }
}
