<?php

namespace App\Services\Invoices;

trait InvoiceSupport
{
    /**
     * تحويل أي قيمة إلى عدد صحيح (INT) مع إزالة أي رموز غير رقمية.
     */
    protected function toInt($value): int
    {
        if (is_numeric($value)) {
            return (int) $value;
        }

        $cleaned = preg_replace('/[^\d]/', '', (string) $value);

        return $cleaned !== '' ? (int) $cleaned : 0;
    }

    /**
     * التحقق من صحة عناصر الفاتورة
     *
     * يدعم الحقول التالية في بيانات الفورم:
     * - items           (المصفوفة المدمجة بعد mergeFormItems)
     * - ready_items     (للخلفية، لو تم النداء قبل الدمج)
     * - accessory_items (للخلفية، لو تم النداء قبل الدمج)
     *
     * في النهاية نتحقق أن كل عنصر لديه:
     * - quantity > 0
     * - unit_price > 0
     */
    protected function validateItems(array $data): void
    {
        $items = [];

        // إذا تم الدمج مسبقاً (الحالة المطلوبة)
        if (isset($data['items']) && is_array($data['items'])) {
            $items = $data['items'];
        } else {
            // دعم نداء قديم قبل الدمج (احتياطي فقط)
            foreach (['ready_items', 'items', 'accessory_items'] as $key) {
                if (isset($data[$key]) && is_array($data[$key])) {
                    $items = array_merge($items, $data[$key]);
                }
            }
        }

        if (empty($items)) {
            // CreateInvoiceService / UpdateInvoiceService سيتكفلان برمي الخطأ لو لم توجد عناصر
            return;
        }

        foreach ($items as $index => $item) {
            if (! is_array($item)) {
                continue;
            }

            $itemNumber = $index + 1;

            // الكمية
            if (! isset($item['quantity']) || (int) $item['quantity'] <= 0) {
                throw new \InvalidArgumentException(
                    "العنصر رقم {$itemNumber}: الكمية يجب أن تكون أكبر من صفر"
                );
            }

            // سعر الوحدة
            if (! isset($item['unit_price']) || $this->toInt($item['unit_price']) <= 0) {
                throw new \InvalidArgumentException(
                    "العنصر رقم {$itemNumber}: سعر الوحدة يجب أن يكون أكبر من صفر"
                );
            }
        }
    }

    /**
     * التحقق من البيانات الأساسية للفاتورة
     */
    protected function validateBasicInvoiceData(array $data): void
    {
        $errors = [];

        if (empty($data['branch_id'])) {
            $errors[] = 'معرف الفرع مطلوب';
        }

        if (empty($data['client_id'])) {
            $errors[] = 'معرف العميل مطلوب';
        }

        if (! empty($errors)) {
            throw new \InvalidArgumentException(implode(' • ', $errors));
        }
    }
}
