<?php

namespace App\Services\Invoices;

use App\Models\Invoice;
use App\Models\StockMovement;
use App\Services\Accounting\ClientAccountingService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DeleteInvoiceService
{
    /**
     * حذف فاتورة مع تنظيف:
     * - المحاسبة (دفتر العميل، الوصولات، الأقساط، ملخص الحساب)
     * - حركات المخزون المرتبطة بالفاتورة
     * - عناصر الفاتورة (invoice_items)
     * - رأس الفاتورة
     */
    public function execute(Invoice $invoice): void
    {
        DB::transaction(function () use ($invoice) {

            Log::info('DeleteInvoiceService::execute started', [
                'invoice_id'     => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
            ]);

            /** @var ClientAccountingService $acc */
            $acc = app(ClientAccountingService::class);

            // 1) تنظيف المحاسبة (الأقساط، التخصيصات، دفتر العميل، ملخص الحساب)
            $acc->deleteInvoiceAccountingCleanup($invoice);

            // 2) حركات المخزون المرتبطة بهذه الفاتورة (إن وجدت)
            if (method_exists($invoice, 'deleteStockMovements')) {
                $invoice->deleteStockMovements();

                Log::info('DeleteInvoiceService::stock movements deleted via model method', [
                    'invoice_id' => $invoice->id,
                ]);
            } else {
                StockMovement::query()
                    ->where('source_type', Invoice::class)
                    ->where('source_id', $invoice->id)
                    ->delete();

                Log::info('DeleteInvoiceService::stock movements deleted via query', [
                    'invoice_id' => $invoice->id,
                ]);
            }

            // 3) حذف عناصر الفاتورة (كل الأنواع: منتج / مخصص / إكسسوار)
            $itemsCount = $invoice->items()->count();
            $invoice->items()->delete();

            Log::info('DeleteInvoiceService::invoice items deleted', [
                'invoice_id'  => $invoice->id,
                'items_count' => $itemsCount,
            ]);

            // 4) حذف رأس الفاتورة
            $invoice->delete();

            Log::info('DeleteInvoiceService::invoice deleted', [
                'invoice_id' => $invoice->id,
            ]);
        });
    }
}
