<?php

namespace App\Services\Invoices;

use App\Enums\ClientReceiptType;
use App\Enums\MovementDirection;
use App\Models\BranchWarehouse;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\StockBalance;
use App\Models\StockMovement;
use App\Services\Accounting\ClientAccountingService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CreateInvoiceService
{
    use InvoiceSupport;

    /* =========================================================
     * أدوات مساعدة لقراءة بيانات الفورم
     * =======================================================*/

    /**
     * دمج عناصر:
     * - ready_items      (منتجات جاهزة من المعرض)
     * - items            (عناصر مخصصة حسب التصميم)
     * - accessory_items  (إكسسوارات)
     *
     * في مصفوفة واحدة items تخص جدول invoice_items فقط.
     *
     * كل سطر يحمل item_type:
     * - product   : منتج جاهز
     * - custom    : مخصص
     * - accessory : إكسسوار
     */
    protected function mergeFormItems(array $data): array
    {
        $readyItems = isset($data['ready_items']) && is_array($data['ready_items'])
            ? $data['ready_items']
            : [];

        $customItems = isset($data['items']) && is_array($data['items'])
            ? $data['items']
            : [];

        $accessoryItems = isset($data['accessory_items']) && is_array($data['accessory_items'])
            ? $data['accessory_items']
            : [];

        // توحيد نوع العنصر للمنتجات الجاهزة
        $readyItems = array_map(function ($row) {
            if (! is_array($row)) {
                return [];
            }

            // المنتجات الجاهزة = product
            $row['item_type'] = $row['item_type'] ?? 'product';

            return $row;
        }, $readyItems);

        // توحيد نوع العنصر للعناصر المخصصة
        $customItems = array_map(function ($row) {
            if (! is_array($row)) {
                return [];
            }

            // العناصر المخصصة = custom
            $row['item_type'] = $row['item_type'] ?? 'custom';

            return $row;
        }, $customItems);

        // توحيد نوع العنصر للإكسسوارات
        $accessoryItems = array_map(function ($row) {
            if (! is_array($row)) {
                return [];
            }

            // الإكسسوارات = accessory
            $row['item_type'] = $row['item_type'] ?? 'accessory';

            return $row;
        }, $accessoryItems);

        // دمج الجميع في items (هي التي تُستخدم لإنشاء invoice_items)
        $data['items'] = array_values(array_merge(
            $readyItems,
            $customItems,
            $accessoryItems
        ));

        return $data;
    }

    /**
     * تهيئة القيم المالية من الفورم (دون فصل الإكسسوارات عن الفاتورة).
     *
     * نتعامل مع الحقول:
     * - invoice_subtotal
     * - invoice_discount
     * - invoice_shipping
     * - invoice_paid
     *
     * بالإضافة إلى كرت المجاميع النهائية إن وُجد:
     * - subtotal, total_discount, total_shipping, grand_total, total_paid, total_due
     *
     * ملاحظة:
     * - المجاميع النهائية في الفورم (subtotal / grand_total / ...) هي مجموع
     *   الفاتورة الأساسية + الإكسسوارات (تم حسابها في الـ Form).
     */
    protected function normalizeFinancialFromForm(array $data): array
    {
        // ===== الفاتورة الأساسية =====
        $invoiceSubtotal = $this->toInt($data['invoice_subtotal'] ?? 0);
        $invoiceDiscount = $this->toInt($data['invoice_discount'] ?? 0);
        $invoiceShipping = $this->toInt($data['invoice_shipping'] ?? 0);
        $invoicePaid = $this->toInt($data['invoice_paid'] ?? 0);

        $invoiceNet = max($invoiceSubtotal - $invoiceDiscount + $invoiceShipping, 0);
        $invoicePaid = min($invoicePaid, $invoiceNet);
        $invoiceDue = max($invoiceNet - $invoicePaid, 0);

        // ===== المجاميع النهائية (كما أرسلها الفورم) =====
        // في الـ Form: subtotal / total_discount / total_shipping / grand_total / total_paid / total_due
        // تشمل الفاتورة الأساسية + الإكسسوارات
        $combinedSubtotal = $this->toInt(
            $data['subtotal'] ?? $data['invoice_subtotal'] ?? $invoiceSubtotal
        );

        $combinedDiscount = $this->toInt(
            $data['total_discount'] ?? $data['invoice_discount'] ?? $invoiceDiscount
        );

        $combinedShipping = $this->toInt(
            $data['total_shipping'] ?? $data['invoice_shipping'] ?? $invoiceShipping
        );

        $combinedGrandFromForm = $this->toInt($data['grand_total'] ?? 0);
        $combinedGrandCalc = max($combinedSubtotal - $combinedDiscount + $combinedShipping, 0);
        $combinedGrand = $combinedGrandFromForm > 0
            ? $combinedGrandFromForm
            : $combinedGrandCalc;

        $combinedPaidFromForm = $this->toInt($data['total_paid'] ?? 0);
        $combinedPaidCalc = $invoicePaid; // (سابقًا لا نفرّق الإكسسوارات)
        $combinedPaid = $combinedPaidFromForm > 0
            ? $combinedPaidFromForm
            : $combinedPaidCalc;

        $combinedPaid = min($combinedPaid, $combinedGrand);

        $combinedDueFromForm = $this->toInt($data['total_due'] ?? 0);
        $combinedDueCalc = max($combinedGrand - $combinedPaid, 0);
        $combinedDue = $combinedDueFromForm > 0
            ? $combinedDueFromForm
            : $combinedDueCalc;

        $combinedDue = max($combinedDue, 0);

        // ------ تخزين تفاصيل الفاتورة الأساسية (لحفظها في جدول invoices) ------
        $data['__invoice_subtotal'] = $invoiceSubtotal;
        $data['__invoice_discount'] = $invoiceDiscount;
        $data['__invoice_shipping'] = $invoiceShipping;
        $data['__invoice_paid'] = $invoicePaid;
        $data['__invoice_net'] = $invoiceNet;
        $data['__invoice_due'] = $invoiceDue;

        // ------ تخزين المجاميع النهائية (للمحاسبة فقط) ------
        $data['__subtotal_combined'] = $combinedSubtotal;
        $data['__discount_combined'] = $combinedDiscount;
        $data['__shipping_combined'] = $combinedShipping;
        $data['__grand_total_combined'] = $combinedGrand;
        $data['__paid_combined'] = $combinedPaid;
        $data['__due_combined'] = $combinedDue;

        // نوع التسوية: نقداً / أقساط (من payment_status في الفورم)
        $paymentStatusUi = $data['payment_status'] ?? 'paid';
        $data['settlement_type'] = $paymentStatusUi === 'installment'
            ? 'installment'
            : 'cash';

        return $data;
    }

    /* =========================================================
     * إنشاء الفاتورة من الفورم
     * =======================================================*/

    public function createFromForm(array $data): Invoice
    {
        // دمج عناصر الفاتورة الأساسية + الإكسسوارات في items
        $data = $this->mergeFormItems($data);

        // تهيئة القيم المالية (بما فيها المجاميع النهائية)
        $data = $this->normalizeFinancialFromForm($data);

        Log::info('CreateInvoiceService::createFromForm', [
            'branch_id' => $data['branch_id'] ?? null,
            'client_id' => $data['client_id'] ?? null,
            'sale_type' => $data['sale_type'] ?? null,
            'items_count' => isset($data['items']) && is_array($data['items'])
                ? count($data['items'])
                : 0,
        ]);

        // التحقق من البيانات الأساسية
        $this->validateBasicInvoiceData($data);

        // التحقق من العناصر
        if (isset($data['items'])) {
            $this->validateItems($data);
        }

        $branchId = (int) ($data['branch_id'] ?? (function_exists('user_info') ? (user_info('branch_id') ?? 0) : 0));
        $clientId = (int) ($data['client_id'] ?? 0);
        $invoiceType = $data['invoice_type'] ?? 'ready';   // جاهزة / مخصصة
        $saleType = $data['sale_type'] ?? 'retail';     // مفرد / جملة / وكيل
        $status = $data['status'] ?? 'preparing';

        return DB::transaction(function () use ($data, $branchId, $clientId, $invoiceType, $saleType, $status) {

            // ===== قيم الفاتورة الأساسية =====
            $invoiceSubtotal = $this->toInt($data['__invoice_subtotal'] ?? 0);
            $invoiceDiscount = $this->toInt($data['__invoice_discount'] ?? 0);
            $invoiceShipping = $this->toInt($data['__invoice_shipping'] ?? 0);
            $invoiceNet = $this->toInt($data['__invoice_net'] ?? 0);
            $invoicePaid = $this->toInt($data['__invoice_paid'] ?? 0);
            $invoiceDue = $this->toInt($data['__invoice_due'] ?? 0);

            // ===== المجاميع النهائية (قد تشمل إكسسوارات في الفورم) =====
            $subtotalFull = $this->toInt($data['__subtotal_combined'] ?? 0);
            $discountFull = $this->toInt($data['__discount_combined'] ?? 0);
            $shippingFull = $this->toInt($data['__shipping_combined'] ?? 0);
            $grandFull = $this->toInt($data['__grand_total_combined'] ?? 0);
            $paidFull = $this->toInt($data['__paid_combined'] ?? 0);
            $dueFull = $this->toInt($data['__due_combined'] ?? 0);

            $paidFull = min($paidFull, $grandFull);
            $dueFull = max($grandFull - $paidFull, 0);

            /** @var Invoice $invoice */
            $invoice = Invoice::create([
                'branch_id' => $branchId,
                'client_id' => $clientId ?: null,
                'design_contract_id' => $data['design_contract_id'] ?? null,
                'invoice_number' => $data['invoice_number'] ?? null,
                'invoice_type' => $invoiceType,
                'sale_type' => $saleType,
                'status' => $status,
                'payment_status' => 'unpaid',
                'invoice_date' => $data['invoice_date'] ?? now('Asia/Baghdad')->toDateString(),

                // ====== المبالغ المالية للفاتورة الأساسية ======
                'invoice_subtotal' => $invoiceSubtotal,
                'invoice_discount' => $invoiceDiscount,
                'invoice_shipping' => $invoiceShipping,
                'invoice_net_total' => $invoiceNet,
                'invoice_paid' => $invoicePaid,
                'invoice_due' => $invoiceDue,

                // أعمدة الإكسسوارات في جدول الفواتير (يمكن لاحقاً تفعيلها إذا اخترت فصلها)
                'accessory_subtotal' => 0,
                'accessory_discount' => 0,
                'accessory_net_total' => 0,
                'accessory_paid' => 0,
                'accessory_due' => 0,

                // ====== أعمدة الفاتورة العامة = المجاميع النهائية ======
                'subtotal' => $subtotalFull,
                'discount_total' => $discountFull,
                'shipping_total' => $shippingFull,
                'grand_total' => $grandFull,
                'paid_amount' => $paidFull,
                'due_amount' => $dueFull,

                'notes' => $data['notes'] ?? null,
                'user_id' => (int) (function_exists('user_info') ? (user_info('id') ?? 0) : 0),

                'approved_by' => null,
                'approved_at' => null,
            ]);

            Log::info('CreateInvoiceService - Invoice created', [
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
            ]);

            // ===== عناصر الفاتورة (منتج / مخصص / إكسسوار) =====
            if (isset($data['items']) && is_array($data['items']) && count($data['items']) > 0) {
                $this->createInvoiceItems($invoice, $data['items']);

                Log::info('CreateInvoiceService - Invoice items created', [
                    'invoice_id' => $invoice->id,
                    'items_count' => count($data['items']),
                ]);
            }

            // المعالجة الموحّدة (مديونية، أقساط، مخزون ... إلخ)
            return $this->processAfterItemsPersisted(
                $invoice->fresh('items'),
                $data,
                $invoiceShipping,
                $invoiceDiscount,
                $paidFull
            );
        });
    }

    /**
     * الاستعمال الخاص بـ Filament عند الإنشاء عبر العلاقات مباشرة.
     */
    public function processAfterFilamentCreation(Invoice $invoice, array $data): Invoice
    {
        $data = $this->normalizeFinancialFromForm($data);

        $shippingBase = $this->toInt(
            $data['__shipping_combined']
                ?? $data['__invoice_shipping']
                ?? $invoice->invoice_shipping
                ?? 0
        );
        $discountBase = $this->toInt(
            $data['__discount_combined']
                ?? $data['__invoice_discount']
                ?? $invoice->invoice_discount
                ?? 0
        );
        $paidFull = $this->toInt($data['__paid_combined'] ?? 0);

        return DB::transaction(function () use ($invoice, $data, $shippingBase, $discountBase, $paidFull) {
            return $this->processAfterItemsPersisted($invoice->fresh('items'), $data, $shippingBase, $discountBase, $paidFull);
        });
    }

    /* =========================================================
     * منطق مشترك بعد حفظ العناصر
     * =======================================================*/

    protected function processAfterItemsPersisted(
        Invoice $invoice,
        array $data,
        int $shippingBase,
        int $discountBase,
        int $paidFull
    ): Invoice {

        $invoice->loadMissing('items');

        $itemsCount = $invoice->items()->count();
        if ($itemsCount === 0) {
            Log::warning('CreateInvoiceService - No items found after persistence', [
                'invoice_id' => $invoice->id,
            ]);

            throw new \RuntimeException(
                'لم يتم إضافة أي عناصر للفاتورة. يرجى التأكد من إضافة عنصر واحد على الأقل.'
            );
        }

        Log::info('CreateInvoiceService - Items count after refresh', [
            'invoice_id' => $invoice->id,
            'items_count' => $itemsCount,
        ]);

        // -------- تحديث بيانات الفاتورة الأساسية من الفورم --------
        $invoiceSubtotal = $this->toInt($data['__invoice_subtotal'] ?? $invoice->invoice_subtotal ?? 0);
        $invoiceDiscount = $this->toInt($data['__invoice_discount'] ?? $invoice->invoice_discount ?? 0);
        $invoiceShipping = $this->toInt($data['__invoice_shipping'] ?? $invoice->invoice_shipping ?? 0);
        $invoiceNet = $this->toInt($data['__invoice_net'] ?? $invoice->invoice_net_total ?? 0);
        $invoicePaidBase = $this->toInt($data['__invoice_paid'] ?? $invoice->invoice_paid ?? 0);
        $invoiceDueBase = $this->toInt($data['__invoice_due'] ?? $invoice->invoice_due ?? 0);

        $invoice->fill([
            'invoice_type' => $data['invoice_type'] ?? $invoice->invoice_type ?? 'ready',
            'sale_type' => $data['sale_type'] ?? $invoice->sale_type ?? 'retail',

            'notes' => $data['notes'] ?? $invoice->notes,
            'status' => $data['status'] ?? $invoice->status ?? 'preparing',

            // الفاتورة الأساسية
            'invoice_subtotal' => $invoiceSubtotal,
            'invoice_discount' => $invoiceDiscount,
            'invoice_shipping' => $invoiceShipping,
            'invoice_net_total' => $invoiceNet,
            'invoice_paid' => $invoicePaidBase,
            'invoice_due' => $invoiceDueBase,

            // أعمدة الإكسسوارات تبقى صفر (حالياً)
            'accessory_subtotal' => 0,
            'accessory_discount' => 0,
            'accessory_net_total' => 0,
            'accessory_paid' => 0,
            'accessory_due' => 0,
        ]);

        // -------- المجاميع النهائية من الـ __ fields --------
        $combinedSubtotal = $this->toInt(
            $data['__subtotal_combined'] ?? $invoiceSubtotal
        );
        $combinedDiscount = $this->toInt(
            $data['__discount_combined'] ?? $invoiceDiscount
        );
        $combinedShipping = $this->toInt(
            $data['__shipping_combined'] ?? $invoiceShipping
        );
        $combinedGrand = $this->toInt(
            $data['__grand_total_combined'] ?? $invoiceNet
        );
        $combinedPaid = $this->toInt(
            $data['__paid_combined'] ?? $invoicePaidBase
        );

        $combinedPaid = min($combinedPaid, $combinedGrand);
        $combinedDue = max($combinedGrand - $combinedPaid, 0);

        // تخزينها في أعمدة الفاتورة العامة (المجاميع النهائية)
        $invoice->subtotal = $combinedSubtotal;
        $invoice->discount_total = $combinedDiscount;
        $invoice->shipping_total = $combinedShipping;
        $invoice->grand_total = $combinedGrand;
        $invoice->paid_amount = $combinedPaid;
        $invoice->due_amount = $combinedDue;

        // حالة الدفع
        if ($combinedPaid <= 0) {
            $invoice->payment_status = 'unpaid';
        } elseif ($combinedDue <= 0) {
            $invoice->payment_status = 'paid';
        } else {
            $invoice->payment_status = 'partial';
        }

        $invoice->save();

        /** @var ClientAccountingService $acc */
        $acc = app(ClientAccountingService::class);

        // -------- المديونية الكاملة (حسب المجاميع النهائية) --------
        $grandFull = $this->toInt(
            $data['__grand_total_combined']
                ?? $invoice->invoice_net_total
                ?? 0
        );

        $paidFull = $this->toInt(
            $data['__paid_combined']
                ?? $invoice->invoice_paid
        );
        $paidFull = min($paidFull, $grandFull);
        $dueFull = max($grandFull - $paidFull, 0);

        // قيد مديونية الفاتورة
        if ($invoice->client_id && $grandFull > 0) {
            $acc->appendLedgerEntry(
                clientId: (int) $invoice->client_id,
                branchId: (int) $invoice->branch_id,
                entryType: 'invoice',
                referenceNumber: $invoice->invoice_number,
                referenceType: Invoice::class,
                referenceId: $invoice->id,
                debit: $grandFull,
                credit: 0,
                description: 'فاتورة بيع #'.($invoice->invoice_number ?? $invoice->id),
                entryDate: $invoice->invoice_date?->toDateString() ?? now('Asia/Baghdad')->toDateString(),
            );
        }

        // تحصيل الدفعة الأولى (إن وُجدت)
        if ($invoice->client_id && $paidFull > 0) {
            $acc->collectPaymentForInvoice(
                invoice: $invoice,
                amount: $paidFull,
                receiptType: ClientReceiptType::COLLECTION,
                paymentType: $data['payment_type'] ?? 'cash',
                paymentDetails: $data['payment_details'] ?? 'دفعة أولى على الفاتورة'
            );
        }

        // إعادة احتساب المبالغ من خلال المحاسبة
        $invoice = $acc->recomputeInvoicePaidAndRemaining($invoice->fresh());

        // المبلغ المتبقي الكلي
        $remainingCombined = $this->toInt($data['__due_combined'] ?? $dueFull);

        // معالجة الأقساط
        $this->handleInstallments($invoice, $data, $acc, $paidFull, $remainingCombined);

        // تحديث حساب العميل
        if ($invoice->client_id) {
            $acc->recomputeClientAccount((int) $invoice->client_id, (int) $invoice->branch_id);
        }

        // حركات المخزون للمنتجات الجاهزة (والإكسسوارات المبنية على product_variant)
        if ($invoice->invoice_type === 'ready') {
            $this->handleStockMovementsForReadyItems($invoice->fresh('items'), $data);
        }

        Log::info('CreateInvoiceService::processAfterItemsPersisted done', ['invoice_id' => $invoice->id]);

        return $invoice->fresh(['items']);
    }

    /* =========================================================
     * معالجة الأقساط
     * =======================================================*/

    protected function handleInstallments(
        Invoice $invoice,
        array $data,
        ClientAccountingService $acc,
        int $paidFull,
        int $remaining
    ): void {
        $settlementType = $data['settlement_type'] ?? 'cash';
        $installmentsCnt = (int) ($data['installments_count'] ?? 0);
        $firstDueDate = $data['first_due_date'] ?? null;
        $scheduleRowsRaw = $data['installments'] ?? null;

        $scheduleRows = [];
        if (is_array($scheduleRowsRaw)) {
            $seq = 1;
            foreach ($scheduleRowsRaw as $row) {
                if (! is_array($row)) {
                    continue;
                }

                $amt = $this->toInt($row['amount'] ?? 0);
                if ($amt <= 0) {
                    continue;
                }

                $scheduleRows[] = [
                    'sequence' => $seq++,
                    'amount' => $amt,
                    'due_date' => $row['due_date']
                        ?? $firstDueDate
                        ?? now('Asia/Baghdad')->addMonth()->toDateString(),
                    'status' => $row['status'] ?? 'pending',
                ];
            }
        }

        if (
            $invoice->client_id &&
            $settlementType === 'installment' &&
            $remaining > 0 &&
            (
                $installmentsCnt > 0 ||
                (is_array($scheduleRows) && count($scheduleRows) > 0)
            )
        ) {
            $acc->createInstallmentPlanForInvoice(
                invoice: $invoice,
                remainingAmount: $remaining,
                installmentsCount: ($installmentsCnt > 0
                    ? $installmentsCnt
                    : (is_array($scheduleRows) ? count($scheduleRows) : null)),
                startDate: $firstDueDate,
                downPayment: $paidFull,
                notes: $data['notes'] ?? null,
                scheduleRows: $scheduleRows,
            );
        }
    }

    /* =========================================================
     * إنشاء عناصر الفاتورة من مصفوفة items
     * =======================================================*/

    protected function createInvoiceItems(Invoice $invoice, array $items): void
    {
        $rows = [];

        foreach ($items as $row) {
            if (! is_array($row)) {
                continue;
            }

            $quantity = (int) ($row['quantity'] ?? 0);
            if ($quantity <= 0) {
                continue;
            }

            $unitPrice = $this->toInt($row['unit_price'] ?? 0);
            $measurement = isset($row['measurement'])
                ? (float) $row['measurement']
                : 1.00;

            // قراءة نوع السطر من الفورم (product / custom / accessory)
            $itemType = $row['item_type'] ?? null;

            // في حال لم يأتِ item_type لأي سبب، نحدده من شكل البيانات
            if (! $itemType) {
                if (! empty($row['product_variant_id']) || ! empty($row['product_id'])) {
                    // نفترض منتج
                    $itemType = 'product';
                } else {
                    // سطر غير واضح، نتجاوزه
                    continue;
                }
            }

            $lineTotal = $this->toInt($row['line_total'] ?? 0);

            if ($lineTotal <= 0) {
                if ($itemType === 'custom') {
                    // مخصص: سعر * قياس * كمية
                    $lineTotal = (int) ($unitPrice * $measurement * $quantity);
                } else {
                    // منتج / إكسسوار: سعر * كمية
                    $lineTotal = $unitPrice * $quantity;
                }
            }

            // السطور المبنية على منتجات (منتج جاهز / إكسسوار)
            $isProductBased = in_array($itemType, ['product', 'accessory'], true);

            $rows[] = [
                'invoice_id' => $invoice->id,

                // نوع السطر
                'item_type' => $itemType,

                // المنتجات / الإكسسوارات
                'product_id' => $isProductBased ? ($row['product_id'] ?? null) : null,
                'product_variant_id' => $isProductBased ? ($row['product_variant_id'] ?? null) : null,

                'measurement' => $measurement,
                'unit_price' => $unitPrice,
                'quantity' => $quantity,
                'line_total' => $lineTotal,
                'notes' => $row['notes'] ?? null,
            ];
        }

        if (! empty($rows)) {
            InvoiceItem::insert($rows);
        }
    }

    /* =========================================================
     * حركات المخزون للمنتجات الجاهزة
     * =======================================================*/

    protected function handleStockMovementsForReadyItems(Invoice $invoice, array $data): void
    {
        if ($invoice->invoice_type !== 'ready') {
            return;
        }

        $items = $invoice->items()
            ->whereNotNull('product_variant_id')
            ->get();

        if ($items->isEmpty()) {
            return;
        }

        $branchId = (int) $invoice->branch_id;
        $warehouseId = null;

        if (! empty($data['warehouse_id'])) {
            $warehouseId = (int) $data['warehouse_id'];
        } else {
            $primary = BranchWarehouse::query()
                ->where('branch_id', $branchId)
                ->where('is_primary', true)
                ->first();

            if ($primary) {
                $warehouseId = (int) $primary->warehouse_id;
            } else {
                $any = BranchWarehouse::query()
                    ->where('branch_id', $branchId)
                    ->first();

                $warehouseId = $any?->warehouse_id;
            }
        }

        if (! $warehouseId) {
            Log::warning('CreateInvoiceService - no warehouse found', [
                'branch_id' => $branchId,
                'invoice_id' => $invoice->id,
            ]);

            return;
        }

        $userId = (int) (function_exists('user_info') ? (user_info('id') ?? 0) : 0);
        $movedAt = $invoice->invoice_date
            ? $invoice->invoice_date->copy()->startOfDay()
            : now('Asia/Baghdad');

        foreach ($items as $item) {
            $qty = (int) $item->quantity;
            if ($qty <= 0 || ! $item->product_variant_id) {
                continue;
            }

            $balance = StockBalance::query()
                ->where('warehouse_id', $warehouseId)
                ->where('product_variant_id', $item->product_variant_id)
                ->first();

            $unitCost = (int) ($balance?->last_unit_cost ?? 0);
            $totalCost = $unitCost * $qty;

            StockMovement::create([
                'warehouse_id' => $warehouseId,
                'product_variant_id' => $item->product_variant_id,
                'direction' => MovementDirection::OUT,
                'source_type' => Invoice::class,
                'source_id' => $invoice->id,
                'quantity' => $qty,
                'unit_cost' => $unitCost,
                'total_cost' => $totalCost,
                'moved_at' => $movedAt,
                'reference' => $invoice->invoice_number,
                'note' => $invoice->notes ?? 'بيع جاهز من الفاتورة #'.($invoice->invoice_number ?? $invoice->id),
                'user_id' => $userId,
            ]);
        }
    }
}
