<?php

namespace App\Services\Finance;

use App\Models\Expense;
use App\Models\Vault;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

/**
 * تسجيل مصروف وسحب قيمته من الخزنة بذريّة (Transaction).
 *
 * ملاحظات:
 * - لا تُرمى استثناءات من هذه الخدمة عند نقص/انعدام الرصيد؛ تُعاد null ويُترك
 *   للمُستدعي (صفحة CreateExpense) مسؤولية إظهار الإشعار وإيقاف العملية.
 * - تُجرد الخدمة الفواصل من amount في الطبقة الأعلى (الصفحة) قبل تمريرها هنا.
 */
class RecordExpense
{
    /**
     * إنشاء مصروف وسحب قيمته من خزنة الاشتراك الحالي بشكل ذرّي.
     *
     * @param  int         $categoryId   معرف فئة المصروف
     * @param  int         $amount       قيمة المصروف (بالدينار كوحدة صحيحة موجبة)
     * @param  string|null $description  وصف المصروف
     * @param  string|null $spentAt      تاريخ/وقت الصرف (Y-m-d أو Y-m-d H:i:s) اختياري
     * @param  int|null    $orderId      ربط اختياري بحجز/فاتورة
     * @return \App\Models\Expense|null  يعيد Expense عند نجاح العملية، أو null عند الفشل (مثل رصيد غير كافٍ)
     */
    public static function create(
        int $categoryId,
        int $amount,
        ?string $description = null,
        ?string $spentAt = null,
        ?int $orderId = null,
    ): ?Expense {
        // حماية أساسية: مبلغ غير صالح
        if ($amount <= 0) {
            return null;
        }
        $userId = (int) (Auth::id() ?? 0);

        // جلب خزنة الفرع
        $vault = Vault::forBranch(user_info('branch_id'));

        // شروط الرصيد (بدون استثناءات)
        if ($vault->balance <= 0) {
            return null;
        }
        if ($vault->balance < $amount) {
            return null;
        }

        // تنفيذ العملية بذريّة
        return DB::transaction(function () use ($categoryId, $amount, $description, $spentAt, $userId, $vault) {
            // 1) إنشاء المصروف
            /** @var Expense $expense */
            $expense = Expense::create([
                'expense_category_id' => $categoryId,
                'amount'              => $amount,
                'description'         => $description,
                'spent_at'            => $spentAt,
                'branch_id'           => user_info('branch_id'),
                'user_id'             => $userId ?: null,
            ]);

            // 2) السحب من الخزنة وربط الحركة بالمصروف كمصدر مورف
            // ملاحظة: Vault::withdraw يتوقع التوقيع: withdraw(Model $source, int $amount, ?string $description = null, ?int $userId = null)
            $vault->withdraw(
                source: $expense,
                amount: $amount,
                description: $description ?: ('سحب مقابل مصروف #' . $expense->id),
                userId: $userId ?: null,
            );

            return $expense;
        });
    }
}
