<?php

namespace App\Services\Finance;

use App\Models\Order;
use App\Models\Invoice;
use App\Models\ServiceInvoice;
use App\Models\Installment;
use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Models\Vault;
use Illuminate\Database\Eloquent\Model;

class DebtSettlementService
{
    /**
     * يُستدعى بعد إنشاء الطلب مباشرةً. لا يفتح Transaction (المستدعي مسؤول عنها).
     * - ينشئ سند قبض واحد عند وجود دفعات أولية في order/serviceInvoice/invoice.
     * - ينشئ بنود السند + أقساط + إيداعات خزنة + تحديث paid/remaining.
     *
     * مبالغ الدفعات تُمرّر من صفحة الإنشاء: $orderUpfront, $serviceUpfront, $foodUpfront
     */
    public function handleAfterOrderCreate(
        Order $order,
        ?ServiceInvoice $serviceInvoice,
        ?Invoice $foodInvoice,
        ?Vault $vault,
        int $actorUserId,
        ?string $method = 'cash',
        int $orderUpfront = 0,
        int $serviceUpfront = 0,
        int $foodUpfront = 0
    ): ?Receipt {
        // 0) مبالغ الدفعات الأولية
        $orderAmt = max(0, (int) $orderUpfront);
        $svcAmt   = max(0, (int) $serviceUpfront);
        $foodAmt  = max(0, (int) $foodUpfront);

        $totalUpfront = $orderAmt + $svcAmt + $foodAmt;
        if ($totalUpfront <= 0) {
            return null; // لا توجد دفعات أولية
        }

        // 1) المتبقي "قبل السداد" للكيانات الجديدة (من الإجمالي - الخصم فقط)
        $beforeOrderRem = max(0, (int) $order->total_price  - (int) $order->discount);
        $beforeSvcRem   = $serviceInvoice ? max(0, (int) $serviceInvoice->total_amount - (int) $serviceInvoice->discount) : 0;
        $beforeFoodRem  = $foodInvoice    ? max(0, (int) $foodInvoice->total_amount    - (int) $foodInvoice->discount)    : 0;

        // 2) مديونية العميل السابقة (قبل هذا الطلب/الفواتير) باستثناء الكيانات الحالية
        $prevDebt = $this->getClientPreviousDebt($order);

        // 3) احسب المتبقي "بعد السداد" للكيانات الجديدة
        $afterOrderRem = max(0, $beforeOrderRem - min($orderAmt, $beforeOrderRem));
        $afterSvcRem   = max(0, $beforeSvcRem   - min($svcAmt,   $beforeSvcRem));
        $afterFoodRem  = max(0, $beforeFoodRem  - min($foodAmt,  $beforeFoodRem));

        // المديونية بعد السند = الديون السابقة + المتبقي من الكيانات الجديدة بعد تطبيق الدفعات
        $clientDebtAfter = $prevDebt + $afterOrderRem + $afterSvcRem + $afterFoodRem;

        // 4) إنشاء رأس السند (نمرّر before/after مباشرةً لتجنّب NULL)
        $receipt = Receipt::create([
            'subscription_id'      => $order->subscription_id,
            'client_id'            => $order->client_id,
            'user_id'              => $actorUserId,
            'vault_id'             => $vault?->id,
            'vault_transaction_id' => null, // قد نحدّثه لاحقاً إن وُجد إيداع وحيد
            'number'               => $this->generateNumber((int) $order->subscription_id),
            'issue_date'           => now()->toDateString(),
            'total_amount'         => $totalUpfront,
            'client_debt_before'   => $prevDebt,
            'client_debt_after'    => $clientDebtAfter,
            'method'               => $method ?: 'cash',
            'notes'                => 'دفعات أولية عند إنشاء الطلب',
            'status'               => 'posted',
        ]);

        // 5) بنود + أقساط + إيداعات + تحديث paid/remaining
        $vaultTxIds = [];

        if ($orderAmt > 0) {
            $txId = $this->applyLine(
                $receipt,
                $order,
                $orderAmt,
                $beforeOrderRem,
                $vault,
                $actorUserId,
                $method,
                $this->buildOrderDesc($order)
            );
            if ($txId) $vaultTxIds[] = $txId;
        }

        if ($serviceInvoice && $svcAmt > 0) {
            $txId = $this->applyLine(
                $receipt,
                $serviceInvoice,
                $svcAmt,
                $beforeSvcRem,
                $vault,
                $actorUserId,
                $method,
                "فاتورة خدمات - حجز #{$order->id}"
            );
            if ($txId) $vaultTxIds[] = $txId;
        }

        if ($foodInvoice && $foodAmt > 0) {
            $txId = $this->applyLine(
                $receipt,
                $foodInvoice,
                $foodAmt,
                $beforeFoodRem,
                $vault,
                $actorUserId,
                $method,
                "فاتورة طعام - حجز #{$order->id}"
            );
            if ($txId) $vaultTxIds[] = $txId;
        }

        // لو كان هناك معاملة خزنة واحدة فقط، اربطها في رأس السند
        if (count($vaultTxIds) === 1) {
            $receipt->vault_transaction_id = $vaultTxIds[0];
            $receipt->save();
        }

        return $receipt;
    }

    /**
     * ينشئ بند سند + قسط + إيداع خزنة + يحدّث paid/remaining للكيان القابل للدفع.
     * يُعيد معرّف معاملة الخزنة إن وُجد (otherwise null).
     */
    protected function applyLine(
        Receipt $receipt,
        Model $payable, // Order | ServiceInvoice | Invoice
        int $amount,
        int $remainingBefore,
        ?Vault $vault,
        int $actorUserId,
        ?string $method = 'cash',
        ?string $description = null
    ): ?int {
        // حماية من over-payment
        $lineAmount = min($amount, max(0, $remainingBefore));

        // 1) بند السند (مع المتبقي قبل/بعد)
        $line = ReceiptLine::create([
            'receipt_id'       => $receipt->id,
            'payable_type'     => get_class($payable),
            'payable_id'       => $payable->getKey(),
            'amount'           => $lineAmount,
            'description'      => $description,
            'remaining_before' => $remainingBefore,
            'remaining_after'  => max(0, $remainingBefore - $lineAmount),
        ]);

        // 2) قسط
        $installment = Installment::create([
            'subscription_id'      => $receipt->subscription_id,
            'client_id'            => $receipt->client_id,
            'vault_id'             => $vault?->id,
            'vault_transaction_id' => null, // سنحدّثه لو توفر
            'payable_type'         => get_class($payable),
            'payable_id'           => $payable->getKey(),
            'amount'               => $lineAmount,
            'method'               => $method ?: 'cash',
            'description'          => $description,
            'paid_at'              => now(),
            'user_id'              => $actorUserId,
            'status'               => 'completed',
        ]);

        // اربط البند بالقسط
        $line->update(['installment_id' => $installment->id]);

        // 3) إيداع في الخزنة + التقاط معرف المعاملة
        $txId = null;
        if ($vault && $lineAmount > 0) {
            $tx = method_exists($payable, 'depositToVault')
                ? $payable->depositToVault($vault, $lineAmount, $description ?? 'دفعة', $actorUserId)
                : $vault->deposit($payable, $lineAmount, $description, $actorUserId);

            if (is_object($tx) && isset($tx->id)) {
                $txId = (int) $tx->id;
            } elseif (is_numeric($tx)) {
                $txId = (int) $tx;
            }

            if ($txId) {
                $installment->vault_transaction_id = $txId;
                $installment->save();
            }
        }

        // 4) تحديث paid/remaining للكيان
        $this->incrementPaidAndRemaining($payable, $lineAmount);

        return $txId;
    }

    /**
     * تحديث الحقول paid/remaining على Order/ServiceInvoice/Invoice.
     */
    protected function incrementPaidAndRemaining(Model $payable, int $lineAmount): void
    {
        $totalField = $payable instanceof Order ? 'total_price' : 'total_amount';
        $total      = (int) ($payable->{$totalField} ?? 0);
        $discount   = (int) ($payable->discount ?? 0);
        $newPaid    = (int) ($payable->paid ?? 0) + $lineAmount;
        $newRemaining = max(0, $total - $discount - $newPaid);

        $payable->update([
            'paid'      => $newPaid,
            'remaining' => $newRemaining,
        ]);
    }

    /**
     * يحسب مديونية العميل السابقة بدون احتساب الكيانات الجاري إنشاؤها.
     * يعتمد على الربط بجدول orders لأن service_invoices/invoices لا يملكان client_id.
     */
    protected function getClientPreviousDebt(Order $currentOrder): int
    {
        $clientId = (int) $currentOrder->client_id;
        $currentOrderId = (int) $currentOrder->id;

        // ديون الطلبات الأخرى لنفس العميل
        $ordersDebt = Order::query()
            ->where('client_id', $clientId)
            ->where('id', '!=', $currentOrderId)
            ->whereNull('deleted_at')
            ->sum('remaining');

        // ديون فواتير الخدمات السابقة عبر orders
        $svcDebt = ServiceInvoice::query()
            ->join('orders', 'orders.id', '=', 'service_invoices.order_id')
            ->where('orders.client_id', $clientId)
            ->where('orders.id', '!=', $currentOrderId)
            ->whereNull('service_invoices.deleted_at')
            ->sum('service_invoices.remaining');

        // ديون فواتير الطعام السابقة عبر orders
        $foodDebt = Invoice::query()
            ->join('orders', 'orders.id', '=', 'invoices.order_id')
            ->where('orders.client_id', $clientId)
            ->where('orders.id', '!=', $currentOrderId)
            ->whereNull('invoices.deleted_at')
            ->sum('invoices.remaining');

        return (int) $ordersDebt + (int) $svcDebt + (int) $foodDebt;
    }

    /**
     * توليد رقم سند بسيط (يفضّل استبداله لاحقاً بنظام تسلسل رسمي).
     */
    protected function generateNumber(int $subscriptionId): string
    {
        return 'RC-' . $subscriptionId . '-' . now()->format('YmdHis');
    }

    protected function buildOrderDesc(Order $order): string
    {
        $clientName = optional($order->client)->name ?? 'عميل';
        $h = trim(($order->husband ?? '') . ' & ' . ($order->wife ?? ''));
        return "دفعة حجز - {$clientName}" . ($h ? " - {$h}" : '');
    }
}
