<?php

namespace App\Services\Accounting\Receipts;

use App\Models\ClientReceipt;
use App\Models\Invoice;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class UpdateClientReceiptService
{
    /* ========= Helpers: Integer ========= */

    protected function toInt(null|string|int $v): int
    {
        if ($v === null || $v === '') {
            return 0;
        }
        if (is_int($v)) {
            return $v;
        }

        $s = (string) $v;

        $s = strtr($s, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        return (int) str_replace([',', ' ', '٫', '.'], '', $s);
    }

    protected function invoiceGrandTotal(Invoice $inv): int
    {
        $grand = (int) ($inv->grand_total ?? 0);
        if ($grand > 0) {
            return $grand;
        }

        $subtotal = (int) ($inv->subtotal ?? 0);
        $discount = (int) ($inv->discount_total ?? 0);
        $shipping = (int) ($inv->shipping_total ?? 0);

        return max(0, $subtotal - $discount + $shipping);
    }

    /**
     * حذف تخصيصات الوصل + إرجاع أثرها من الفواتير
     * يرجع invoice_ids المتأثرة
     */
    protected function rollbackAllocations(ClientReceipt $receipt): array
    {
        if (! Schema::hasTable('receipt_invoice_allocations')) {
            return [];
        }

        $allocs = DB::table('receipt_invoice_allocations')
            ->where('client_receipt_id', $receipt->id)
            ->get(['invoice_id', 'paid_amount']);

        $byInvoice = [];
        foreach ($allocs as $a) {
            $iid = (int) ($a->invoice_id ?? 0);
            $amt = (int) ($a->paid_amount ?? 0);
            if ($iid <= 0 || $amt <= 0) {
                continue;
            }
            $byInvoice[$iid] = ($byInvoice[$iid] ?? 0) + $amt;
        }

        // حذف التخصيصات
        DB::table('receipt_invoice_allocations')
            ->where('client_receipt_id', $receipt->id)
            ->delete();

        // rollback على الفواتير
        foreach ($byInvoice as $invoiceId => $allocPaid) {
            /** @var Invoice|null $inv */
            $inv = Invoice::query()->find($invoiceId);
            if (! $inv) {
                continue;
            }

            $grand = $this->invoiceGrandTotal($inv);
            $oldPaid = (int) ($inv->paid_amount ?? 0);

            $newPaid = max(0, $oldPaid - (int) $allocPaid);
            $newDue = max(0, $grand - $newPaid);

            $paymentStatus = $newDue <= 0
                ? 'paid'
                : ($newPaid > 0 ? 'partial' : 'unpaid');

            $payload = [
                'paid_amount' => $newPaid,
                'due_amount' => $newDue,
            ];

            if (Schema::hasColumn('invoices', 'payment_status')) {
                $payload['payment_status'] = $paymentStatus;
            }

            // لو عندك grand_total = 0 سابقاً، نقدر نثبته هنا (اختياري)
            if (Schema::hasColumn('invoices', 'grand_total') && (int) ($inv->grand_total ?? 0) <= 0) {
                $payload['grand_total'] = $grand;
            }

            $inv->forceFill($payload)->saveQuietly();
        }

        return array_keys($byInvoice);
    }

    /* ========= Ledger ========= */

    protected function appendLedgerEntry(
        int $clientId,
        int $branchId,
        string $entryType,
        string $referenceNumber,
        string $referenceType,
        int $referenceId,
        int $debit,
        int $credit,
        string $description,
        string $entryDate
    ): void {
        $debit = max(0, $this->toInt($debit));
        $credit = max(0, $this->toInt($credit));

        $last = DB::table('client_ledger_entries')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->orderByDesc('entry_date')
            ->orderByDesc('id')
            ->value('balance');

        $balance = (int) ($last ?? 0) + $debit - $credit;

        DB::table('client_ledger_entries')->insert([
            'client_id' => $clientId,
            'branch_id' => $branchId,
            'entry_type' => $entryType,
            'reference_number' => $referenceNumber,
            'reference_type' => Schema::hasColumn('client_ledger_entries', 'reference_type') ? $referenceType : null,
            'reference_id' => Schema::hasColumn('client_ledger_entries', 'reference_id') ? $referenceId : null,
            'debit' => $debit,
            'credit' => $credit,
            'balance' => $balance,
            'entry_date' => $entryDate,
            'description' => $description,
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }

    protected function rebuildClientLedgerBalances(int $clientId, int $branchId): void
    {
        if (! Schema::hasTable('client_ledger_entries') || ! Schema::hasColumn('client_ledger_entries', 'balance')) {
            return;
        }

        $entries = DB::table('client_ledger_entries')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->orderBy('entry_date')
            ->orderBy('id')
            ->get(['id', 'debit', 'credit']);

        $balance = 0;
        foreach ($entries as $e) {
            $balance += (int) ($e->debit ?? 0) - (int) ($e->credit ?? 0);
            DB::table('client_ledger_entries')->where('id', $e->id)->update([
                'balance' => $balance,
                'updated_at' => now(),
            ]);
        }
    }

    /* ========= ClientAccount ========= */

    protected function recomputeClientAccount(int $clientId, int $branchId): void
    {
        if (! class_exists(\App\Models\ClientAccount::class)) {
            return;
        }

        $invQuery = Invoice::query()
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->whereNull('deleted_at');

        if (Schema::hasColumn('invoices', 'status')) {
            $invQuery->whereNotIn('status', ['canceled', 'cancelled']);
        }

        $receivableTotal = (int) $invQuery->sum('grand_total');
        $receivablePaid = (int) $invQuery->sum('paid_amount');
        $receivableRemaining = (int) $invQuery->sum('due_amount');

        // إجمالي الوصولات
        $receiptsTotal = (int) DB::table('client_receipts')
            ->whereNull('deleted_at')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->sum('total_paid');

        // إجمالي ما تم تخصيصه على الفواتير
        $allocTotal = 0;

        if (Schema::hasTable('receipt_invoice_allocations')) {
            $alloc = DB::table('receipt_invoice_allocations');

            if (Schema::hasColumn('receipt_invoice_allocations', 'client_id')) {
                $alloc->where('client_id', $clientId);
            } else {
                $alloc->whereIn('client_receipt_id', function ($q) use ($clientId, $branchId) {
                    $q->select('id')
                        ->from('client_receipts')
                        ->whereNull('deleted_at')
                        ->where('client_id', $clientId)
                        ->where('branch_id', $branchId);
                });
            }

            if (Schema::hasColumn('receipt_invoice_allocations', 'branch_id')) {
                $alloc->where('branch_id', $branchId);
            }

            $allocTotal = (int) $alloc->sum('paid_amount');
        }

        $advance = max(0, $receiptsTotal - $allocTotal);

        \App\Models\ClientAccount::query()->updateOrCreate(
            ['client_id' => $clientId, 'branch_id' => $branchId],
            [
                'receivable_total' => $receivableTotal,
                'receivable_paid' => $receivablePaid,
                'receivable_remaining' => $receivableRemaining,
                'client_advance' => $advance,
                'last_transaction_at' => now(),
            ]
        );
    }

    /* ========= Execute ========= */

    public function execute(ClientReceipt $receipt, array $data): ClientReceipt
    {
        $tz = config('app.timezone', 'Asia/Baghdad');

        $newDate = ! empty($data['receipt_date'])
            ? Carbon::parse($data['receipt_date'], $tz)->format('Y-m-d')
            : (string) ($receipt->receipt_date ?? now($tz)->format('Y-m-d'));

        $newClient = (int) ($data['client_id'] ?? $receipt->client_id);
        $newBranch = (int) ($data['branch_id'] ?? $receipt->branch_id);

        // ✅ المبلغ الصحيح: نستخدم total_paid أولاً
        $newPaid = $this->toInt($data['total_paid'] ?? $data['paid_amount'] ?? $data['total_amount'] ?? $receipt->total_paid);

        // ✅ طريقة الدفع فقط
        $newPaymentMethodId = (int) ($data['payment_method_id'] ?? ($receipt->payment_method_id ?? 0));

        $pdetails = $data['payment_details'] ?? $receipt->payment_details;

        return DB::transaction(function () use ($receipt, $newDate, $newClient, $newBranch, $newPaid, $newPaymentMethodId, $pdetails) {

            $oldClient = (int) $receipt->client_id;
            $oldBranch = (int) $receipt->branch_id;
            $oldPaid = (int) $receipt->total_paid;

            // 1) rollback تخصيصات الوصل القديمة + تعديل الفواتير القديمة
            $this->rollbackAllocations($receipt);

            // 2) تعديل الخزنة بفارق المبلغ (وصل قبض collection = deposit سابقاً)
            $delta = $newPaid - $oldPaid;
            if ($delta !== 0 && function_exists('current_vault') && ($vault = current_vault())) {
                $uid = (int) (function_exists('user_info') ? (user_info('id') ?? 0) : 0);

                if ($delta > 0) {
                    $vault->deposit(
                        source: $receipt,
                        amount: $delta,
                        description: 'تعديل — زيادة على وصل '.$receipt->receipt_number,
                        userId: $uid
                    );
                } else {
                    $vault->withdraw(
                        source: $receipt,
                        amount: (int) abs($delta),
                        description: 'تعديل — تخفيض على وصل '.$receipt->receipt_number,
                        userId: $uid
                    );
                }
            }

            // 3) تحديث رأس الوصل (بدون payment_type)
            $receipt->update([
                'receipt_date' => $newDate,
                'client_id' => $newClient,
                'branch_id' => $newBranch,
                'total_paid' => $newPaid,
                'total_amount' => Schema::hasColumn('client_receipts', 'total_amount') ? $newPaid : ($receipt->total_amount ?? null),
                'payment_method_id' => Schema::hasColumn('client_receipts', 'payment_method_id') ? ($newPaymentMethodId ?: null) : null,
                'payment_details' => $pdetails,
            ]);

            // 4) إعادة توزيع على أقدم الفواتير حسب العميل/الفرع الجديد
            /** @var CreateClientReceiptService $creator */
            $creator = app(CreateClientReceiptService::class);

            $allocatedTotal = 0;
            $creator->allocateAcrossOldestInvoices($receipt, $newPaid, $allocatedTotal);

            $receipt->forceFill([
                'total_remaining' => max(0, $newPaid - (int) $allocatedTotal),
                'updated_at' => now(),
            ])->saveQuietly();

            // 5) حذف قيد الأستاذ السابق (للقديم)
            $ledger = DB::table('client_ledger_entries')
                ->where('client_id', $oldClient)
                ->where('branch_id', $oldBranch)
                ->where('reference_number', $receipt->receipt_number)
                ->whereIn('entry_type', ['receipt', 'payment']);

            if (Schema::hasColumn('client_ledger_entries', 'reference_type')) {
                $ledger->where('reference_type', ClientReceipt::class);
            }
            if (Schema::hasColumn('client_ledger_entries', 'reference_id')) {
                $ledger->where('reference_id', $receipt->id);
            }

            $ledger->delete();

            // 6) إنشاء قيد جديد (Receipt = Credit)
            $this->appendLedgerEntry(
                clientId: $newClient,
                branchId: $newBranch,
                entryType: 'receipt',
                referenceNumber: $receipt->receipt_number,
                referenceType: ClientReceipt::class,
                referenceId: (int) $receipt->id,
                debit: 0,
                credit: $newPaid,
                description: 'تعديل — قبض من العميل — وصل '.$receipt->receipt_number,
                entryDate: $newDate
            );

            // 7) تحديث حساب العميل (للجديد + للقديم إذا تغير)
            $this->recomputeClientAccount($newClient, $newBranch);
            if ($oldClient !== $newClient || $oldBranch !== $newBranch) {
                $this->recomputeClientAccount($oldClient, $oldBranch);
            }

            // 8) إعادة بناء balances للأستاذ (للجديد + للقديم إذا تغير)
            $this->rebuildClientLedgerBalances($newClient, $newBranch);
            if ($oldClient !== $newClient || $oldBranch !== $newBranch) {
                $this->rebuildClientLedgerBalances($oldClient, $oldBranch);
            }

            return $receipt->fresh();
        });
    }
}
