<?php

namespace App\Services\Accounting\Receipts;

use App\Models\ClientReceipt;
use App\Models\Invoice;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class DeleteClientReceiptService
{
    /**
     * حذف وصل قبض مع تنظيف كامل:
     * تخصيصات/فواتير/خزنة/دفتر الأستاذ/حساب العميل
     */
    public function execute(ClientReceipt $receipt, bool $force = false): void
    {
        DB::transaction(function () use ($receipt, $force) {

            $clientId = (int) $receipt->client_id;
            $branchId = (int) $receipt->branch_id;
            $paid = (int) $receipt->total_paid;

            // 1) تجميع تخصيصات الوصل
            $byInvoice = [];

            if (Schema::hasTable('receipt_invoice_allocations')) {
                $allocs = DB::table('receipt_invoice_allocations')
                    ->where('client_receipt_id', $receipt->id)
                    ->get(['invoice_id', 'paid_amount']);

                foreach ($allocs as $a) {
                    $iid = (int) ($a->invoice_id ?? 0);
                    $amt = (int) ($a->paid_amount ?? 0);
                    if ($iid <= 0 || $amt <= 0) {
                        continue;
                    }
                    $byInvoice[$iid] = ($byInvoice[$iid] ?? 0) + $amt;
                }

                // 2) حذف التخصيصات
                DB::table('receipt_invoice_allocations')
                    ->where('client_receipt_id', $receipt->id)
                    ->delete();
            }

            // 3) إرجاع أثر التخصيص على الفواتير
            foreach ($byInvoice as $invoiceId => $allocPaid) {
                /** @var Invoice|null $inv */
                $inv = Invoice::query()->find($invoiceId);
                if (! $inv) {
                    continue;
                }

                $grand = (int) ($inv->grand_total ?? 0);
                if ($grand <= 0) {
                    $subtotal = (int) ($inv->subtotal ?? 0);
                    $discount = (int) ($inv->discount_total ?? 0);
                    $shipping = (int) ($inv->shipping_total ?? 0);
                    $grand = max(0, $subtotal - $discount + $shipping);
                }

                $oldPaid = (int) ($inv->paid_amount ?? 0);

                $newPaid = max(0, $oldPaid - (int) $allocPaid);
                $newDue = max(0, $grand - $newPaid);

                $paymentStatus = $newDue <= 0
                    ? 'paid'
                    : ($newPaid > 0 ? 'partial' : 'unpaid');

                $payload = [
                    'paid_amount' => $newPaid,
                    'due_amount' => $newDue,
                ];

                if (Schema::hasColumn('invoices', 'payment_status')) {
                    $payload['payment_status'] = $paymentStatus;
                }

                if (Schema::hasColumn('invoices', 'grand_total') && (int) ($inv->grand_total ?? 0) <= 0) {
                    $payload['grand_total'] = $grand;
                }

                $inv->forceFill($payload)->saveQuietly();
            }

            // 4) تعديل الخزنة: وصل قبض => نسحب نفس المبلغ
            if ($paid > 0 && function_exists('current_vault') && ($vault = current_vault())) {
                $vault->withdraw(
                    source: $receipt,
                    amount: $paid,
                    description: 'إلغاء وصل '.$receipt->receipt_number,
                    userId: (int) (function_exists('user_info') ? (user_info('id') ?? 0) : 0),
                );
            }

            // 5) حذف قيد الأستاذ الخاص بالوصل
            if (Schema::hasTable('client_ledger_entries')) {
                $ledger = DB::table('client_ledger_entries')
                    ->where('client_id', $clientId)
                    ->where('branch_id', $branchId)
                    ->whereIn('entry_type', ['receipt', 'payment'])
                    ->where('reference_number', $receipt->receipt_number);

                if (Schema::hasColumn('client_ledger_entries', 'reference_type')) {
                    $ledger->where('reference_type', ClientReceipt::class);
                }
                if (Schema::hasColumn('client_ledger_entries', 'reference_id')) {
                    $ledger->where('reference_id', $receipt->id);
                }

                $ledger->delete();
            }

            // 6) حذف الوصل (سوفت/نهائي)
            $force ? $receipt->forceDelete() : $receipt->delete();

            // 7) تحديث حساب العميل
            $this->recomputeClientAccount($clientId, $branchId);

            // 8) إعادة بناء balances للأستاذ
            $this->rebuildClientLedgerBalances($clientId, $branchId);
        });
    }

    /* ========= ClientAccount ========= */

    protected function recomputeClientAccount(int $clientId, int $branchId): void
    {
        if (! class_exists(\App\Models\ClientAccount::class)) {
            return;
        }

        $invQuery = Invoice::query()
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->whereNull('deleted_at');

        if (Schema::hasColumn('invoices', 'status')) {
            $invQuery->whereNotIn('status', ['canceled', 'cancelled']);
        }

        $receivableTotal = (int) $invQuery->sum('grand_total');
        $receivablePaid = (int) $invQuery->sum('paid_amount');
        $receivableRemaining = (int) $invQuery->sum('due_amount');

        $receiptsTotal = (int) DB::table('client_receipts')
            ->whereNull('deleted_at')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->sum('total_paid');

        $allocTotal = 0;
        if (Schema::hasTable('receipt_invoice_allocations')) {
            $alloc = DB::table('receipt_invoice_allocations');

            if (Schema::hasColumn('receipt_invoice_allocations', 'client_id')) {
                $alloc->where('client_id', $clientId);
            } else {
                $alloc->whereIn('client_receipt_id', function ($q) use ($clientId, $branchId) {
                    $q->select('id')
                        ->from('client_receipts')
                        ->whereNull('deleted_at')
                        ->where('client_id', $clientId)
                        ->where('branch_id', $branchId);
                });
            }

            if (Schema::hasColumn('receipt_invoice_allocations', 'branch_id')) {
                $alloc->where('branch_id', $branchId);
            }

            $allocTotal = (int) $alloc->sum('paid_amount');
        }

        $advance = max(0, $receiptsTotal - $allocTotal);

        \App\Models\ClientAccount::query()->updateOrCreate(
            ['client_id' => $clientId, 'branch_id' => $branchId],
            [
                'receivable_total' => $receivableTotal,
                'receivable_paid' => $receivablePaid,
                'receivable_remaining' => $receivableRemaining,
                'client_advance' => $advance,
                'last_transaction_at' => now(),
            ]
        );
    }

    /* ========= Ledger rebuild ========= */

    protected function rebuildClientLedgerBalances(int $clientId, int $branchId): void
    {
        if (! Schema::hasTable('client_ledger_entries') || ! Schema::hasColumn('client_ledger_entries', 'balance')) {
            return;
        }

        $entries = DB::table('client_ledger_entries')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->orderBy('entry_date')
            ->orderBy('id')
            ->get(['id', 'debit', 'credit']);

        $balance = 0;
        foreach ($entries as $e) {
            $balance += (int) ($e->debit ?? 0) - (int) ($e->credit ?? 0);
            DB::table('client_ledger_entries')->where('id', $e->id)->update([
                'balance' => $balance,
                'updated_at' => now(),
            ]);
        }
    }
}
