<?php

namespace App\Services\Accounting\Receipts;

use App\Models\ClientAccount;
use App\Models\ClientLedgerEntry;
use App\Models\ClientReceipt;
use App\Models\Invoice;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class CreateClientReceiptService
{
    /* ===================== أدوات رقمية (أعداد صحيحة فقط) ===================== */

    protected function toInt(null|string|int $v): int
    {
        if ($v === null || $v === '') {
            return 0;
        }
        if (is_int($v)) {
            return $v;
        }

        $s = (string) $v;

        $s = strtr($s, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        // أرقام فقط + إشارة سالب إن وجدت
        $s = str_replace([',', ' ', '٫', '.'], '', $s);

        // حماية: لو صار نص غير رقمي
        if ($s === '' || ! preg_match('/^-?\d+$/', $s)) {
            return 0;
        }

        return (int) $s;
    }

    protected function pickCol(string $table, array $candidates, string $fallback): string
    {
        foreach ($candidates as $c) {
            if (Schema::hasColumn($table, $c)) {
                return $c;
            }
        }

        return $fallback;
    }

    protected function hasAllocTable(): bool
    {
        return Schema::hasTable('receipt_invoice_allocations');
    }

    protected function allocHasClientId(): bool
    {
        return $this->hasAllocTable() && Schema::hasColumn('receipt_invoice_allocations', 'client_id');
    }

    protected function allocHasBranchId(): bool
    {
        return $this->hasAllocTable() && Schema::hasColumn('receipt_invoice_allocations', 'branch_id');
    }

    protected function allocHasInvoiceDate(): bool
    {
        return $this->hasAllocTable() && Schema::hasColumn('receipt_invoice_allocations', 'invoice_date');
    }

    /* ===================== عدّاد أرقام الوصول ===================== */

    protected function nextCounterNumber(int $branchId): array
    {
        $tz = config('app.timezone', 'Asia/Baghdad');
        $year = now($tz)->year;

        $seq = DB::transaction(function () use ($branchId, $year) {
            $row = DB::table('receipt_counters')
                ->where('branch_id', $branchId)
                ->where('year', $year)
                ->lockForUpdate()
                ->first();

            $next = (int) ($row->current_sequence ?? 0) + 1;

            DB::table('receipt_counters')->updateOrInsert(
                ['branch_id' => $branchId, 'year' => $year],
                ['current_sequence' => $next, 'updated_at' => now()]
            );

            return $next;
        });

        return [sprintf('RCPT-%d-%05d', $year, $seq), $year, $seq];
    }

    /* ===================== Ledger: إنشاء قيد + احتساب الرصيد ===================== */

    protected function appendLedgerEntry(
        int $clientId,
        int $branchId,
        string $entryType,
        string $referenceNumber,
        string $referenceType,
        int $referenceId,
        int $debit,
        int $credit,
        string $description,
        string $entryDate
    ): ClientLedgerEntry {
        $debit = max(0, $this->toInt($debit));
        $credit = max(0, $this->toInt($credit));

        $last = ClientLedgerEntry::query()
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->orderByDesc('entry_date')
            ->orderByDesc('id')
            ->value('balance');

        $lastBalance = (int) ($last ?? 0);

        // الرصيد = السابق + debit - credit
        $balance = $lastBalance + $debit - $credit;

        return ClientLedgerEntry::query()->create([
            'client_id' => $clientId,
            'branch_id' => $branchId,
            'entry_type' => $entryType,
            'reference_number' => $referenceNumber,
            'reference_type' => $referenceType,
            'reference_id' => $referenceId,
            'debit' => $debit,
            'credit' => $credit,
            'balance' => $balance,
            'entry_date' => $entryDate,
            'description' => $description,
        ]);
    }

    /* ===================== حساب العميل: إعادة حساب صحيحة من allocations ===================== */

    protected function recomputeClientAccount(int $clientId, int $branchId): void
    {
        // 1) إجمالي الذمم من الفواتير
        $invQuery = Invoice::query()
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->whereNull('deleted_at');

        if (Schema::hasColumn('invoices', 'status')) {
            $invQuery->whereNotIn('status', ['canceled', 'cancelled']);
        }

        // نفترض grand_total / paid_amount / due_amount أعداد صحيحة
        $receivableTotal = (int) $invQuery->sum('grand_total');

        // 2) المدفوع الصافي من allocations (collection - payment) إن وجد
        $receivablePaid = 0;

        if ($this->hasAllocTable()) {
            $base = DB::table('receipt_invoice_allocations as a')
                ->join('client_receipts as r', 'r.id', '=', 'a.client_receipt_id')
                ->whereNull('r.deleted_at')
                ->where('r.branch_id', $branchId)
                ->where('r.client_id', $clientId)
                ->where('r.status', 'issued')
                ->where('a.invoice_type', Invoice::class);

            // إن كان جدول allocations يحتوي client_id/branch_id استعملهما كفلتر إضافي (تسريع)
            if ($this->allocHasClientId()) {
                $base->where('a.client_id', $clientId);
            }
            if ($this->allocHasBranchId()) {
                $base->where('a.branch_id', $branchId);
            }

            $collections = (int) (clone $base)
                ->where('r.receipt_type', 'collection')
                ->sum('a.paid_amount');

            $payments = (int) (clone $base)
                ->where('r.receipt_type', 'payment')
                ->sum('a.paid_amount');

            $receivablePaid = max(0, $collections - $payments);
        } else {
            // fallback (لو ما عندك allocations): من أعمدة invoices
            $receivablePaid = (int) $invQuery->sum('paid_amount');
        }

        $receivableRemaining = max(0, $receivableTotal - $receivablePaid);

        // 3) سلفة العميل (قبض غير موزّع): مجموع الوصولات (collection) - مجموع تخصيصات (collection)
        $clientAdvance = 0;

        if ($this->hasAllocTable()) {
            $collectionsPaid = (int) DB::table('client_receipts')
                ->whereNull('deleted_at')
                ->where('client_id', $clientId)
                ->where('branch_id', $branchId)
                ->where('status', 'issued')
                ->where('receipt_type', 'collection')
                ->sum('total_paid');

            $collectionsAlloc = (int) DB::table('receipt_invoice_allocations as a')
                ->join('client_receipts as r', 'r.id', '=', 'a.client_receipt_id')
                ->whereNull('r.deleted_at')
                ->where('r.branch_id', $branchId)
                ->where('r.client_id', $clientId)
                ->where('r.status', 'issued')
                ->where('r.receipt_type', 'collection')
                ->where('a.invoice_type', Invoice::class)
                ->sum('a.paid_amount');

            $clientAdvance = max(0, $collectionsPaid - $collectionsAlloc);
        }

        ClientAccount::query()->updateOrCreate(
            ['client_id' => $clientId, 'branch_id' => $branchId],
            [
                'receivable_total' => $receivableTotal,
                'receivable_paid' => $receivablePaid,
                'receivable_remaining' => $receivableRemaining,
                'client_advance' => $clientAdvance,
                'last_transaction_at' => now(),
            ]
        );
    }

    /* ===================== التوزيع على الفواتير (أقدم فأقدم) ===================== */

    public function allocateAcrossOldestInvoices(ClientReceipt $receipt, int $amount, ?int &$allocatedTotal = null): array
    {
        $amount = $this->toInt($amount);
        $allocatedTotal = 0;

        if ($amount <= 0) {
            return [];
        }

        if (! $this->hasAllocTable()) {
            // إذا ما عندك جدول allocations لا نكمل
            return [];
        }

        $dateCol = $this->pickCol('invoices', ['invoice_date', 'date', 'created_at'], 'created_at');

        $query = Invoice::query()
            ->where('client_id', $receipt->client_id)
            ->where('branch_id', $receipt->branch_id)
            ->whereNull('deleted_at')
            ->orderBy($dateCol)
            ->orderBy('id');

        if (Schema::hasColumn('invoices', 'status')) {
            $query->whereNotIn('status', ['canceled', 'cancelled']);
        }

        $invoices = $query->get([
            'id',
            'invoice_number',
            'invoice_date',
            'status',
            'payment_status',
            'subtotal',
            'discount_total',
            'shipping_total',
            'grand_total',
            'paid_amount',
            'due_amount',
        ]);

        $affected = [];

        foreach ($invoices as $inv) {
            if ($amount <= 0) {
                break;
            }

            $grand = (int) ($inv->grand_total ?? 0);
            if ($grand <= 0) {
                $subtotal = (int) ($inv->subtotal ?? 0);
                $discount = (int) ($inv->discount_total ?? 0);
                $shipping = (int) ($inv->shipping_total ?? 0);
                $grand = max(0, $subtotal - $discount + $shipping);
            }

            $paid = (int) ($inv->paid_amount ?? 0);

            $remain = (int) ($inv->due_amount ?? 0);
            if ($remain <= 0) {
                $remain = max(0, $grand - $paid);
            }

            if ($remain <= 0) {
                continue;
            }

            $alloc = min($amount, $remain);
            if ($alloc <= 0) {
                continue;
            }

            $row = [
                'client_receipt_id' => (int) $receipt->id,
                // ✅ مهم جداً: توحيد نوع الفاتورة
                'invoice_type' => Invoice::class,
                'invoice_id' => (int) $inv->id,
                'invoice_number' => $inv->invoice_number ?: (string) $inv->id,
                'invoice_total' => (int) $grand,
                'invoice_remaining_before' => (int) $remain,
                'paid_amount' => (int) $alloc,
                'invoice_remaining_after' => (int) max(0, $remain - $alloc),
                'created_at' => now(),
                'updated_at' => now(),
            ];

            if ($this->allocHasInvoiceDate()) {
                $row['invoice_date'] = $inv->invoice_date
                    ? Carbon::parse($inv->invoice_date)->format('Y-m-d')
                    : null;
            }

            if ($this->allocHasClientId()) {
                $row['client_id'] = (int) $receipt->client_id;
            }
            if ($this->allocHasBranchId()) {
                $row['branch_id'] = (int) $receipt->branch_id;
            }

            DB::table('receipt_invoice_allocations')->insert($row);

            // تحديث الفاتورة
            $newPaid = $paid + $alloc;
            $newDue = max(0, $grand - $newPaid);

            $paymentStatus = $newDue <= 0
                ? 'paid'
                : ($newPaid > 0 ? 'partial' : 'unpaid');

            $inv->forceFill([
                'paid_amount' => (int) $newPaid,
                'due_amount' => (int) $newDue,
                'payment_status' => Schema::hasColumn('invoices', 'payment_status')
                    ? $paymentStatus
                    : ($inv->payment_status ?? null),
            ])->saveQuietly();

            $affected[] = (int) $inv->id;

            $allocatedTotal += $alloc;
            $amount -= $alloc;
        }

        return array_values(array_unique($affected));
    }

    /* ===================== إنشاء وصل قبض ===================== */

    public function execute(array $data): ClientReceipt
    {
        $tz = config('app.timezone', 'Asia/Baghdad');

        $date = ! empty($data['receipt_date'])
            ? Carbon::parse($data['receipt_date'], $tz)->format('Y-m-d')
            : now($tz)->format('Y-m-d');

        $branchId = (int) ($data['branch_id'] ?? (function_exists('user_info') ? (user_info('branch_id') ?? 0) : 0));
        $clientId = (int) ($data['client_id'] ?? 0);

        $paid = $this->toInt($data['total_paid'] ?? $data['total_amount'] ?? $data['paid_amount'] ?? 0);

        $ptype = $data['payment_type'] ?? 'cash';
        $pdetails = $data['payment_details'] ?? ($data['notes'] ?? null);

        abort_unless($branchId > 0 && $clientId > 0, 422, 'يجب تحديد العميل والفرع.');
        abort_unless($paid > 0, 422, 'المبلغ المدفوع يجب أن يكون أكبر من صفر.');

        return DB::transaction(function () use ($date, $branchId, $clientId, $paid, $ptype, $pdetails) {

            // ✅ تنظيف بيانات قديمة: توحيد invoice_type للصفوف القديمة إن وجدت (مرة داخل نفس المعاملة)
            if ($this->hasAllocTable()) {
                DB::table('receipt_invoice_allocations')
                    ->where('invoice_type', 'invoice')
                    ->update(['invoice_type' => Invoice::class]);
            }

            [$number, $year, $sequence] = $this->nextCounterNumber($branchId);

            $issuedBy = (int) (function_exists('user_info') ? (user_info('id') ?? 0) : 0);

            /** @var ClientReceipt $receipt */
            $receipt = ClientReceipt::query()->create([
                'receipt_number' => $number,
                'year' => $year,
                'sequence' => $sequence,
                'receipt_date' => $date,
                'client_id' => $clientId,
                'branch_id' => $branchId,
                'receipt_type' => 'collection',

                // ✅ هذا "مبلغ الوصل"
                'total_amount' => (int) $paid,
                'total_paid' => (int) $paid,

                // ✅ سنجعله "الدين بعد الوصل" وليس unallocated
                'total_remaining' => 0,

                'payment_type' => $ptype,
                'payment_details' => $pdetails,
                'issued_by' => $issuedBy,
                'status' => 'issued',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // إيداع الخزنة
            if (function_exists('current_vault') && ($vault = current_vault()) && $paid > 0) {
                $vault->deposit(
                    source: $receipt,
                    amount: (int) $paid,
                    description: 'قبض من العميل — وصل '.$receipt->receipt_number,
                    userId: $issuedBy,
                );
            }

            // قيد دفتر الأستاذ (collection تقلل ذمم العميل => credit)
            $this->appendLedgerEntry(
                clientId: $clientId,
                branchId: $branchId,
                entryType: 'receipt',
                referenceNumber: $receipt->receipt_number,
                referenceType: ClientReceipt::class,
                referenceId: (int) $receipt->id,
                debit: 0,
                credit: (int) $paid,
                description: 'قبض من العميل — وصل '.$receipt->receipt_number,
                entryDate: $receipt->receipt_date instanceof \DateTimeInterface
                    ? $receipt->receipt_date->format('Y-m-d')
                    : (string) $receipt->receipt_date,
            );

            // توزيع على الفواتير
            $allocatedTotal = 0;
            $this->allocateAcrossOldestInvoices($receipt, (int) $paid, $allocatedTotal);

            // إعادة حساب حساب العميل (من allocations)
            $this->recomputeClientAccount($clientId, $branchId);

            // ✅ اجعل total_remaining = الدين بعد الوصل (الذي يظهر في السند)
            $acc = ClientAccount::query()
                ->where('client_id', $clientId)
                ->where('branch_id', $branchId)
                ->first();

            $dueAfter = (int) ($acc->receivable_remaining ?? 0);

            $receipt->forceFill([
                'total_remaining' => $dueAfter,
                'updated_at' => now(),
            ])->saveQuietly();

            return $receipt->fresh();
        });
    }
}
