<?php

namespace App\Services\Accounting;

use App\Enums\ClientReceiptType;
use App\Enums\ReceiptStatus;
use App\Models\ClientReceipt;
use App\Models\Invoice;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class ClientAccountingService
{
    /* ===================== أدوات عامة ===================== */

    protected function normalizeDigits(string $s): string
    {
        return strtr($s, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);
    }

    public function toInt(mixed $v): int
    {
        if ($v === null) {
            return 0;
        }

        $s = $this->normalizeDigits((string) $v);
        $s = preg_replace('/[^\d\-]/', '', $s) ?? '0';

        return (int) $s;
    }

    protected function toUInt(mixed $v): int
    {
        return max(0, $this->toInt($v));
    }

    /* ===================== فواتير البيع (Invoice) ===================== */

    /**
     * المجموع الكلي النهائي للفاتورة
     * - المرجع: invoices.grand_total
     * - احتياط: subtotal - discount_total + shipping_total
     */
    public function invoiceGrandTotal(Invoice $invoice): int
    {
        $grand = $this->toUInt($invoice->grand_total ?? 0);
        if ($grand > 0) {
            return $grand;
        }

        $subtotal = $this->toUInt($invoice->subtotal ?? 0);
        $discount = $this->toUInt($invoice->discount_total ?? 0);
        $shipping = $this->toUInt($invoice->shipping_total ?? 0);

        return max(0, $subtotal - $discount + $shipping);
    }

    /**
     * مجموع المدفوع "الصافي" على الفاتورة من خلال الوصولات:
     * - collection تزيد المدفوع
     * - payment (استرجاع/دفع للعميل) تنقص المدفوع
     *
     * ملاحظة مهمة: جدول receipt_invoice_allocations.paid_amount UNSIGNED
     * لذلك لا نستخدم قيم سالبة أبداً.
     */
    public function invoicePaidSum(Invoice $invoice): int
    {
        [$col, $pay] = $this->invoicePaidSumsDetailed($invoice->id, (int) $invoice->branch_id);

        // المدفوع الصافي لا ينزل تحت الصفر
        return max(0, $col - $pay);
    }

    protected function invoicePaidSumsDetailed(int $invoiceId, int $branchId): array
    {
        $base = DB::table('receipt_invoice_allocations as a')
            ->join('client_receipts as r', 'r.id', '=', 'a.client_receipt_id')
            ->where('a.invoice_type', Invoice::class)
            ->where('a.invoice_id', $invoiceId)
            ->where('r.branch_id', $branchId)
            ->where('r.status', ReceiptStatus::ISSUED->value)
            ->whereNull('r.deleted_at');

        $collections = (int) (clone $base)
            ->where('r.receipt_type', ClientReceiptType::COLLECTION->value)
            ->sum('a.paid_amount');

        $payments = (int) (clone $base)
            ->where('r.receipt_type', ClientReceiptType::PAYMENT->value)
            ->sum('a.paid_amount');

        return [$collections, $payments];
    }

    /**
     * إعادة حساب (paid_amount + due_amount + payment_status) في الفاتورة
     * بالاعتماد على allocations (collection - payment)
     */
    public function recomputeInvoicePaidAndRemaining(Invoice $invoice): Invoice
    {
        $grand = $this->invoiceGrandTotal($invoice);
        $paid = $this->invoicePaidSum($invoice);
        $remain = max(0, $grand - $paid);

        $paymentStatus =
            $remain <= 0 ? 'paid' : ($paid > 0 ? 'partial' : 'unpaid');

        $invoice->forceFill([
            'grand_total' => $grand,
            'paid_amount' => $paid,
            'due_amount' => $remain,
            'payment_status' => $paymentStatus,
        ])->saveQuietly();

        return $invoice->refresh();
    }

    /* ===================== أرقام الوصول ===================== */

    protected function nextReceiptNumber(int $branchId): array
    {
        $tz = 'Asia/Baghdad';
        $year = now($tz)->year;

        $sequence = DB::transaction(function () use ($branchId, $year) {
            $row = DB::table('receipt_counters')
                ->where('branch_id', $branchId)
                ->where('year', $year)
                ->lockForUpdate()
                ->first();

            $seq = (int) ($row->current_sequence ?? 0) + 1;

            DB::table('receipt_counters')->updateOrInsert(
                ['branch_id' => $branchId, 'year' => $year],
                ['current_sequence' => $seq, 'updated_at' => now()]
            );

            return $seq;
        });

        return [sprintf('RCPT-%d-%05d', $year, $sequence), $year, $sequence];
    }

    /* ===================== قبض/تخصيص على Invoice ===================== */

    public function collectPaymentForInvoice(
        Invoice $invoice,
        int $amount,
        ClientReceiptType $receiptType = ClientReceiptType::COLLECTION,
        string $paymentType = 'cash',
        ?string $paymentDetails = null,
    ): ClientReceipt {
        $amount = $this->toUInt($amount);

        if ($amount <= 0) {
            throw ValidationException::withMessages([
                'amount' => ['قيمة الدفعة يجب أن تكون أكبر من صفر.'],
            ]);
        }

        if (! $invoice->client_id || ! $invoice->branch_id) {
            throw ValidationException::withMessages([
                'invoice' => ['الفاتورة يجب أن تكون مرتبطة بعميل وفرع.'],
            ]);
        }

        $tz = 'Asia/Baghdad';
        $branchId = (int) $invoice->branch_id;
        $clientId = (int) $invoice->client_id;

        return DB::transaction(function () use ($invoice, $amount, $receiptType, $paymentType, $paymentDetails, $tz, $branchId, $clientId) {
            $grandTotal = $this->invoiceGrandTotal($invoice);
            $paidBefore = $this->invoicePaidSum($invoice);
            $remainBefore = max(0, $grandTotal - $paidBefore);

            // في القبض: لا نتجاوز المتبقي
            // في الدفع (payment): لا نتجاوز المدفوع الصافي (كأنه استرجاع)
            $allocAmount = $receiptType === ClientReceiptType::COLLECTION
                ? min($amount, $remainBefore)
                : min($amount, $paidBefore);

            [$number, $year, $sequence] = $this->nextReceiptNumber($branchId);

            /** @var ClientReceipt $receipt */
            $receipt = ClientReceipt::query()->create([
                'receipt_number' => $number,
                'year' => $year,
                'sequence' => $sequence,
                'receipt_date' => now($tz)->toDateString(),
                'client_id' => $clientId,
                'branch_id' => $branchId,

                // enums في DB هي strings
                'receipt_type' => $receiptType->value,
                'status' => ReceiptStatus::ISSUED->value,

                // معلوماتياً: إجمالي الفاتورة
                'total_amount' => $grandTotal,
                'total_paid' => $amount,
                'total_remaining' => 0,

                'payment_type' => $paymentType,
                'payment_details' => $paymentDetails
                    ?? (($receiptType === ClientReceiptType::COLLECTION ? 'دفعة' : 'دفع/استرجاع')
                        .' على فاتورة بيع #'.($invoice->invoice_number ?? $invoice->id)),

                'issued_by' => (function_exists('user_info') ? (user_info('id') ?? null) : null),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // تخصيص على الفاتورة (paid_amount UNSIGNED دائماً)
            DB::table('receipt_invoice_allocations')->insert([
                'client_receipt_id' => $receipt->id,
                'invoice_type' => Invoice::class,
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number ?: (string) $invoice->id,
                'invoice_date' => $invoice->invoice_date?->toDateString(),
                'invoice_total' => $grandTotal,
                'invoice_remaining_before' => $remainBefore,
                'paid_amount' => $allocAmount,
                'invoice_remaining_after' => $receiptType === ClientReceiptType::COLLECTION
                    ? max(0, $remainBefore - $allocAmount)
                    : min($grandTotal, $remainBefore + $allocAmount),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // إعادة حساب مبالغ الفاتورة (الصافي = collection - payment)
            $invoiceFresh = $this->recomputeInvoicePaidAndRemaining($invoice->fresh());

            // أقساط:
            if ($receiptType === ClientReceiptType::COLLECTION) {
                $this->applyAllocToInstallments($invoiceFresh, $allocAmount);
            } else {
                $this->unapplyRefundFromInstallments($invoiceFresh, $allocAmount);
                $this->recomputeInstallmentPlanTotals($invoiceFresh);
            }

            // تحديث المتبقي في الوصل = المتبقي الفعلي للفاتورة
            $receipt->update(['total_remaining' => (int) ($invoiceFresh->due_amount ?? 0)]);

            // حركة الخزنة
            $vault = function_exists('current_vault') ? current_vault() : null;
            if ($vault && $amount > 0) {
                $userId = (int) (function_exists('user_info') ? (user_info('id') ?? 0) : 0);

                if ($receiptType === ClientReceiptType::COLLECTION) {
                    $vault->deposit(
                        source: $receipt,
                        amount: $amount,
                        description: 'قبض على فاتورة #'.($invoiceFresh->invoice_number ?? $invoiceFresh->id)
                            .' (الإجمالي: '.number_format($grandTotal).')',
                        userId: $userId,
                    );
                } else {
                    $vault->withdraw(
                        source: $receipt,
                        amount: $amount,
                        description: 'دفع/استرجاع على فاتورة #'.($invoiceFresh->invoice_number ?? $invoiceFresh->id)
                            .' (الإجمالي: '.number_format($grandTotal).')',
                        userId: $userId,
                    );
                }
            }

            // دفتر الأستاذ (الرصيد = ذمم العميل)
            // collection تقلل الرصيد => credit
            // payment/Refund تزيد الرصيد => debit
            $entryType = $receiptType === ClientReceiptType::COLLECTION ? 'receipt_collection' : 'receipt_payment';

            $debit = $receiptType === ClientReceiptType::COLLECTION ? 0 : $amount;
            $credit = $receiptType === ClientReceiptType::COLLECTION ? $amount : 0;

            $this->appendLedgerEntry(
                clientId: $clientId,
                branchId: $branchId,
                entryType: $entryType,
                referenceNumber: $receipt->receipt_number,
                debit: $debit,
                credit: $credit,
                description: 'وصل رقم '.$receipt->receipt_number.' على الفاتورة #'.($invoiceFresh->invoice_number ?? $invoiceFresh->id),
                entryDate: $receipt->receipt_date,
                referenceType: ClientReceipt::class,
                referenceId: $receipt->id,
            );

            // تحديث حساب العميل
            $this->recomputeClientAccount($clientId, $branchId);

            return $receipt->fresh();
        });
    }

    /**
     * استرجاع/دفع للعميل على فاتورة:
     * - ينشئ receipt_type = payment
     * - ويسجل allocation بقيمة موجبة (لأن paid_amount UNSIGNED)
     * - ويُطرح من المدفوع الصافي عبر invoicePaidSum()
     */
    public function refundForInvoice(
        Invoice $invoice,
        int $amount,
        string $paymentType = 'cash',
        ?string $paymentDetails = null,
    ): ClientReceipt {
        // نعاملها كـ collectPaymentForInvoice ولكن receiptType = PAYMENT
        return $this->collectPaymentForInvoice(
            invoice: $invoice,
            amount: $amount,
            receiptType: ClientReceiptType::PAYMENT,
            paymentType: $paymentType,
            paymentDetails: $paymentDetails
                ?? ('استرجاع/دفع على فاتورة بيع #'.($invoice->invoice_number ?? $invoice->id)),
        );
    }

    /* ===================== الأقساط ===================== */

    protected function activePlanIdForInvoice(Invoice $invoice): ?int
    {
        return DB::table('client_installment_plans')
            ->where('invoice_type', Invoice::class)
            ->where('invoice_id', $invoice->id)
            ->where('status', 'active')
            ->value('id');
    }

    public function applyAllocToInstallments(Invoice $invoice, int $allocated): void
    {
        $allocated = $this->toUInt($allocated);
        if ($allocated <= 0) {
            return;
        }

        $planId = $this->activePlanIdForInvoice($invoice);
        if (! $planId) {
            return;
        }

        $rows = DB::table('installment_schedules')
            ->where('installment_plan_id', $planId)
            ->where('status', 'pending')
            ->orderBy('sequence')
            ->lockForUpdate()
            ->get(['id', 'amount']);

        foreach ($rows as $row) {
            if ($allocated <= 0) {
                break;
            }

            $amt = (int) $row->amount;

            if ($allocated >= $amt) {
                DB::table('installment_schedules')
                    ->where('id', $row->id)
                    ->update([
                        'status' => 'paid',
                        'updated_at' => now(),
                    ]);

                $allocated -= $amt;
            } else {
                break;
            }
        }

        $this->recomputeInstallmentPlanTotals($invoice);
    }

    /**
     * عكس (Refund) على الأقساط: نفك آخر أقساط مدفوعة (من الأخير للأول)
     */
    public function unapplyRefundFromInstallments(Invoice $invoice, int $refunded): void
    {
        $refunded = $this->toUInt($refunded);
        if ($refunded <= 0) {
            return;
        }

        $planId = $this->activePlanIdForInvoice($invoice);
        if (! $planId) {
            return;
        }

        $rows = DB::table('installment_schedules')
            ->where('installment_plan_id', $planId)
            ->where('status', 'paid')
            ->orderByDesc('sequence')
            ->lockForUpdate()
            ->get(['id', 'amount']);

        foreach ($rows as $row) {
            if ($refunded <= 0) {
                break;
            }

            DB::table('installment_schedules')
                ->where('id', $row->id)
                ->update([
                    'status' => 'pending',
                    'updated_at' => now(),
                ]);

            $refunded -= (int) $row->amount;
        }
    }

    /**
     * تحديث ملخص الخطة بحسب جدول الأقساط + تحديث حالة الخطة
     * (حسب جداولك: لا يوجد total_paid/remaining_amount، فنحسبها من schedules)
     */
    public function recomputeInstallmentPlanTotals(Invoice $invoice): void
    {
        $plan = DB::table('client_installment_plans')
            ->where('invoice_type', Invoice::class)
            ->where('invoice_id', $invoice->id)
            ->whereIn('status', ['active', 'completed'])
            ->first(['id', 'status', 'down_payment']);

        if (! $plan) {
            return;
        }

        $planId = (int) $plan->id;
        $grandTotal = $this->invoiceGrandTotal($invoice);

        $total = (int) DB::table('installment_schedules')
            ->where('installment_plan_id', $planId)
            ->sum('amount');

        $paidSum = (int) DB::table('installment_schedules')
            ->where('installment_plan_id', $planId)
            ->where('status', 'paid')
            ->sum('amount');

        $remain = max(0, $total - $paidSum);

        $count = (int) DB::table('installment_schedules')
            ->where('installment_plan_id', $planId)
            ->count();

        $installmentAmount = $count > 0 ? (int) intdiv($total, $count) : 0;

        $startDate = DB::table('installment_schedules')
            ->where('installment_plan_id', $planId)
            ->min('due_date');

        $endDate = DB::table('installment_schedules')
            ->where('installment_plan_id', $planId)
            ->max('due_date');

        $newStatus = $remain <= 0 ? 'completed' : 'active';

        DB::table('client_installment_plans')
            ->where('id', $planId)
            ->update([
                'total_amount' => $grandTotal,
                'financed_amount' => $remain,              // المتبقي الفعلي على الخطة
                'installment_count' => max(1, $count),
                'installment_amount' => $installmentAmount,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'status' => $newStatus,
                'updated_at' => now(),
            ]);
    }

    /**
     * إنشاء خطة أقساط (متوافقة مع مهاجرتك)
     */
    public function createInstallmentPlanForInvoice(
        Invoice $invoice,
        int $financedAmount,
        ?int $installmentsCount = null,
        ?string $startDate = null,
        ?string $notes = null,
        int $downPayment = 0,
        ?array $scheduleRows = null,
    ): void {
        $financedAmount = $this->toUInt($financedAmount);
        $downPayment = $this->toUInt($downPayment);

        if ($financedAmount <= 0) {
            return;
        }

        DB::transaction(function () use ($invoice, $financedAmount, $installmentsCount, $startDate, $notes, $downPayment, $scheduleRows) {
            $rows = [];

            if (is_array($scheduleRows) && count($scheduleRows)) {
                $seq = 1;
                foreach ($scheduleRows as $r) {
                    $amt = $this->toUInt($r['amount'] ?? 0);
                    if ($amt <= 0) {
                        continue;
                    }

                    $rows[] = [
                        'sequence' => isset($r['sequence']) ? (int) $r['sequence'] : $seq,
                        'amount' => $amt,
                        'due_date' => $r['due_date'] ?? now('Asia/Baghdad')->toDateString(),
                        'status' => $r['status'] ?? 'pending',
                    ];
                    $seq++;
                }
            } else {
                $count = max(1, (int) ($installmentsCount ?? 1));
                $base = intdiv($financedAmount, $count);
                $rem = $financedAmount - ($base * $count);

                $start = $startDate ? now('Asia/Baghdad')->parse($startDate) : now('Asia/Baghdad')->addMonth();

                for ($i = 0; $i < $count; $i++) {
                    $rows[] = [
                        'sequence' => $i + 1,
                        'amount' => $base + ($i < $rem ? 1 : 0),
                        'due_date' => $start->copy()->addMonths($i)->toDateString(),
                        'status' => 'pending',
                    ];
                }
            }

            // ضبط آخر قسط عند وجود فرق
            $sum = array_sum(array_column($rows, 'amount'));
            if ($sum !== $financedAmount && count($rows) > 0) {
                $rows[count($rows) - 1]['amount'] += ($financedAmount - $sum);
            }

            $grandTotal = $this->invoiceGrandTotal($invoice);

            $planId = DB::table('client_installment_plans')->insertGetId([
                'client_id' => (int) $invoice->client_id,
                'branch_id' => (int) $invoice->branch_id,

                'invoice_type' => Invoice::class,
                'invoice_id' => (int) $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'invoice_date' => $invoice->invoice_date?->toDateString(),

                'total_amount' => $grandTotal,
                'down_payment' => $downPayment,
                'financed_amount' => $financedAmount,

                'installment_count' => max(1, count($rows)),
                'installment_amount' => count($rows) > 0 ? (int) intdiv($financedAmount, max(1, count($rows))) : 0,

                'start_date' => $rows[0]['due_date'] ?? null,
                'end_date' => $rows[count($rows) - 1]['due_date'] ?? null,

                'status' => 'active',
                'user_id' => (function_exists('user_info') ? (user_info('id') ?? null) : null),
                'notes' => $notes,

                'created_at' => now(),
                'updated_at' => now(),
            ]);

            foreach ($rows as $r) {
                DB::table('installment_schedules')->insert([
                    'installment_plan_id' => $planId,
                    'sequence' => (int) $r['sequence'],
                    'amount' => (int) $r['amount'],
                    'due_date' => $r['due_date'],
                    'status' => $r['status'],
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            $this->recomputeInstallmentPlanTotals($invoice->fresh());
        });
    }

    /* ===================== دفتر الأستاذ ===================== */

    public function appendLedgerEntry(
        int $clientId,
        int $branchId,
        string $entryType,
        ?string $referenceNumber,
        int $debit,
        int $credit,
        ?string $description,
        string $entryDate,
        ?string $referenceType = null,
        ?int $referenceId = null,
    ): void {
        $debit = $this->toUInt($debit);
        $credit = $this->toUInt($credit);

        $lastBalance = (int) DB::table('client_ledger_entries')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->orderByDesc('id')
            ->value('balance');

        // balance UNSIGNED => لا نسمح بالسالب
        $newBalance = max(0, $lastBalance + ($debit - $credit));

        DB::table('client_ledger_entries')->insert([
            'client_id' => $clientId,
            'branch_id' => $branchId,
            'entry_type' => $entryType,
            'reference_number' => $referenceNumber,
            'reference_type' => $referenceType,
            'reference_id' => $referenceId,
            'debit' => $debit,
            'credit' => $credit,
            'balance' => $newBalance,
            'entry_date' => $entryDate,
            'description' => $description,
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }

    /**
     * إعادة بناء أرصدة دفتر الأستاذ (ضروري بعد delete/update داخل الوسط)
     */
    public function rebuildClientLedgerBalances(int $clientId, int $branchId): void
    {
        DB::transaction(function () use ($clientId, $branchId) {
            $rows = DB::table('client_ledger_entries')
                ->where('client_id', $clientId)
                ->where('branch_id', $branchId)
                ->orderBy('id')
                ->lockForUpdate()
                ->get(['id', 'debit', 'credit']);

            $balance = 0;

            foreach ($rows as $r) {
                $balance = max(0, $balance + ((int) $r->debit - (int) $r->credit));

                DB::table('client_ledger_entries')
                    ->where('id', $r->id)
                    ->update([
                        'balance' => $balance,
                        'updated_at' => now(),
                    ]);
            }
        });
    }

    /* ===================== حساب العميل ===================== */

    public function recomputeClientAccount(int $clientId, int $branchId): void
    {
        // إجمالي الذمم من الفواتير (نستثني المحذوفة والملغاة إن أمكن)
        $receivableTotal = (int) DB::table('invoices')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->whereNull('deleted_at')
            ->whereNotIn('status', ['canceled', 'cancelled'])
            ->sum(DB::raw('
                CASE
                    WHEN COALESCE(grand_total, 0) > 0 THEN grand_total
                    ELSE GREATEST(0, COALESCE(subtotal,0) - COALESCE(discount_total,0) + COALESCE(shipping_total,0))
                END
            '));

        // المدفوعات الصافية = allocations(collection) - allocations(payment)
        $allocBase = DB::table('receipt_invoice_allocations as a')
            ->join('client_receipts as r', 'r.id', '=', 'a.client_receipt_id')
            ->join('invoices as i', function ($j) {
                $j->on('i.id', '=', 'a.invoice_id')
                    ->where('a.invoice_type', Invoice::class);
            })
            ->where('i.client_id', $clientId)
            ->where('i.branch_id', $branchId)
            ->whereNull('i.deleted_at')
            ->where('r.status', ReceiptStatus::ISSUED->value)
            ->whereNull('r.deleted_at');

        $allocCollections = (int) (clone $allocBase)
            ->where('r.receipt_type', ClientReceiptType::COLLECTION->value)
            ->sum('a.paid_amount');

        $allocPayments = (int) (clone $allocBase)
            ->where('r.receipt_type', ClientReceiptType::PAYMENT->value)
            ->sum('a.paid_amount');

        $receivablePaid = max(0, $allocCollections - $allocPayments);
        $receivableRemaining = max(0, $receivableTotal - $receivablePaid);

        // لا يوجد payable حالياً
        $payableTotal = 0;
        $payablePaid = 0;
        $payableRemaining = 0;

        // سلفة العميل: مجموع قبض (total_paid) - ما تم تخصيصه من قبض
        $collectionsPaid = (int) DB::table('client_receipts')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->where('status', ReceiptStatus::ISSUED->value)
            ->where('receipt_type', ClientReceiptType::COLLECTION->value)
            ->whereNull('deleted_at')
            ->sum('total_paid');

        $collectionsAlloc = (int) DB::table('receipt_invoice_allocations as a')
            ->join('client_receipts as r', 'r.id', '=', 'a.client_receipt_id')
            ->where('r.client_id', $clientId)
            ->where('r.branch_id', $branchId)
            ->where('r.status', ReceiptStatus::ISSUED->value)
            ->where('r.receipt_type', ClientReceiptType::COLLECTION->value)
            ->whereNull('r.deleted_at')
            ->sum('a.paid_amount');

        $clientAdvance = max(0, $collectionsPaid - $collectionsAlloc);

        // سلفة الشركة: مجموع دفع (total_paid) - ما تم تخصيصه من دفع
        $paymentsPaid = (int) DB::table('client_receipts')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->where('status', ReceiptStatus::ISSUED->value)
            ->where('receipt_type', ClientReceiptType::PAYMENT->value)
            ->whereNull('deleted_at')
            ->sum('total_paid');

        $paymentsAlloc = (int) DB::table('receipt_invoice_allocations as a')
            ->join('client_receipts as r', 'r.id', '=', 'a.client_receipt_id')
            ->where('r.client_id', $clientId)
            ->where('r.branch_id', $branchId)
            ->where('r.status', ReceiptStatus::ISSUED->value)
            ->where('r.receipt_type', ClientReceiptType::PAYMENT->value)
            ->whereNull('r.deleted_at')
            ->sum('a.paid_amount');

        $companyAdvance = max(0, $paymentsPaid - $paymentsAlloc);

        // آخر حركة
        $lastInvoiceDate = DB::table('invoices')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->whereNull('deleted_at')
            ->max('invoice_date');

        $lastReceiptDate = DB::table('client_receipts')
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->where('status', ReceiptStatus::ISSUED->value)
            ->whereNull('deleted_at')
            ->max('receipt_date');

        $lastTx = max($lastInvoiceDate ?? '1970-01-01', $lastReceiptDate ?? '1970-01-01');
        $lastTs = $lastTx ? now()->parse($lastTx) : null;

        DB::table('client_accounts')->updateOrInsert(
            ['client_id' => $clientId, 'branch_id' => $branchId],
            [
                'receivable_total' => $receivableTotal,
                'receivable_paid' => $receivablePaid,
                'receivable_remaining' => $receivableRemaining,

                'payable_total' => $payableTotal,
                'payable_paid' => $payablePaid,
                'payable_remaining' => $payableRemaining,

                'client_advance' => $clientAdvance,
                'company_advance' => $companyAdvance,

                'last_transaction_at' => $lastTs,
                'updated_at' => now(),
                'created_at' => now(),
            ]
        );
    }

    /* ===================== الأقساط المتأخرة ===================== */

    public function markOverdueInstallments(?string $date = null): int
    {
        $today = $date
            ? now('Asia/Baghdad')->parse($date)->toDateString()
            : now('Asia/Baghdad')->toDateString();

        return DB::table('installment_schedules')
            ->where('status', 'pending')
            ->whereDate('due_date', '<', $today)
            ->update([
                'status' => 'overdue',
                'updated_at' => now(),
            ]);
    }

    /* ===================== حذف فاتورة Invoice ===================== */

    /**
     * تنظيف محاسبي عند حذف الفاتورة:
     * - نحذف تخصيصات الوصول على الفاتورة (بدون لمس الخزنة)
     * - نلغي خطة الأقساط (إن وجدت)
     * - نحذف قيود الأستاذ المتعلقة بالفاتورة (إن كنت تربطها كـ reference_type = Invoice::class)
     * - نعيد حساب حساب العميل
     */
    public function deleteInvoiceAccountingCleanup(Invoice $invoice): void
    {
        DB::transaction(function () use ($invoice) {
            // 1) إلغاء خطط الأقساط
            DB::table('installment_schedules')
                ->whereIn('installment_plan_id', function ($q) use ($invoice) {
                    $q->select('id')
                        ->from('client_installment_plans')
                        ->where('invoice_type', Invoice::class)
                        ->where('invoice_id', $invoice->id);
                })
                ->update([
                    'status' => 'canceled',
                    'updated_at' => now(),
                ]);

            DB::table('client_installment_plans')
                ->where('invoice_type', Invoice::class)
                ->where('invoice_id', $invoice->id)
                ->update([
                    'status' => 'canceled',
                    'updated_at' => now(),
                ]);

            // 2) حذف التخصيصات على الفاتورة
            DB::table('receipt_invoice_allocations')
                ->where('invoice_type', Invoice::class)
                ->where('invoice_id', $invoice->id)
                ->delete();

            // 3) حذف قيود الأستاذ الخاصة بهذه الفاتورة (إن وجدت)
            if ($invoice->client_id && $invoice->branch_id) {
                DB::table('client_ledger_entries')
                    ->where('client_id', (int) $invoice->client_id)
                    ->where('branch_id', (int) $invoice->branch_id)
                    ->where('reference_type', Invoice::class)
                    ->where('reference_id', $invoice->id)
                    ->delete();

                // مهم لإصلاح balances بعد الحذف
                $this->rebuildClientLedgerBalances((int) $invoice->client_id, (int) $invoice->branch_id);

                $this->recomputeClientAccount((int) $invoice->client_id, (int) $invoice->branch_id);
            }
        });
    }
}
