<?php

namespace App\Providers\Filament;

use Andreia\FilamentUiSwitcher\FilamentUiSwitcherPlugin;
use App\Filament\Pages\AccountSettings;
use App\Filament\Pages\Auth\Login;
use App\Filament\Pages\Dashboard;
use App\Models\Branch;
use App\Models\BranchSetting;
use Filafly\Icons\Phosphor\PhosphorIcons;
use Filament\Actions\Action;
use Filament\Facades\Filament;
use Filament\Http\Middleware\Authenticate;
use Filament\Http\Middleware\AuthenticateSession;
use Filament\Http\Middleware\DisableBladeIconComponents;
use Filament\Http\Middleware\DispatchServingFilamentEvent;
use Filament\Panel;
use Filament\PanelProvider;
use Filament\Support\Assets\Js;
use Filament\Support\Colors\Color;
use Filament\Support\Enums\Width;
use Filament\View\PanelsRenderHook;
use Illuminate\Cookie\Middleware\AddQueuedCookiesToResponse;
use Illuminate\Cookie\Middleware\EncryptCookies;
use Illuminate\Foundation\Http\Middleware\VerifyCsrfToken;
use Illuminate\Routing\Middleware\SubstituteBindings;
use Illuminate\Session\Middleware\StartSession;
use Illuminate\Support\Facades\Vite;
use Illuminate\View\Middleware\ShareErrorsFromSession;

class AdminPanelProvider extends PanelProvider
{
    /* ===================== أدوات داخلية ===================== */

    /** يرجع إعدادات فرع "السياق الحالي" (تيننت إن وُجد، وإلا فرع المستخدم) مع الميديا محمّلة */
    protected function getCurrentBranchSettings(): ?BranchSetting
    {
        $tenant = Filament::getTenant();

        if ($tenant instanceof Branch) {
            return $tenant->settings()?->with('media')->first();
        }

        $user = user_info();
        if (! $user) {
            return null;
        }

        $user->loadMissing('branch.settings.media');

        return $user->branch?->settings;
    }

    /** يبني رابط الميديا مع cache-busting أو يرجع null */
    protected function mediaUrlWithVersion(?BranchSetting $settings, string $collection): ?string
    {
        if (! $settings) {
            return null;
        }

        $media = $settings->getFirstMedia($collection);
        if (! $media) {
            return null;
        }

        $url = $media->getUrl();

        return $url ? ($url.'?v='.($media->updated_at?->timestamp ?? time())) : null;
    }

    public function panel(Panel $panel): Panel
    {
        /* الشعاران: نمرر كولشرات تُحتسب وقت العرض بعد ضبط التيننت */
        $brandLogo = function () {
            $settings = $this->getCurrentBranchSettings();
            $light = $this->mediaUrlWithVersion($settings, 'logo_light');
            $dark = $this->mediaUrlWithVersion($settings, 'logo_dark');

            return $light ?? $dark ?? asset('images/logo/light.png');
        };

        $brandLogoDark = function () {
            $settings = $this->getCurrentBranchSettings();
            $dark = $this->mediaUrlWithVersion($settings, 'logo_dark');
            $light = $this->mediaUrlWithVersion($settings, 'logo_light');

            return $dark ?? $light ?? asset('images/logo/dark.png');
        };

        return $panel
            ->default()
            ->id('admin')
            ->path('/admin')
            ->spa(true)
            ->login(Login::class)
            ->passwordReset()
            ->colors(['primary' => Color::Rose])
            // ->profile(isSimple: false)
            // الثيم والخط
            ->viteTheme('resources/css/filament/admin/theme.css')
            ->assets([
                Js::make('flatpickr-ar-am-pm', Vite::asset('resources/js/filament/flatpickr.js')),
            ])
            ->brandLogo($brandLogo)
            ->darkModeBrandLogo($brandLogoDark)
            ->brandName('ايوان')
            // إعدادات الواجهة
            ->sidebarFullyCollapsibleOnDesktop()
            ->maxContentWidth(Width::Full)
            // إشعارات قاعدة البيانات
            ->databaseNotifications()
            ->databaseNotificationsPolling('10s')
            ->discoverResources(in: app_path('Filament/Resources'), for: 'App\Filament\Resources')
            ->discoverPages(in: app_path('Filament/Pages'), for: 'App\Filament\Pages')
            ->pages([
                Dashboard::class,
            ])
            ->discoverWidgets(in: app_path('Filament/Widgets'), for: 'App\Filament\Widgets')
            ->widgets([])

            // قوائم المستخدم
            ->renderHook('panels::user-menu.profile.before', fn () => view('filament.admin.user-menu-header'))
            ->userMenuItems([
                'profile' => Action::make('profile')
                    ->label('الملف الشخصي')
                    ->icon('heroicon-m-user')
                    ->url(fn () => AccountSettings::getUrl()),
                Action::make('docs')
                    ->label('الدعم')->icon('heroicon-m-book-open')
                    // ->url(route('filament.admin.pages.docs'), shouldOpenInNewTab: true)
                    ->color('gray'),
            ])
            ->middleware([
                EncryptCookies::class,
                AddQueuedCookiesToResponse::class,
                StartSession::class,
                AuthenticateSession::class,
                ShareErrorsFromSession::class,
                VerifyCsrfToken::class,
                SubstituteBindings::class,
                DisableBladeIconComponents::class,
                DispatchServingFilamentEvent::class,
            ])
            ->authMiddleware([
                Authenticate::class,
            ])
            ->plugin(
                FilamentUiSwitcherPlugin::make()
                    ->iconRenderHook(PanelsRenderHook::USER_MENU_BEFORE),
                PhosphorIcons::make(),

            );
    }
}
