<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Carbon\Carbon;

class WorkSchedule extends Model
{
    protected $fillable = [
        'branch_id',
        'employee_id',
        'name',
        'weekday',
        'expected_start',
        'expected_end',
        'expected_minutes',
        'grace_period_minutes',
        'overtime_rate_percentage',
        'is_active',
        'effective_from',
        'effective_to',
    ];

    protected $casts = [
        'expected_start' => 'datetime:H:i:s',
        'expected_end' => 'datetime:H:i:s',
        'effective_from' => 'date',
        'effective_to' => 'date',
        'is_active' => 'boolean',
        'weekday' => 'integer',
        'expected_minutes' => 'integer',
        'grace_period_minutes' => 'integer',
        'overtime_rate_percentage' => 'integer',
    ];

    // ==================== العلاقات ====================

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class);
    }

    public function attendances(): HasMany
    {
        return $this->hasMany(Attendance::class);
    }

    public function overtimeRecords(): HasMany
    {
        return $this->hasMany(OvertimeRecord::class);
    }

    // ==================== Accessors ====================

    /**
     * الحصول على عدد ساعات العمل المتوقعة
     */
    public function getExpectedHoursAttribute(): float
    {
        return round($this->expected_minutes / 60, 2);
    }

    /**
     * الحصول على اسم اليوم بالعربي
     */
    public function getWeekdayNameAttribute(): string
    {
        $days = ['الأحد', 'الإثنين', 'الثلاثاء', 'الأربعاء', 'الخميس', 'الجمعة', 'السبت'];
        return $days[$this->weekday] ?? '';
    }

    /**
     * الحصول على وقت البداية المتوقع منسق
     */
    public function getExpectedStartFormattedAttribute(): string
    {
        return $this->expected_start->format('h:i A');
    }

    /**
     * الحصول على وقت النهاية المتوقع منسق
     */
    public function getExpectedEndFormattedAttribute(): string
    {
        return $this->expected_end->format('h:i A');
    }

    // ==================== دوال الحساب ====================

    /**
     * حساب دقائق التأخير
     */
    public function calculateLateMinutes(Carbon $checkIn): int
    {
        $expectedStart = Carbon::parse($checkIn->format('Y-m-d') . ' ' . $this->expected_start->format('H:i:s'));
        $graceEnd = $expectedStart->copy()->addMinutes($this->grace_period_minutes);

        if ($checkIn->lessThanOrEqualTo($graceEnd)) {
            return 0;
        }

        return $checkIn->diffInMinutes($expectedStart);
    }

    /**
     * حساب دقائق الوقت الإضافي
     */
    public function calculateOvertimeMinutes(Carbon $checkOut): int
    {
        $expectedEnd = Carbon::parse($checkOut->format('Y-m-d') . ' ' . $this->expected_end->format('H:i:s'));

        if ($checkOut->lessThanOrEqualTo($expectedEnd)) {
            return 0;
        }

        return $checkOut->diffInMinutes($expectedEnd);
    }

    /**
     * حساب دقائق العمل الفعلية
     */
    public function calculateWorkedMinutes(Carbon $checkIn, Carbon $checkOut): int
    {
        return $checkIn->diffInMinutes($checkOut);
    }

    /**
     * حساب مبلغ الوقت الإضافي
     */
    public function calculateOvertimeAmount(int $baseSalary, int $overtimeMinutes): int
    {
        $overtimeHourlyRate = $this->getOvertimeHourlyRate($baseSalary);
        $overtimeHours = $overtimeMinutes / 60;

        return (int) ($overtimeHours * $overtimeHourlyRate);
    }

    /**
     * حساب أجر الساعة الإضافية بناءً على الراتب الأساسي
     */
    public function getOvertimeHourlyRate(int $baseSalary): int
    {
        // الراتب اليومي = الراتب الشهري / 30
        $dailySalary = $baseSalary / 30;

        // الراتب في الساعة = الراتب اليومي / عدد ساعات العمل
        $hourlyRate = $dailySalary / ($this->expected_minutes / 60);

        // أجر الساعة الإضافية = الراتب في الساعة * نسبة الوقت الإضافي
        return (int) ($hourlyRate * ($this->overtime_rate_percentage / 100));
    }

    // ==================== دوال التحقق ====================

    /**
     * التحقق من أن الجدول نشط في تاريخ معين
     */
    public function isActiveOn(Carbon $date): bool
    {
        if (!$this->is_active) {
            return false;
        }

        if ($this->effective_from && $date->lessThan($this->effective_from)) {
            return false;
        }

        if ($this->effective_to && $date->greaterThan($this->effective_to)) {
            return false;
        }

        // التحقق من يوم الأسبوع
        return $date->dayOfWeek === $this->weekday;
    }

    /**
     * التحقق من التأخير
     */
    public function isLate(Carbon $checkIn): bool
    {
        return $this->calculateLateMinutes($checkIn) > 0;
    }

    /**
     * التحقق من وجود وقت إضافي
     */
    public function hasOvertime(Carbon $checkOut): bool
    {
        return $this->calculateOvertimeMinutes($checkOut) > 0;
    }

    /**
     * التحقق من أن الجدول عام (غير مخصص لموظف)
     */
    public function isGeneral(): bool
    {
        return is_null($this->employee_id);
    }

    /**
     * التحقق من أن الجدول مخصص لموظف
     */
    public function isEmployeeSpecific(): bool
    {
        return !is_null($this->employee_id);
    }

    // ==================== Scopes ====================

    /**
     * الجداول النشطة فقط
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * الجداول الخاصة بفرع معين
     */
    public function scopeForBranch($query, $branchId)
    {
        return $query->where('branch_id', $branchId);
    }

    /**
     * الجداول الخاصة بموظف معين
     */
    public function scopeForEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * الجداول العامة فقط (غير مخصصة لموظف)
     */
    public function scopeGeneral($query)
    {
        return $query->whereNull('employee_id');
    }

    /**
     * الجداول المخصصة لموظفين
     */
    public function scopeEmployeeSpecific($query)
    {
        return $query->whereNotNull('employee_id');
    }

    /**
     * الجداول الخاصة بيوم معين من الأسبوع
     */
    public function scopeForWeekday($query, int $weekday)
    {
        return $query->where('weekday', $weekday);
    }

    /**
     * الجداول السارية في تاريخ معين
     */
    public function scopeEffectiveOn($query, Carbon $date)
    {
        return $query->where('is_active', true)
            ->where(function ($q) use ($date) {
                $q->whereNull('effective_from')
                    ->orWhereDate('effective_from', '<=', $date);
            })
            ->where(function ($q) use ($date) {
                $q->whereNull('effective_to')
                    ->orWhereDate('effective_to', '>=', $date);
            });
    }

    /**
     * الجداول السارية حالياً
     */
    public function scopeCurrentlyEffective($query)
    {
        return $query->effectiveOn(now());
    }

    /**
     * الجداول المنتهية الصلاحية
     */
    public function scopeExpired($query)
    {
        return $query->whereNotNull('effective_to')
            ->whereDate('effective_to', '<', now());
    }
}
