<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * نموذج المستودع
 */
class Warehouse extends Model
{
    protected $fillable = [
        'city_id',
        'name',
        'is_active',
        'user_id',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    /** العلاقات */
    public function city(): BelongsTo
    {
        return $this->belongsTo(City::class);
    }

    // هذا العلاقة ليست مستعملة غالباً لأن الربط يتم عبر pivot (branches)
    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function balances(): HasMany
    {
        return $this->hasMany(StockBalance::class);
    }

    public function movements(): HasMany
    {
        return $this->hasMany(StockMovement::class);
    }

    public function branches()
    {
        return $this->belongsToMany(Branch::class, 'branch_warehouse')
            ->withPivot(['is_primary', 'assigned_at', 'unassigned_at'])
            ->withTimestamps();
    }

    /** سكوبات مساعدة */
    public function scopeActive($q)
    {
        return $q->where('is_active', true);
    }

    public function scopeForCity($q, int $cityId)
    {
        return $q->where('city_id', $cityId);
    }

    /**
     * المستودعات المرتبطة بفرع معيّن عبر جدول branch_warehouse
     */
    public function scopeForBranch($q, int $branchId)
    {
        return $q->whereHas('branches', function ($qq) use ($branchId) {
            $qq->where('branches.id', $branchId);
        });
    }

    /** إجماليات سريعة */
    public function totalOnHand(): int
    {
        return (int) $this->balances()->sum('on_hand');
    }

    public function totalReserved(): int
    {
        return (int) $this->balances()->sum('reserved_qty');
    }
}
