<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class VaultTransaction extends Model
{
    public const TYPE_DEPOSIT = 1; // إيداع

    public const TYPE_WITHDRAW = 2; // سحب

    public const TYPE_RESET = 3; // تصفير

    protected $fillable = [
        'vault_id',
        'amount',
        'transaction_type',
        'source_type',
        'source_id',
        'description',
        'user_id',
        'balance_after',
    ];

    protected $casts = [
        'vault_id' => 'integer',
        'amount' => 'integer',
        'transaction_type' => 'integer',
        'user_id' => 'integer',
        'balance_after' => 'integer',
    ];

    /* ========= علاقات ========= */

    public function vault(): BelongsTo
    {
        return $this->belongsTo(Vault::class);
    }

    public function source(): MorphTo
    {
        return $this->morphTo();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /* ========= دعم الاسمين amount / paid_amount ========= */

    // عند الإسناد: يقبل paid_amount كبديل لـ amount
    public function setPaidAmountAttribute($value): void
    {
        $this->attributes['amount'] = (int) str_replace(',', '', (string) $value);
    }

    // للقراءة فقط: ترجع نفس قيمة amount
    public function getPaidAmountAttribute(): int
    {
        return (int) ($this->attributes['amount'] ?? 0);
    }

    /* ========= عروض مساعدة ========= */

    public function getTypeNameAttribute(): string
    {
        return match ($this->transaction_type) {
            self::TYPE_DEPOSIT => 'إيداع',
            self::TYPE_WITHDRAW => 'سحب',
            self::TYPE_RESET => 'تصفير',
            default => 'غير معروف',
        };
    }

    public function getFormattedAmountAttribute(): string
    {
        return number_format((int) $this->amount, 0, '.', ',').' د.ع';
    }

    public function isDeposit(): bool
    {
        return $this->transaction_type === self::TYPE_DEPOSIT;
    }

    public function isWithdraw(): bool
    {
        return $this->transaction_type === self::TYPE_WITHDRAW;
    }

    public function getSignedAmountAttribute(): int
    {
        return $this->isDeposit() ? (int) $this->amount : -(int) $this->amount;
    }

    /* ========= Scopes ========= */

    public function scopeDeposits($q)
    {
        return $q->where('transaction_type', self::TYPE_DEPOSIT);
    }

    public function scopeWithdraws($q)
    {
        return $q->where('transaction_type', self::TYPE_WITHDRAW);
    }

    public function scopeForVault($q, int $vaultId)
    {
        return $q->where('vault_id', $vaultId);
    }

    public function scopeByUser($q, int $userId)
    {
        return $q->where('user_id', $userId);
    }

    public function scopeDateRange($q, $from, $to)
    {
        return $q->whereBetween('created_at', [$from, $to]);
    }

    public function scopeLatest($q)
    {
        return $q->orderByDesc('created_at');
    }

    public static function transactionTypes(): array
    {
        return [
            self::TYPE_DEPOSIT => 'إيداع',
            self::TYPE_WITHDRAW => 'سحب',
            self::TYPE_RESET => 'تصفير',
        ];
    }
}
