<?php

namespace App\Models;

use App\Models\Concerns\HasBranch;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class Vault extends Model
{
    use HasBranch;

    protected $fillable = [
        'branch_id',
        'balance',
        'password',
    ];

    protected $casts = [
        'branch_id' => 'integer',
        'balance' => 'integer',
    ];

    /* =========================
     * تعيين كلمة السر (مشفر)
     * ========================= */
    public function setPassword(string $password): void
    {
        $this->update([
            'password' => Hash::make($password),
        ]);
    }

    /* =========================
     * التحقق من كلمة السر
     * ========================= */
    public function checkPassword(string $password): bool
    {
        return Hash::check($password, $this->password);
    }

    public function transactions(): HasMany
    {
        return $this->hasMany(VaultTransaction::class);
    }

    /* ========= عروض/مساعدات ========= */

    public function getFormattedBalanceAttribute(): string
    {
        return number_format((int) $this->balance, 0, '.', ',').' د.ع';
    }

    public function getIsEmptyAttribute(): bool
    {
        return (int) $this->balance <= 0;
    }

    public function hasSufficientBalance(int $amount): bool
    {
        return (int) $this->balance >= $amount;
    }

    public static function forBranch(int $branchId): self
    {
        return static::firstOrCreate(['branch_id' => $branchId], ['balance' => 0]);
    }

    public function availableBalance(): int
    {
        return (int) $this->balance;
    }

    /* ========= عمليات برصيد ذري + رسائل تحقق مفهومة ========= */

    /** إيداع بمصدر مورفي */
    public function deposit(Model $source, int $amount, ?string $description = null, ?int $userId = null): VaultTransaction
    {
        $this->validateAmount($amount);
        $userId = $userId ?: (int) user_info('id');

        return DB::transaction(function () use ($source, $amount, $description, $userId) {
            $this->incrementBalance($amount);

            return VaultTransaction::create([
                'vault_id' => $this->id,
                'transaction_type' => VaultTransaction::TYPE_DEPOSIT,
                'amount' => $amount,
                'description' => $description,
                'user_id' => $userId,
                'balance_after' => (int) $this->balance,
                'source_type' => get_class($source),
                'source_id' => (int) $source->getKey(),
            ]);
        });
    }

    /** سحب بمصدر مورفي (يرمي ValidationException إن لم يكفِ الرصيد) */
    public function withdraw(Model $source, int $amount, ?string $description = null, ?int $userId = null): VaultTransaction
    {
        $this->validateAmount($amount);
        $userId = $userId ?: (int) user_info('id');

        return DB::transaction(function () use ($source, $amount, $description, $userId) {
            $this->atomicDecrementOrFail($amount);

            return VaultTransaction::create([
                'vault_id' => $this->id,
                'transaction_type' => VaultTransaction::TYPE_WITHDRAW,
                'amount' => $amount,
                'description' => $description,
                'user_id' => $userId,
                'balance_after' => (int) $this->balance,
                'source_type' => get_class($source),
                'source_id' => (int) $source->getKey(),
            ]);
        });
    }

    /** إيداع مباشر: المصدر = المستخدم الحالي */
    public function depositDirect(int $amount, ?string $description = null, ?int $userId = null): VaultTransaction
    {
        $this->validateAmount($amount);
        $userId = $userId ?: (int) user_info('id');

        return DB::transaction(function () use ($amount, $description, $userId) {
            $this->incrementBalance($amount);

            return VaultTransaction::create([
                'vault_id' => $this->id,
                'transaction_type' => VaultTransaction::TYPE_DEPOSIT,
                'amount' => $amount,
                'description' => $description,
                'user_id' => $userId,
                'balance_after' => (int) $this->balance,
                'source_type' => \App\Models\User::class,
                'source_id' => $userId,
            ]);
        });
    }

    /** سحب مباشر: المصدر = المستخدم الحالي */
    public function withdrawDirect(int $amount, ?string $description = null, ?int $userId = null): VaultTransaction
    {
        $this->validateAmount($amount);
        $userId = $userId ?: (int) user_info('id');

        return DB::transaction(function () use ($amount, $description, $userId) {
            $this->atomicDecrementOrFail($amount);

            return VaultTransaction::create([
                'vault_id' => $this->id,
                'transaction_type' => VaultTransaction::TYPE_WITHDRAW,
                'amount' => $amount,
                'description' => $description,
                'user_id' => $userId,
                'balance_after' => (int) $this->balance,
                'source_type' => \App\Models\User::class,
                'source_id' => $userId,
            ]);
        });
    }

    /* ========= عمليات الرصيد الداخلية ========= */

    protected function incrementBalance(int $amount): void
    {
        static::whereKey($this->id)->update([
            'balance' => DB::raw('balance + '.(int) $amount),
        ]);
        $this->refresh();
    }

    /**
     * إن لم يكفِ الرصيد ستُرمى ValidationException برسالة على حقل paid_amount
     * لتظهر في الـ UI كإشعار مفهوم.
     */
    protected function atomicDecrementOrFail(int $amount): void
    {
        // تحديث ذري مع شرط الرصيد
        $affected = static::whereKey($this->id)
            ->where('balance', '>=', (int) $amount)
            ->update(['balance' => DB::raw('balance - '.(int) $amount)]);

        if ($affected === 0) {
            throw ValidationException::withMessages([
                'paid_amount' => [
                    'الرصيد في الخزنة غير كافٍ لهذه العملية. الرصيد المتاح: '.
                        $this->formatted_balance.'، والمطلوب: '.
                        number_format($amount, 0, '.', ',').' د.ع',
                ],
            ]);
        }

        $this->refresh();
    }

    /* احتفاظ بتطبيق decrementBalance القديم عند الحاجة الداخلية */
    protected function decrementBalance(int $amount): void
    {
        static::whereKey($this->id)->update([
            'balance' => DB::raw('balance - '.(int) $amount),
        ]);
        $this->refresh();
    }

    public function recalculateBalance(): void
    {
        $total = (int) $this->transactions()
            ->selectRaw(
                'COALESCE(SUM(CASE WHEN transaction_type = ? THEN amount ELSE -amount END), 0) AS s',
                [VaultTransaction::TYPE_DEPOSIT]
            )->value('s');

        $this->update(['balance' => $total]);
    }

    protected function validateAmount(int $amount): void
    {
        if ($amount <= 0) {
            throw ValidationException::withMessages([
                'amount' => ['المبلغ يجب أن يكون أكبر من صفر.'],
            ]);
        }
    }

    /* ========= Scopes ========= */

    public function scopeForBranches(Builder $q, array $ids): Builder
    {
        return $q->whereIn('branch_id', $ids);
    }

    public function scopeForBranchs(Builder $q, array $ids): Builder
    {
        return $this->scopeForBranches($q, $ids);
    }

    public function scopeWithBalance(Builder $q): Builder
    {
        return $q->where('balance', '>', 0);
    }

    public function scopeEmpty(Builder $q): Builder
    {
        return $q->where('balance', '<=', 0);
    }
}
