<?php

namespace App\Models;

use Andreia\FilamentUiSwitcher\Models\Traits\HasUiPreferences;
use App\Models\Concerns\HasBranch;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable implements HasMedia
{
    use HasBranch;
    use HasFactory;
    use HasRoles;
    use HasUiPreferences;
    use InteractsWithMedia;
    use Notifiable;

    protected string $guard_name = 'web';

    protected $appends = [
        'avatar_url',
    ];

    protected $fillable = [
        'name',
        'username',
        'email',
        'phone',
        'password',
        'address',
        'status',
        'branch_id',
        'ui_preferences',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'status' => 'boolean',
            'ui_preferences' => 'array',
        ];
    }

    /* ===================== Media Library ===================== */

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('avatar')
            ->singleFile()
            ->useDisk('upload')
            ->acceptsMimeTypes([
                'image/jpeg',
                'image/png',
                'image/webp',
                'image/gif',
            ]);
    }

    public function registerMediaConversions(?Media $media = null): void
    {
        // ✅ صورة مصغرة ثابتة للعرض في الهيدر/البطاقات
        $this->addMediaConversion('thumb')
            ->fit(Fit::Crop, 128, 128)
            ->performOnCollections('avatar')
            ->nonQueued();

        // ✅ (اختياري) نسخة أكبر للمعاينة داخل الصفحات إن احتجتها لاحقاً
        $this->addMediaConversion('preview')
            ->fit(Fit::Crop, 512, 512)
            ->performOnCollections('avatar')
            ->nonQueued();
    }

    public function getAvatarUrlAttribute(): string
    {
        // ✅ جرّب thumb أولاً ثم الأصل ثم صورة افتراضية
        $url = $this->getFirstMediaUrl('avatar', 'thumb');

        if (! empty($url)) {
            return $url;
        }

        $url = $this->getFirstMediaUrl('avatar');

        if (! empty($url)) {
            return $url;
        }

        return asset('images/svg/avatars/blank.svg');
    }

    /* ===================== Model Events ===================== */

    protected static function booted(): void
    {
        static::deleting(function (self $user) {
            // إن كنت تستخدم SoftDeletes: لا تحذف الوسائط عند الحذف اللين
            if (method_exists($user, 'isForceDeleting') && ! $user->isForceDeleting()) {
                return;
            }

            $user->clearMediaCollection('avatar');
        });
    }

    /* ===================== Relations ===================== */

    public function branch(): BelongsTo
    {
        return $this->belongsTo(Branch::class);
    }

    public function payrollRuns()
    {
        return $this->hasMany(PayrollRun::class);
    }

    public function payrolls()
    {
        return $this->hasMany(Payroll::class);
    }

    public function payrollBonuses()
    {
        return $this->hasMany(PayrollBonus::class);
    }

    public function payrollDeductions()
    {
        return $this->hasMany(PayrollDeduction::class);
    }

    public function employeeAdvances()
    {
        return $this->hasMany(EmployeeAdvance::class);
    }

    /* ===================== Scopes ===================== */

    public function scopeForBranch(Builder $query, int $branchId): Builder
    {
        return $query->where('branch_id', $branchId);
    }

    // توجيه من الاسم القديم إن كان مستخدمًا في أماكن أخرى
    public function scopeForSubscription(Builder $query, int $branchId): Builder
    {
        return $this->scopeForBranch($query, $branchId);
    }

    public function scopeActive(Builder $query): Builder
    {
        return $query->where('status', true);
    }

    /* ===================== Role Helpers ===================== */

    public function admin(): bool
    {
        return $this->branch_id === null && $this->hasRole('admin');
    }

    public function isGeneralFinanceManager(): bool
    {
        return $this->hasRole('general-finance-manager');
    }
}
